------------------------------------------------------------------------------
--                                                                          --
--                           GNAT RAVENSCAR for NXT                         --
--                                                                          --
--                       Copyright (C) 2011, AdaCore                        --
--                                                                          --
-- This is free software; you can  redistribute it  and/or modify it under  --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion. This is distributed in the hope that it will be useful, but WITH-  --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNARL; see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- As a special exception,  if other files  instantiate  generics from this --
-- unit, or you link  this unit with other files  to produce an executable, --
-- this  unit  does not  by itself cause  the resulting  executable  to  be --
-- covered  by the  GNU  General  Public  License.  This exception does not --
-- however invalidate  any other reasons why  the executable file  might be --
-- covered by the  GNU Public License.                                      --
--                                                                          --
------------------------------------------------------------------------------

with NXT.Display;  use NXT.Display;

package body Discrete_IO.Display is

   type Screen_Quadrants is (Lower_Left, Lower_Right, Upper_Left, Upper_Right);

   function Flashing_Output
     (Selected : Output_Pin;
      Port     : Sensor_Id;
      Wire     : Wire_Colors)
      return Direction_Indicator;
   --  Returns the string indicating an output, with the provision that if the
   --  values of Port and Wire match those of Selected then the result is the
   --  indicator for the currently selected output (i.e., the asterisk). In
   --  addition, the value alternates between the standard output indicator and
   --  the current selection indicator, such that when displayed the asterisk
   --  "blinks" on and off. This effect serves as an indication that the
   --  program is running but also draws the user's attention to the current
   --  selection (if any). It accesses and updates a global variable to achieve
   --  that effect.

   procedure Display
     (Quadrant : Screen_Quadrants;
      Port     : Sensor_Id;
      Pins     : IO_Pins);
   --  Displays the three pins for the given sensor port in a dedicated
   --  quadrant of the screen. The quadrant assignments are arbitrary and
   --  hard-coded.

   function Image (Color : Wire_Colors) return Character;
   --  returns the image of the color

   function Image (Id : Sensor_Id) return Character;
   --  returns the image of the Sensor_Id as a single number (the input values
   --  are Sensor_1, Sensor_2, and so on)

   function Image (State : Pin_States) return Character;
   --  returns the image of the states High and Low as 1 and 0, respectively

   -------------
   -- Display --
   -------------

   procedure Display
     (Quadrant : Screen_Quadrants;
      Port     : Sensor_Id;
      Pins     : IO_Pins)
   is
      Col : Char_Columns;
      Row : Char_Rows;
   begin
      case Quadrant is
         when Upper_Left =>   Col := 0; Row := 0;
         when Upper_Right =>  Col := 9; Row := 0;
         when Lower_Left =>   Col := 0; Row := 4;
         when Lower_Right =>  Col := 9; Row := 4;
      end case;
      for Wire in Pins'Range loop
         Set_Pos (Col, Row);

         Put_Noupdate (Image (Port));
         Put_Noupdate (".");
         Put_Noupdate (Image (Wire));
         if Pins (Wire) = Input then
            Put_Noupdate (Input_Indicator);
         else
            if Selection_Active then
               Put_Noupdate (Flashing_Output (Selected, Port, Wire));
            else
               Put_Noupdate (Output_Indicator);
            end if;
         end if;
         Put_Noupdate (Image (Value (Port, Wire)));

         Row := Row + 1;
      end loop;
   end Display;

   ------------------------
   -- Display_Pin_States --
   ------------------------

   procedure Display_Pin_States (Pin_Map : IO_Mapping) is
   begin
      Clear_Screen_Noupdate;
      for Port in Pin_Map'Range loop
         case Port is
            when Sensor_1 =>
               Display (Upper_Left,  Port, Pin_Map (Port));
            when Sensor_2 =>
               Display (Upper_Right, Port, Pin_Map (Port));
            when Sensor_3 =>
               Display (Lower_Left,  Port, Pin_Map (Port));
            when Sensor_4 =>
               Display (Lower_Right, Port, Pin_Map (Port));
         end case;
      end loop;
      Screen_Update;
   end Display_Pin_States;

   ------------------
   -- Mark_Current --
   ------------------

   procedure Mark_Current (Port : Sensor_Id;  Wire : Wire_Colors) is
   begin
      Selected.Port := Port;
      Selected.Wire := Wire;
      Selection_Active := True;
   end Mark_Current;

   -----------
   -- Image --
   -----------

   function Image (Color : Wire_Colors) return Character is
   begin
      case Color is
         when White =>  return 'W';
         when Yellow => return 'Y';
         when Blue =>   return 'B';
      end case;
   end Image;

   -----------
   -- Image --
   -----------

   function Image (Id : Sensor_Id) return Character is
   begin
      case Id is
         when Sensor_1 => return '1';
         when Sensor_2 => return '2';
         when Sensor_3 => return '3';
         when Sensor_4 => return '4';
      end case;
   end Image;

   -----------
   -- Image --
   -----------

   function Image (State : Pin_States) return Character is
   begin
      case State is
         when High => return '1';
         when Low  => return '0';
      end case;
   end Image;

   ---------------------
   -- Flashing_Output --
   ---------------------

   function Flashing_Output
     (Selected : Output_Pin;
      Port     : Sensor_Id;
      Wire     : Wire_Colors)
      return Direction_Indicator
   is
      Result : Direction_Indicator := Output_Indicator;
   begin
      if Port = Selected.Port and Wire = Selected.Wire then
         if Selected_Icon_Display_Count mod Update_Interval <= Flash_Count then
            Result := Current_Selection_Indicator;
         end if;
      end if;
      Selected_Icon_Display_Count := Selected_Icon_Display_Count + 1;
      return Result;
   end Flashing_Output;

end Discrete_IO.Display;
