------------------------------------------------------------------------------
--                                                                          --
--                    Copyright (C) 2010-2011, AdaCore                      --
--                                                                          --
-- This  is  free  software;  you  can  redistribute  it and/or  modify  it --
-- under terms of the  GNU General Public License as published  by the Free --
-- Software  Foundation;  either version 2,  or (at your option)  any later --
-- version.  This  is  distributed  in the hope that  it  will  be  useful, --
-- but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANT- --
-- ABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public --
-- License for  more details.  You should have  received  a copy of the GNU --
-- General Public License distributed with GNATStack;  see file COPYING. If --
-- not, write to the  Free Software Foundation,  51 Franklin Street,  Fifth --
-- Floor, Boston, MA 02110-1301, USA.                                       --
------------------------------------------------------------------------------

with Interfaces;       use Interfaces;
with Ada.Real_Time;    use Ada.Real_Time;
with NXT.Bluetooth;    use NXT.Bluetooth;
with NXT.AVR;
with NXT.Display;      use NXT.Display;
with NXT.Motor_Encoders;
with Common;           use Common;
with NXT.Last_Chance;

procedure Vehicle is
   use NXT; -- for types

   Motor_Left  : constant Motor_Id := Motor_A;
   Motor_Right : constant Motor_Id := Motor_C;
   Motor_Steer : constant Motor_Id := Motor_B;

   procedure Connect_Remote is
      Success    : Boolean;
      Rc_Address : BT_Address;
   begin
      NXT.Bluetooth.Initialize;
      NXT.Bluetooth.Set_Discoverable (True);

      loop
         Find_Peer (Rc_Address, Success);
         exit when Success;
         Put_Line ("Trying...");
      end loop;

      Put_Line ("Peer found");
      NXT.Bluetooth.Connect (Rc_Address, Success);
      if Success then
         Put_Line ("connected");
      else
         Put_Line ("Connect failed");
         Power_Off;
      end if;
   end Connect_Remote;

   function Get_Steer_Pos return Integer is
   begin
      return NXT.Motor_Encoders.Encoder_Count (Motor_Steer);
   end Get_Steer_Pos;

   Mid_Pos : Integer;

   procedure Calibrate_Steer is
      Left_Pos  : Integer;
      Right_Pos : Integer;
      Pos       : Integer;
      Speed     : constant PWM_Value := 20;
   begin
      --  Left.
      NXT.AVR.Set_Power (Motor_Steer, Speed, False);

      Left_Pos := Get_Steer_Pos;
      loop
         delay until Clock + Milliseconds (100);
         Pos := Get_Steer_Pos;
         exit when Pos = Left_Pos;
         Left_Pos := Pos;
      end loop;

      --  Right.
      NXT.AVR.Set_Power (Motor_Steer, -Speed, False);

      Right_Pos := Get_Steer_Pos;
      loop
         delay until Clock + Milliseconds (100);
         Pos := Get_Steer_Pos;
         exit when Pos = Right_Pos;
         Right_Pos := Pos;
      end loop;

      --  Middle.
      Mid_Pos := (Left_Pos + Right_Pos) / 2;
      NXT.AVR.Set_Power (Motor_Steer, Speed, False);
      loop
         delay until Clock + Milliseconds (10);
         exit when abs (Get_Steer_Pos - Mid_Pos) < 20;
      end loop;
      NXT.AVR.Set_Power (Motor_Steer, 0, False);
   end Calibrate_Steer;

   Msg            : aliased Remote_Command;
   Turn_Speed     : PWM_Value;
   Steer_Pos      : Integer := 0;
   Steer_Delta    : Integer;
   Steer_Step     : constant Integer := 100;
   Steer_Eps      : constant Integer := 40;
   Steer_Hi_Eps   : constant Integer := 200;
   Steer_Lo_Speed : constant PWM_Value := 50;
   Steer_Hi_Speed : constant PWM_Value := 80;
begin
   --  Calibrate_Steer;
   --  Power_Off;
   Connect_Remote;

   loop
      --  Wait for a message.
      loop
         if NXT.AVR.Button = Power_Button then
            Power_Off;
         end if;

         Command_IO.Receive (Msg'Access);
         exit when Msg.Len /= 0;
         delay until Clock + Milliseconds (10);
      end loop;

      NXT.AVR.Set_Power (Motor_Left, Msg.Speed, True);
      NXT.AVR.Set_Power (Motor_Right, Msg.Speed, True);

      Steer_Delta := Get_Steer_Pos - Steer_Pos;

      if abs Steer_Delta < Steer_Eps then
         case Msg.Turn is
            when Turn_Left =>
               Steer_Pos := Steer_Pos + Steer_Step;
            when Turn_Right =>
               Steer_Pos := Steer_Pos - Steer_Step;
            when Turn_None =>
               null;
         end case;
         Steer_Delta := Get_Steer_Pos - Steer_Pos;
      end if;

      if abs Steer_Delta < Steer_Eps then
         Turn_Speed := 0;
      elsif abs Steer_Delta < Steer_Hi_Eps then
         Turn_Speed := Steer_Lo_Speed;
      else
         Turn_Speed := Steer_Hi_Speed;
      end if;
      if Steer_Delta > 0 then
         Turn_Speed := -Turn_Speed;
      end if;

      if Msg.Speed >= 0 then
         Put_Hex (Unsigned_8 (Msg.Speed));
      else
         Put ('-');
         Put_Hex (Unsigned_8 (-Msg.Speed));
      end if;
      Put_Noupdate (' ');
      Put_Noupdate (NXT.Motor_Encoders.Encoder_Count (Motor_Steer));
      Put_Noupdate (',');
      Put_Noupdate (Steer_Pos);
      Put_Noupdate (',');
      Put_Noupdate (Integer (Turn_Speed));
      Newline;

      NXT.AVR.Set_Power (Motor_Steer, Turn_Speed, False);
   end loop;

end Vehicle;
