------------------------------------------------------------------------------
--                                                                          --
--                       Copyright (C) 2010, AdaCore                        --
--                                                                          --
-- This  is  free  software;  you  can  redistribute  it and/or  modify  it --
-- under terms of the  GNU General Public License as published  by the Free --
-- Software  Foundation;  either version 2,  or (at your option)  any later --
-- version.  This  is  distributed  in the hope that  it  will  be  useful, --
-- but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANT- --
-- ABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public --
-- License for  more details.  You should have  received  a copy of the GNU --
-- General Public License distributed with GNATStack;  see file COPYING. If --
-- not, write to the  Free Software Foundation,  51 Franklin Street,  Fifth --
-- Floor, Boston, MA 02110-1301, USA.                                       --
------------------------------------------------------------------------------

with Interfaces;    use Interfaces;
with Ada.Real_Time; use Ada.Real_Time;
with NXT.Bluetooth; use NXT.Bluetooth;
with NXT.AVR;
with NXT.Display;   use NXT.Display;
with Common;        use Common;
with NXT.Last_Chance;

procedure Remote is
   use NXT; -- for types

   --  The sensors.
   --  Please correctly connect the sensors...
   Sensor_Fwd  : constant Sensor_Id := Sensor_1;
   Sensor_Gyro : constant Sensor_Id := Sensor_2;
   Sensor_Bwd  : constant Sensor_Id := Sensor_4;

   Enable_Bluetooth : constant Boolean := True;

   procedure Connect_Vehicle is
      Vehicle_Address : BT_Address;
      Peer_Addr       : BT_Address;
      Pin             : Pin_Code := (others => 0);
      Success         : Boolean;
   begin
      NXT.Bluetooth.Initialize;
      NXT.Bluetooth.Set_Discoverable (True);

      Put_Line ("Finding peer");
      loop
         Find_Peer (Vehicle_Address, Success);
         exit when Success;
         Put_Line ("Trying...");
      end loop;

      Put_Line ("Waiting peer");
      Accept_Connection (Pin, Peer_Addr, Success);
      if Success then
         Put_Line ("connected");
      else
         Put_Line ("connect failed");
         Power_Off;
      end if;
   end Connect_Vehicle;

   Gyro_Bias : Integer := 600;
   --  The zero of the gyro.

   Theta     : Integer := 0;
   Theta_Old : Integer := 0;

   procedure Calibrate_Steer is
      A  : constant Float := 0.1;
      T  : Time := Clock;
      Td : constant Time_Span := Milliseconds (20);
   begin
      for I in 1 .. 100 loop
         T := T + Td;
         delay until T;

         --  Evaluates the new gyro zero position
         Gyro_Bias := Integer
           ((1.0 - A) * Float (Gyro_Bias)
              + A * Float (NXT.AVR.Raw_Input (Sensor_Gyro)));
      end loop;
   end Calibrate_Steer;

   Turn : Turn_Directions;

   procedure Steer is
      Tmp : Integer;

      Gyro_Scale : constant Integer := 4;
      Alpha      : Integer;
      Noise      : constant Integer := 5;
   begin
      --  Gets a gyro status
      Tmp := Integer (NXT.AVR.Raw_Input (Sensor_Gyro));

      --  Evaluates the current position of steering wheel
      Alpha := (Tmp - Gyro_Bias) / Gyro_Scale;

      Put_Noupdate ("a=");
      Put_Noupdate (Alpha);

      --  If a current measurement is greater then previous value
      if Alpha > Noise then
         Turn := Turn_Left;

         --  If a current measurement is smaller then previous value
      elsif Alpha < -Noise then
         Turn := Turn_Right;

         --  If a current measurement is equal previous value
      else
         Turn := Turn_None;
      end if;
   end Steer;

   Fwd        : Boolean;
   Bwd        : Boolean;
   Speed      : PWM_Value := 0;
   Speed_Incr : constant PWM_Value := 10;
   Msg        : aliased Remote_Command;
begin
   Put_Noupdate ("*************** ");
   Put_Noupdate ("*    Gyro     * ");
   Put_Noupdate ("* Calibration * ");
   Put_Noupdate ("*************** ");
   Screen_Update;
   Calibrate_Steer;
   Clear_Screen;

   if Enable_Bluetooth then
      Connect_Vehicle;
   else
      Put_Line ("Local mode!");
   end if;

   loop
      Steer;
      Fwd := NXT.AVR.Raw_Input (Sensor_Fwd) < 16#200#;
      Bwd := NXT.AVR.Raw_Input (Sensor_Bwd) < 16#200#;

      if Fwd and Bwd then
         Speed := 0;
      elsif Fwd then
         if Speed < PWM_Value'Last - Speed_Incr then
            Speed := Speed + Speed_Incr;
         end if;
      elsif Bwd then
         if Speed > PWM_Value'First + Speed_Incr then
            Speed := Speed - Speed_Incr;
         end if;
      end if;

      if Speed >= 0 then
         Put_Noupdate ('+');
         Put_Hex (Unsigned_8 (Speed));
      else
         Put_Noupdate ('-');
         Put_Hex (Unsigned_8 (-Speed));
      end if;
      case Turn is
         when Turn_Left =>
            Put_Noupdate ('L');
         when Turn_Right =>
            Put_Noupdate ('R');
         when Turn_None =>
            Put_Noupdate (' ');
            null;
      end case;
      if Theta >= 0 then
         Put_Noupdate ('+');
         Put_Hex (Unsigned_16 (Theta));
      else
         Put_Noupdate ('-');
         Put_Hex (Unsigned_16 (-Theta));
      end if;
      Put (ASCII.CR);

      Msg := (Len => Msg'Size / 8 - 1,
              Speed => Speed,
              Turn => Turn);

      if Enable_Bluetooth then
         Command_IO.Send (Msg'Access);
      end if;

      delay until Clock + Milliseconds (50);

      if NXT.AVR.Button = Power_Button then
         Power_Off;
      end if;
   end loop;
end Remote;
