------------------------------------------------------------------------------
--                                                                          --
--                    Copyright (C) 2010-2011, AdaCore                      --
--                                                                          --
-- This  is  free  software;  you  can  redistribute  it and/or  modify  it --
-- under terms of the  GNU General Public License as published  by the Free --
-- Software  Foundation;  either version 2,  or (at your option)  any later --
-- version.  This  is  distributed  in the hope that  it  will  be  useful, --
-- but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANT- --
-- ABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public --
-- License for  more details.  You should have  received  a copy of the GNU --
-- General Public License distributed with GNATStack;  see file COPYING. If --
-- not, write to the  Free Software Foundation,  51 Franklin Street,  Fifth --
-- Floor, Boston, MA 02110-1301, USA.                                       --
------------------------------------------------------------------------------

with NXT.Display;
with NXT.AVR; use NXT.AVR;
with Menus;   use Menus;
with NXT.Motor_Encoders;

package body Menu_Motors is
   use NXT;

   type Speed_Array is array (Motor_Id) of PWM_Value;

   Cur_Motor : Motor_Id := Motor_A;
   Speeds    : Speed_Array := (others => 0);

   Step : constant PWM_Value := 20;

   procedure Put_Motor_Id (Id : Motor_Id) is
      use NXT.Display;
   begin
      Put_Noupdate (Character'Val (Character'Pos ('A') + Motor_Id'Pos (Id)));
   end Put_Motor_Id;

   procedure Next_Motor is
   begin
      if Cur_Motor = Motor_Id'Last then
         Cur_Motor := Motor_Id'First;
      else
         Cur_Motor := Motor_Id'Succ (Cur_Motor);
      end if;
   end Next_Motor;

   procedure Select_Next_Motor is
      use NXT.Display;
   begin
      Next_Motor;
      Put_Noupdate ("Motor ");
      Put_Motor_Id (Cur_Motor);
      Screen_Update;
      Wait_Key;
   end Select_Next_Motor;

   procedure Select_Motor_B is
   begin
      Cur_Motor := Motor_B;
   end Select_Motor_B;

   procedure Select_Motor_C is
   begin
      Cur_Motor := Motor_C;
   end Select_Motor_C;

   procedure Speed_Up is
   begin
      if Speeds (Cur_Motor) < PWM_Value'Last - Step then
         Speeds (Cur_Motor) := Speeds (Cur_Motor) + Step;
      else
         Speeds (Cur_Motor) := PWM_Value'Last;
      end if;
      Set_Power (Cur_Motor, Speeds (Cur_Motor), False);
   end Speed_Up;

   procedure Speed_Down is
   begin
      if Speeds (Cur_Motor) > PWM_Value'First + Step then
         Speeds (Cur_Motor) := Speeds (Cur_Motor) - Step;
      else
         Speeds (Cur_Motor) := PWM_Value'First;
      end if;
      Set_Power (Cur_Motor, Speeds (Cur_Motor), False);
   end Speed_Down;

   procedure Stop is
   begin
      Speeds (Cur_Motor) := 0;
      Set_Power (Cur_Motor, Speeds (Cur_Motor), Brake => False);
   end Stop;

   procedure Brake is
   begin
      Speeds (Cur_Motor) := 0;
      Set_Power (Cur_Motor, Speeds (Cur_Motor), Brake => True);
   end Brake;

   procedure Stop_All is
   begin
      for M in Motor_Id loop
         Speeds (M) := 0;
         Set_Power (M, Speeds (M), False);
      end loop;
   end Stop_All;

   procedure Show_Positions is
      use NXT.Display;
      Last_Button    : Button_Id := NXT.AVR.Button;
      Current_Button : Button_Id;
   begin
      loop
         Clear_Screen_Noupdate;
         for M in Motor_Id loop
            Put_Motor_Id (M);
            Put_Noupdate (": ");
            Put_Noupdate (NXT.Motor_Encoders.Encoder_Count (M));
            Newline_Noupdate;
         end loop;
         Newline_Noupdate;
         Put_Noupdate ("Motor: ");
         Put_Motor_Id (Cur_Motor);
         Screen_Update;
         Wait_Ms;
         Current_Button := NXT.AVR.Button;
         if Current_Button /= Last_Button then
            case Current_Button is
               when Power_Button =>
                  exit;
               when Left_Button =>
                  Speed_Down;
               when Right_Button =>
                  Speed_Up;
               when Middle_Button =>
                  Brake;
               when No_Button =>
                  null;
            end case;
            Last_Button := Current_Button;
         end if;
      end loop;
   end Show_Positions;

   Main_Menu : constant Menu :=
     (
      (new String'("Sel next motor"), Select_Next_Motor'Access),
      (new String'("Speed up"),       Speed_Up'Access),
      (new String'("Speed down"),     Speed_Down'Access),
      (new String'("Stop"),           Stop'Access),
      (new String'("Brake"),          Brake'Access),
      (new String'("Stop all"),       Stop_All'Access),
      (new String'("Positions"),      Show_Positions'Access)
     );

   procedure Do_Menu_Motors is
   begin
      Play_Menu (Main_Menu);
   end Do_Menu_Motors;

end Menu_Motors;
