------------------------------------------------------------------------------
--                                                                          --
--                       Copyright (C) 2010, AdaCore                        --
--                                                                          --
-- This  is  free  software;  you  can  redistribute  it and/or  modify  it --
-- under terms of the  GNU General Public License as published  by the Free --
-- Software  Foundation;  either version 2,  or (at your option)  any later --
-- version.  This  is  distributed  in the hope that  it  will  be  useful, --
-- but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANT- --
-- ABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public --
-- License for  more details.  You should have  received  a copy of the GNU --
-- General Public License distributed with GNATStack;  see file COPYING. If --
-- not, write to the  Free Software Foundation,  51 Franklin Street,  Fifth --
-- Floor, Boston, MA 02110-1301, USA.                                       --
------------------------------------------------------------------------------

with Interfaces;       use Interfaces;
with Ada.Real_Time;    use Ada.Real_Time;
with NXT.AVR;          use NXT.AVR;
with NXT.LCD;
with NXT.BC4.IO;       use NXT.BC4.IO;
with NXT.BC4.Messages; use NXT.BC4.Messages;
with NXT.Bluetooth;    use NXT.Bluetooth;
with NXT.Display;      use NXT.Display;
with Menus;            use Menus;

package body Menu_Bt is

   use NXT; -- for types
   subtype S is String;

   --  Array of known devices.
   Devices     : Device_List (1 .. 32);
   Nbr_Devices : Natural := 0;
   Sel_Device  : Natural := Devices'First;

   procedure Put_Address (Addr : BT_Address) is
   begin
      for I in Addr'Range loop
         Put_Hex (Addr (I));
      end loop;
   end Put_Address;

   procedure Put_Name (Name : Friendly_Name) is
   begin
      if Name (0) = 0 then
         Put_Noupdate ("no-name");
      else
         for I in Name'Range loop
            exit when Name (I) = 0;
            Put_Noupdate (Character'Val (Name (I)));
         end loop;
      end if;
   end Put_Name;

   procedure Show_Local_Addr is
      Cmd   : Packet (0 .. 0);
      Reply : Packet (0 .. 16);
   begin
      Cmd (0) := MSG_GET_LOCAL_ADDR;
      Send_Packet (Cmd);

      loop
         Receive_Packet (Reply);

         if Reply (0) /= 0 and then Reply (1) = MSG_GET_LOCAL_ADDR_RESULT then
            Newline_Noupdate;
            Put_Noupdate ("Addr: ");
            Newline_Noupdate;
            Put_Address (BT_Address (Reply (2 .. 8)));
            Newline;
            exit;
         else
            Wait_Ms;
         end if;
      end loop;
      Wait_Key;
   end Show_Local_Addr;

   procedure Show_Friendly_Name is
      Cmd   : Packet (0 .. 0);
      Reply : Packet (0 .. 23);
   begin
      Cmd (0) := MSG_GET_FRIENDLY_NAME;
      Send_Packet (Cmd);

      loop
         Receive_Packet (Reply);

         if Reply (0) = 0 then
            Wait_Ms;
         elsif Reply (1) = MSG_GET_FRIENDLY_NAME_RESULT then
            Newline_Noupdate;
            Put_Noupdate ("Name: ");
            Newline_Noupdate;
            Put_Name (Friendly_Name (Reply (2 .. 17)));
            Newline;
            exit;
         else
            Put ("Got: ");
            Put_Hex (Reply (1));
            Newline;
         end if;
      end loop;
      Wait_Key;
   end Show_Friendly_Name;

   procedure Low_Set_Friendly_Name is
      Cmd   : Packet (0 .. 16);
      Reply : Packet (0 .. 16);
   begin
      Cmd (0)  := MSG_SET_FRIENDLY_NAME;
      Cmd (1)  := Character'Pos ('N');
      Cmd (2)  := Character'Pos ('X');
      Cmd (3)  := Character'Pos ('T');
      Cmd (4)  := Character'Pos (' ');
      Cmd (5)  := Character'Pos ('A');
      Cmd (6)  := Character'Pos ('d');
      Cmd (7)  := Character'Pos ('a');
      Cmd (8 .. 16) := (others => 0);
      Send_Packet (Cmd);

      loop
         Receive_Packet (Reply);
         if Reply (0) /= 0
           and then Reply (1) = MSG_SET_FRIENDLY_NAME_ACK
         then
            Put_Line ("friendly name");
            exit;
         else
            Wait_Ms;
         end if;
      end loop;
      Wait_Key;
   end Low_Set_Friendly_Name;

   procedure Fill_List is
   begin
      Put_Line ("List of hosts:");
      Get_Known_Peers (Devices, Nbr_Devices);
      Wait_Key;
   end Fill_List;

   procedure Display_Entry is
      Dev : Device renames Devices (Sel_Device);
   begin
      Put_Noupdate ("Address:");
      Newline_Noupdate;
      Put_Address (Dev.Addr);
      Newline_Noupdate;
      Put_Noupdate ("Class:");
      Newline_Noupdate;
      for I in Dev.Class'Range loop
         Put_Hex (Dev.Class (I));
      end loop;
      Newline_Noupdate;
      Put_Noupdate ("Name:");
      Newline_Noupdate;
      Put_Name (Dev.Name);
      Newline_Noupdate;
      Wait_Key;
   end Display_Entry;

   procedure Add_Entry is
      Cmd   : Packet (0 .. 27);
      Reply : Packet (0 .. 3);
   begin
      Cmd (0) := MSG_ADD_DEVICE;
      Cmd (1 .. 7) := Packet (Devices (Sel_Device).Addr);
      Cmd (8 .. 23) := Packet (Devices (Sel_Device).Name);
      Cmd (24 .. 27) := Packet (Devices (Sel_Device).Class);
      Send_Packet (Cmd);

      loop
         Receive_Packet (Reply);
         if Reply (0) = 0 then
            Wait_Ms;
         elsif Reply (1) = MSG_LIST_RESULT then
            if Reply (2) = 16#50# then
               Put_Noupdate ("Added");
               Newline_Noupdate;
            else
               Put_Noupdate ("Err: ");
               Put_Hex (Reply (2));
               Newline_Noupdate;
            end if;
            Wait_Key;
            exit;
         else
            Put ("Got: ");
            Put_Hex (Reply (1));
            Newline;
         end if;
      end loop;
   end Add_Entry;

   procedure Remove_Entry is
      Cmd   : Packet (0 .. 7);
      Reply : Packet (0 .. 3);
   begin
      Cmd (0) := MSG_REMOVE_DEVICE;
      Cmd (1 .. 7) := Packet (Devices (Sel_Device).Addr);
      Send_Packet (Cmd);

      loop
         Receive_Packet (Reply);
         if Reply (0) = 0 then
            Wait_Ms;
         elsif Reply (1) = MSG_LIST_RESULT then
            if Reply (2) = 16#53# then
               Put_Noupdate ("Removed");
               Newline_Noupdate;
               Nbr_Devices := Nbr_Devices - 1;
               if Sel_Device > Nbr_Devices then
                  Sel_Device := Nbr_Devices;
               else
                  Devices (Sel_Device .. Nbr_Devices) :=
                    Devices (Sel_Device + 1 .. Nbr_Devices + 1);
               end if;
            else
               Put_Noupdate ("Err: ");
               Put_Hex (Reply (2));
               Newline_Noupdate;
            end if;
            exit;
         else
            Put ("Got: ");
            Put_Hex (Reply (1));
            Newline;
         end if;
      end loop;
   end Remove_Entry;

   procedure Lookup_Entry_Name is
      Cmd   : Packet (0 .. 7);
      Reply : Packet (0 .. 31);
   begin
      Cmd (0) := MSG_LOOKUP_NAME;
      Send_Packet (Cmd);

      loop
         Receive_Packet (Reply);
         if Reply (0) = 0 then
            Wait_Ms;
         elsif Reply (1) = MSG_LOOKUP_NAME_RESULT then
            Put_Noupdate ("Name:");
            Newline_Noupdate;
            Put_Name (Friendly_Name (Reply (9 .. 24)));
            Newline;
            exit;
         elsif Reply (1) = MSG_LOOKUP_NAME_FAILURE then
            Put_Line ("Failed");
            exit;
         else
            Put ("Got: ");
            Put_Hex (Reply (1));
            Newline;
         end if;
      end loop;
      Wait_Key;
   end Lookup_Entry_Name;

   procedure Connect_Entry is
      Dev : Device renames Devices (Sel_Device);

      Handle : Unsigned_8;
      Cmd    : Packet (0 .. 7);
      Reply  : Packet (0 .. 3);

      Msg : constant String := "Hello from NXT-Ada" & ASCII.LF;
   begin
      Cmd (0) := MSG_CONNECT;
      Cmd (1 .. 7) := Packet (Devices (Sel_Device).Addr);
      Send_Packet (Cmd);

      loop
         Receive_Packet (Reply);
         if Reply (0) = 0 then
            if Button = Power_Button then
               return;
            end if;
            Wait_Ms;
         elsif Reply (1) = MSG_CONNECT_RESULT then
            Put_Noupdate ("Connect ");
            if Reply (2) = 1 then
               Handle := Reply (3);
               Put_Noupdate ("Hnd=");
               Put_Hex (Handle);
               Newline;
               exit;
            else
               Put_Noupdate ("Err ");
               Put_Hex (Reply (2));
               Newline;
               return;
            end if;
         elsif Reply (1) = MSG_CLOSE_CONNECTION_RESULT then
            Put_Noupdate ("Close ");
            Put_Hex (Reply (2));
            Put_Noupdate (' ');
            Put_Hex (Reply (3));
            Newline;
            return;
         else
            Put ("Got: ");
            Put_Hex (Reply (1));
            Newline;
         end if;
      end loop;

      Put_Line ("sending data");

      Cmd (0) := MSG_OPEN_STREAM;
      Cmd (1) := Handle;
      Send_Packet (Cmd (0 .. 1));
      delay until Clock + Milliseconds (100);
      NXT.BC4.Enter_Data_Mode;

      NXT.BC4.Send (Msg'Address, Msg'Length);

      delay until Clock + Milliseconds (100);

      NXT.BC4.Enter_Command_Mode;
      delay until Clock + Milliseconds (50);

      Put_Line ("closing");
      Cmd (0) := MSG_CLOSE_CONNECTION;
      Cmd (1) := Handle;
      Send_Packet (Cmd (0 .. 1));

      loop
         Receive_Packet (Reply);
         if Reply (0) = 0 then
            if Button = Power_Button then
               return;
            end if;
            Wait_Ms;
         elsif Reply (1) = MSG_CLOSE_CONNECTION_RESULT then
            Put_Noupdate ("Close ");
            Put_Hex (Reply (2));
            Put_Noupdate (' ');
            Put_Hex (Reply (3));
            Newline;
            exit;
         else
            Put ("Got: ");
            Put_Hex (Reply (1));
            Newline;
         end if;
      end loop;

      Wait_Key;
   end Connect_Entry;

   Menu_List : constant Menu :=
     (
      (new S'("info"), Display_Entry'Access),
      (new S'("name"), Lookup_Entry_Name'Access),
      (new S'("remove"), Remove_Entry'Access),
      (new S'("add"), Add_Entry'Access),
      (new S'("connect"), Connect_Entry'Access)
     );

   procedure Handle_List is
      use Nxt; -- for types
   begin
      loop
         Clear_Screen_Noupdate;

         if Nbr_Devices = 0 then
            Put_Noupdate ("no devices");
            Newline_Noupdate;
            Wait_Key;
            return;
         else
            for I in 1 .. 4 loop
               exit when I > Nbr_Devices;
               if I = Sel_Device then
                  Put_Noupdate ('*');
               else
                  Put_Noupdate (' ');
               end if;
               Put_Address (Devices (I).Addr);
               Newline_Noupdate;
            end loop;
            Screen_Update;
         end if;

         case Pressed_Key is
            when Left_Button =>
               if Sel_Device > Devices'First then
                  Sel_Device := Sel_Device - 1;
               else
                  Sel_Device := Nbr_Devices;
               end if;
            when Right_Button =>
               if Sel_Device < Nbr_Devices then
                  Sel_Device := Sel_Device + 1;
               else
                  Sel_Device := Devices'First;
               end if;
            when Power_Button =>
               exit;
            when Middle_Button =>
               Play_Menu (Menu_List);
         end case;
      end loop;
   end Handle_List;

   procedure Disp_Operating_Mode is
      Cmd   : Packet (0 .. 0);
      Reply : Packet (0 .. 31);
   begin
      Newline;

      Cmd (0) := MSG_GET_OPERATING_MODE;
      Send_Packet (Cmd);

      loop
         Receive_Packet (Reply);
         if Reply (0) /= 0 then
            if Reply (1) = MSG_SET_OPERATING_MODE_RESULT then
               Put ("Op mode: ");
               if Reply (2) = 1 then
                  Put ("no-brk");
               elsif Reply (2) = 0 then
                  Put ("brk");
               else
                  Put_Hex (Reply (2));
               end if;
               Newline;
               exit;
            else
               Put ("Got: ");
               Put_Hex (Reply (1));
               Newline;
            end if;
         else
            Wait_Ms;
         end if;
      end loop;
      Wait_Key;
   end Disp_Operating_Mode;

   procedure Set_Discoverable is
      Cmd   : Packet (0 .. 1);
      Reply : Packet (0 .. 31);
   begin
      Newline;

      Cmd (0) := MSG_SET_DISCOVERABLE;
      Cmd (1) := 1;
      Send_Packet (Cmd);

      loop
         Receive_Packet (Reply);
         if Reply (0) /= 0 then
            if Reply (1) = MSG_DISCOVERABLE_ACK then
               if Reply (2) = 1 then
                  Put ("Success");
               elsif Reply (2) = 0 then
                  Put ("Write failed");
               else
                  Put_Hex (Reply (2));
               end if;
               Newline;
               exit;
            else
               Put ("Got: ");
               Put_Hex (Reply (1));
               Newline;
            end if;
         else
            Wait_Ms;
         end if;
      end loop;
      Wait_Key;
   end Set_Discoverable;

   procedure Accept_Connection is
      Cmd       : Packet (0 .. 0);
      Pin_Cmd   : Packet (0 .. 23);
      Accpt_Cmd : Packet (0 .. 1);
      Reply     : Packet (0 .. 32);
   begin
      Cmd (0) := MSG_OPEN_PORT;
      Send_Packet (Cmd);

      loop
         Receive_Packet (Reply);
         if Reply (0) = 0 then
            exit when Button = Power_Button;
            Wait_Ms;
         else
            if Reply (1) = MSG_PORT_OPEN_RESULT then
               Put ("Open: ");
               Put_Hex (Reply (2));
               Put (' ');
               Put_Hex (Reply (3));
               Put (' ');
               Put_Hex (Reply (4));
               Newline;
            elsif Reply (1) = MSG_REQUEST_PIN_CODE then
               Put_Line ("Pin...");
               Pin_Cmd (0) := MSG_PIN_CODE;
               Pin_Cmd (1 .. 7) := Reply (2 .. 8);
               Pin_Cmd (8) := Character'Pos ('0');
               Pin_Cmd (9) := Character'Pos ('0');
               Pin_Cmd (10) := Character'Pos ('0');
               Pin_Cmd (11) := Character'Pos ('0');
               Pin_Cmd (12 .. 23) := (others => 0);
               Send_Packet (Pin_Cmd);
            elsif Reply (1) = MSG_PIN_CODE_ACK then
               Put_Line ("Pin ack");
            elsif Reply (1) = MSG_REQUEST_CONNECTION then
               Put_Line ("Request...");
               Accpt_Cmd (0) := MSG_ACCEPT_CONNECTION;
               Accpt_Cmd (1) := 1;
               Send_Packet (Accpt_Cmd);
            elsif Reply (1) = MSG_CONNECT_RESULT then
               Put_Noupdate ("Connect ");
               if Reply (2) = 1 then
                  Put_Noupdate ("Hnd=");
                  Put_Hex (Reply (3));
               else
                  Put_Noupdate ("Err ");
                  Put_Hex (Reply (2));
               end if;
               Newline;
            elsif Reply (1) = MSG_CLOSE_CONNECTION_RESULT then
               Put_Noupdate ("Close ");
               Put_Hex (Reply (2));
               Put_Noupdate (' ');
               Put_Hex (Reply (3));
               Newline;
            else
               Put ("Got: ");
               Put_Hex (Reply (1));
               Newline;
            end if;
         end if;
      end loop;
   end Accept_Connection;

   procedure Inquiry is
   begin
      Newline_Noupdate;
      Put_Line ("Inquiry...");
      Find_Discoverable_Devices (10, Devices, Nbr_Devices);
      Wait_Key;
   end Inquiry;

   procedure Low_Inquiry is
      Cmd   : Packet (0 .. 7);
      Reply : Packet (0 .. 32);
   begin
      Nbr_Devices := 0;
      Sel_Device := Devices'First;

      Newline;

      Cmd (0) := MSG_BEGIN_INQUIRY;
      Cmd (1) := 10;    --  Max dev
      Cmd (2) := 0;     --  Timeout HI
      Cmd (3) := 20;    --  Timeout LO
      Cmd (4) := 0;
      Cmd (5) := 0;
      Cmd (6) := 0;
      Cmd (7) := 0;
      Send_Packet (Cmd);

      loop
         Receive_Packet (Reply);
         if Reply (0) /= 0 then
            if Reply (1) = MSG_INQUIRY_RUNNING then
               Put_Line ("Inquiry...");
            elsif Reply (1) = MSG_INQUIRY_STOPPED then
               Put_Line ("End of Inquiry");
               exit;
            elsif Reply (1) = MSG_INQUIRY_RESULT then
               if Nbr_Devices < Devices'Last then
                  Nbr_Devices := Nbr_Devices + 1;
                  Devices (Nbr_Devices) :=
                    (Addr => BT_Address (Reply (2 .. 8)),
                     Name => Friendly_Name (Reply (9 .. 24)),
                     Class => Class_Service (Reply (25 .. 28)));
               end if;
               Put_Noupdate ("class: ");
               for I in 25 .. 28 loop
                  Put_Hex (Reply (I));
               end loop;
               Newline;
               Put_Noupdate ('@');
               for I in 2 .. 8 loop
                  Put_Hex (Reply (I));
               end loop;
               Newline;
            else
               Put ("Got: ");
               Put_Hex (Reply (1));
               Newline;
            end if;
         else
            Wait_Ms;
         end if;
      end loop;
      Wait_Key;
   end Low_Inquiry;

   Menu_Low : constant Menu :=
     (
      (new S'("show addr"),    Show_Local_Addr'Access),
      (new S'("show name"),    Show_Friendly_Name'Access),
      (new S'("set name"),     Low_Set_Friendly_Name'Access),
      (new S'("show op mode"), Disp_Operating_Mode'Access),
      (new S'("discoverable"), Set_Discoverable'Access),
      (new S'("inquiry"),      Low_Inquiry'Access)
     );

   procedure Low_Level is
   begin
      Play_Menu (Menu_Low);
   end Low_Level;

   Main_Menu : constant Menu :=
     (
      (new S'("handle list"), Handle_List'Access),
      (new S'("known peers"), Fill_List'Access),
      (new S'("inquiry"),     Inquiry'Access),
      (new S'("accept"),      Accept_Connection'Access),
      (new S'("low level"),   Low_Level'Access)
     );

   procedure Do_Menu_Bt is
   begin
      Newline_Noupdate;
      Put_Line ("Init BT");

      NXT.Bluetooth.Initialize;

      Get_Known_Peers (Devices, Nbr_Devices);

      Play_Menu (Main_Menu);
   end Do_Menu_Bt;

end Menu_Bt;
