/* Low-level statistical profiling support function.  Mostly POSIX.1 version.
   Copyright (C) 1996,1997,1998,2002,2004,2005,2006
	Free Software Foundation, Inc.
   Copyright (C) 2007-2009, AdaCore
   This file is part of the GNU C Library.

   The GNU C Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   The GNU C Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with the GNU C Library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307 USA.  */

#include <windows.h>
#include <tlhelp32.h>
#include <stdio.h>
#include <sys/types.h>
#include <errno.h>
#include <math.h>

#include "profil.h"
#include "atomic.h"

/* thread list handling part */
WINBASEAPI HANDLE WINAPI OpenThread(DWORD,BOOL,DWORD);

struct profthread {
  HANDLE thr;    /* examined thread's handle */
  DWORD  thrid;  /* its id */
  BOOL   alive;  /* used to check if the thread is still alive */
};

struct profthreadlist {
  struct profthread *threads;
  u_int size;
} profthreadlist;

#define DEFAULT_THREADS_LIST_SIZE 1

static void
profStartThreadCheck (struct profthreadlist *p)
{
  int j;
  for (j = 0; j < p->size; j++)
    p->threads[j].alive = FALSE;
}


static HANDLE
profUpdateThread (struct profthreadlist *p, DWORD thrId)
{
  int j;
  int firstFree;
  struct profthread threadInfo;
  HANDLE t;

  firstFree = -1;

  /* First look for an existing thread */
  for (j = 0; j < p->size; j++)
    {
      threadInfo = p->threads[j];
      if (threadInfo.thrid == thrId)
	{
	  p->threads[j].alive = TRUE;
	  return threadInfo.thr;
	}
      else if (threadInfo.thrid == 0)
	firstFree = j;
    }

  /* We don't have any existing thread with this id, let's open one */
  t = OpenThread (THREAD_ALL_ACCESS, FALSE, thrId);
  if (t)
    {
      if (firstFree == -1) {
	if (p->size == 0) {
	  p->size = DEFAULT_THREADS_LIST_SIZE;
	  p->threads = malloc (p->size * sizeof (struct profthread));
	  memset (p->threads, 0, p->size * sizeof (struct profthread));
	  firstFree = 0;
	} else {
	  /* reallocate */
	  struct profthread *old;
	  u_int oldsize;
          old = p->threads;
	  oldsize = p->size;
	  p->size = p->size * 2;
          p->threads = malloc (p->size * sizeof (struct profthread));
	  memset (p->threads, 0, p->size * sizeof (struct profthread));
	  memcpy (p->threads, old, oldsize * sizeof (struct profthread));
          firstFree = oldsize;
	}
      }

      /* let's append in the treads list */

      p->threads[firstFree].thrid = thrId;
      p->threads[firstFree].thr = t;
      p->threads[firstFree].alive = TRUE;
      return t;
    }
  return NULL;
}

static void
profStopThreadCheck (struct profthreadlist *p)
{
  int j;
  for (j = 0; j < p->size; j++)
    if (!p->threads[j].alive && p->threads[j].thrid != 0) {
      CloseHandle (p->threads[j].thr);
      p->threads[j].thrid = 0;
    }
}

/* NOW, the profil methods and global variables */

static u_short *samples;
static size_t nsamples;
static size_t pc_offset;
static u_int pc_scale;
static HANDLE profthr;
static DWORD profthrid;
static DWORD profprocid;
static struct profthreadlist profthreads;
static HANDLE proftimer[2];
static HANDLE profsync;

static inline void
profil_count (HANDLE thread)
{
  void *pc = NULL;
  int res;
  CONTEXT ctx;

  /* From MSDN documentation :
   * "You cannot get a valid context for a running thread. Use the
   *  SuspendThread function to suspend the thread before calling
   *  GetThreadContext." */
  if (SuspendThread (thread) < 0) return;

  /* on x86 machines, the EIP register is the Program Counter */
  /* We retrieve it from Context */
  ctx.ContextFlags = CONTEXT_CONTROL;
  res = GetThreadContext (thread, &ctx);
  ResumeThread (thread);

  /* Invalid result, do not try to get eip from context, just return */
  if (res < 0) return;

#ifdef _WIN64
  pc = (void*)ctx.Rip;
#else
  pc = (void*)ctx.Eip;
#endif

  size_t i = (pc - pc_offset - (void *) 0) / 2;

  if (sizeof (unsigned long long int) > sizeof (size_t))
    i = (unsigned long long int) i * pc_scale / 65536;
  else
    i = i / 65536 * pc_scale + i % 65536 * pc_scale / 65536;

  if (i < nsamples)
    ++samples[i];
}

static DWORD CALLBACK
profil_thr (LPVOID arg)
{
  // release the main task
  ReleaseMutex (profsync);

  while (WaitForMultipleObjects (2, proftimer, FALSE, INFINITE) == 0)
  {
    HANDLE snapshot;

    snapshot = CreateToolhelp32Snapshot(TH32CS_SNAPTHREAD, profprocid);
    if (snapshot)
    {
      THREADENTRY32 thread;
      BOOL res;

      profStartThreadCheck (&profthreads);
      thread.dwSize = sizeof (thread);

      for (res = Thread32First (snapshot, &thread);
           res;
           res = Thread32Next (snapshot, &thread))
      {
        if ((thread.th32OwnerProcessID == profprocid) &&
            (thread.th32ThreadID != profthrid))
        {
          HANDLE t = profUpdateThread (&profthreads, thread.th32ThreadID);
	  if (t) profil_count (t);
	}
      }
      profStopThreadCheck (&profthreads);
      CloseHandle (snapshot);
    }
  }
  // printf ("Returned\n");
  CloseHandle (proftimer[0]);
  CloseHandle (proftimer[1]);
  return 0;
}

int
profil_off ()
{
  /* Stop the profiling task */
  if (proftimer[1]) ReleaseMutex (proftimer[1]);
  if (proftimer[0]) CancelWaitableTimer (proftimer[0]);

  /* Call to profStartThreadCheck immediately followed by profStopThreadCheck
     will close all remaining thread handles */
  profStartThreadCheck (&profthreads);
  profStopThreadCheck (&profthreads);
  free (profthreads.threads);
  profthreads.size = 0;
  samples = NULL;

  return 0;
}

int
profil_on ()
{
  LARGE_INTEGER liDueTime;

  /* Set the current process id */
  profprocid = GetCurrentProcessId ();

  proftimer[0] = CreateWaitableTimer (NULL, FALSE, "profTimer");
  if (!proftimer[0]) return -1;

  liDueTime.QuadPart = -1;
  if (!SetWaitableTimer
         (proftimer[0], &liDueTime, 1000 / __profile_frequency(),
          NULL, NULL, FALSE))
    return -1;

  proftimer[1] = CreateMutex (NULL, TRUE, "profterminate");
  if (!proftimer[1]) return -1;

  profsync = CreateMutex (NULL, TRUE, "profstarted");
  if (!profsync) return -1;

  /* Create the profiling thread */
  profthr = CreateThread
              (0, 0, profil_thr, NULL, CREATE_SUSPENDED, &profthrid);
  if (!profthr)
    return -1;

  /* Use a high priority thread */
  SetThreadPriority (profthr, THREAD_PRIORITY_TIME_CRITICAL);

  /* start it */
  ResumeThread (profthr);

  /* Wait until the thread is actually started */
  WaitForSingleObject (profsync, INFINITE);
  CloseHandle (profsync);

  return 0;
}

/* Enable statistical profiling, writing samples of the PC into at most
   SIZE bytes of SAMPLE_BUFFER; every processor clock tick while profiling
   is enabled, the system examines the user PC and increments
   SAMPLE_BUFFER[((PC - OFFSET) / 2) * SCALE / 65536].  If SCALE is zero,
   disable profiling.  Returns zero on success, -1 on error.  */

int
profil (u_short *sample_buffer, size_t size, size_t offset, u_int scale)
{
  if (size == 0)
    {
      /* Disable profiling.  */
      if (samples == NULL)
	/* Wasn't turned on.  */
	return 0;

      return profil_off();
    }

  if (samples)
    {
      /* Was already turned on.  Restore old timer and signal handler
	 first.  */
      if (profil_off() < 0)
	return -1;
    }

  samples = sample_buffer;
  nsamples = size / sizeof *samples;
  pc_offset = offset;
  pc_scale = scale;
  profthreads.size = 0;

  return profil_on();
}

