------------------------------------------------------------------------------
--                                                                          --
--                           GNAT PRO FOR ERC32 (DEMO)                      --
--                                                                          --
--                              W O R K L O A D                             --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                      Copyright (C) 2004 ACT Europe                       --
--                                                                          --
-- The  Open  Ravenscar  Validation  Test Suite is  free  software; you can --
-- redistribute  it  and/or  modify  it under  terms  of  the  GNU  General --
-- Public  License as published  by  the Free Software  Foundation;  either --
-- version 2,  or (at your option)  any  later  version.  The test suite is --
-- distributed  in  the  hope  that  it  will  be  useful, but WITHOUT  ANY --
-- WARRANTY;  without  even  the  implied  warranty  of  MERCHANTABILITY or --
-- FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for --
-- more details. You should have received a copy of the GNU  General Public --
-- License  distributed  with  this  test  suite; see file COPYING. If not, --
-- write to the Free  Software  Foundation,  59  Temple Place  -  Suite 30, --
-- Boston, MA 02111-1307, USA.                                              --
--                                                                          --
-- As a special exception,  if other files  instantiate  generics from this --
-- unit, or you link  this unit with other files  to produce an executable, --
-- this  unit  does not  by itself cause  the resulting  executable  to  be --
-- covered  by the  GNU  General  Public  License.  This exception does not --
-- however invalidate  any other reasons why  the executable file  might be --
-- covered by the  GNU Public License.                                      --
--                                                                          --
------------------------------------------------------------------------------

--  with Float_Math_Lib; use Float_Math_Lib;
with Ada.Numerics.Elementary_Functions;
use Ada.Numerics.Elementary_Functions;

package body Workload is

   --  subtype Whet_Float is Double;
   subtype Whet_Float is Float;

   ---------------------
   -- Small_Whetstone --
   ---------------------

   --  This version of the Whetstone benchmark is designed to have an inner
   --  loop which executes only 1000 Whetstone instructions.  This is so that
   --  smaller units of CPU time can be requested for benchmarks, especially
   --  real-time benchmarks.  The parameter "Kilo_Whets" determines the number
   --  of loop iterations; a value of 1 means the loop will execute 1000
   --  Whetstone Instructions. A Whetstone Instruction corresponds to about
   --  1.3 machine instructions on a conventional machine with floating point.
   --
   --  Small_Whetstone was developed by Brian Wichmann of the UK National
   --  Physical Laboratory (NPL).  The Ada version was translated at the
   --  Carnegie Mellon University Software Engineering Institute from the
   --  original standard Pascal language version (see references below).
   --  This Hartstone version has been adapted from the Ada standard
   --  version by making the Kilo_Whets variable a passed parameter, and
   --  by raising an exception, rather than printing an error message, if
   --  the benchmark's internal consistency check fails.
   --
   --  Small_Whetstone uses the following mathematical functions, which are
   --  listed here using the ISO/WG9 Numerics Rapporteur Group proposed
   --  standard names for functions of a Generic_Elementary_Functions library
   --  (Float_Type is a generic type definition):
   --
   --    function Cos  (X : Float_Type) return Float_Type;
   --    function Exp  (X : Float_Type) return Float_Type;
   --    function Log  (X : Float_Type) return Float_Type; -- Natural logs
   --    function Sin  (X : Float_Type) return Float_Type;
   --    function Sqrt (X : Float_Type) return Float_Type;
   --
   --  The name of the actual mathematical library and the functions it
   --  provides are implementation-dependent.  For Small_Whetstone, the
   --  function name to be careful of is the natural logarithm function;
   --  some vendors call it "Log" while others call it "Ln".  A renaming
   --  declaration has been provided to rename the function according to
   --  the ISO/WG9 name.
   --  Another implementation-dependent area is the accuracy of floating-
   --  point types.  One vendor's Float is another's Long_Float, or even
   --  Short_Float.  The subtype Whet_Float is provided so that the change
   --  can be made in a single place; users should modify it as necessary
   --  to ensure comparability of their test runs.
   --
   --  Examples of some vendor mathematical library and log function names,
   --  and the values of the 'Digits attribute for the floating-point types
   --  are provided in the comments below.  The ONLY changes a user should
   --  make to run Small_Whetstone are (a) the library name, (b) the log
   --  function name, if necessary, and (c) the definition of the subtype
   --  Whet_Float, if necessary.  Any changes should be documented along
   --  with reported results.
   --
   --  References:
   --    The first two apply only to the full version of Whetstone.  The
   --    first includes a listing of the original Algol version.  The second
   --    includes an Ada listing.  The third reference also deals mostly with
   --    the full Whetstone benchmark but in addition contains a brief
   --    rationale for the Small_Whetstone benchmark and a listing of its
   --    standard Pascal version.
   --
   --    Curnow, H.J., and Wichmann, B.A.
   --      A Synthetic Benchmark
   --      The Computer Journal, Vol. 19, No. 1, February 1976, pp. 43-49.
   --
   --    Harbaugh, S., and Forakis, J.A.
   --      Timing Studies Using a Synthetic Whetstone Benchmark
   --      Ada Letters, Vol. 4, No. 2, 1984, pp. 23-34.
   --
   --    Wichmann, B.A.,
   --      Validation Code for the Whetstone Benchmark
   --      NPL report DITC 107/88, March 1988.
   --      National Physical Laboratory,
   --      Teddington, Middlesex TW11 OLW, UK.
   --

   procedure Small_Whetstone (Kilo_Whets : in Positive) is

      T  : constant := 0.499975; --  Values from the original Algol
      T1 : constant := 0.50025;  --   Whetstone program and the
      T2 : constant := 2.0;      --   Pascal SmallWhetstone program

      N8 : constant := 10;       --  Loop iteration count for module 8
      N9 : constant :=  7;       --  Loop iteration count for module 9

      Value     : constant := 0.941377; --  Value calculated in main loop
      Tolerance : constant := 0.00001;  --  Determined by interval arithmetic

      I   : Integer;
      IJ  : Integer := 1;
      IK  : Integer := 2;
      IL  : Integer := 3;

      Y   : Whet_Float := 1.0; --  Constant within loop
      Z   : Whet_Float;
      Sum : Whet_Float := 0.0; --  Accumulates value of Z

      subtype Index is Integer range 1 .. N9;
      E1  : array (Index) of Whet_Float;

      procedure Clear_Array;

      procedure Clear_Array is
      begin
         for Loop_Var in E1'Range loop
            E1 (Loop_Var) := 0.0;
         end loop;
      end Clear_Array;

      procedure P0;

      procedure P0 is
      begin
         E1 (IJ) := E1 (IK);
         E1 (IK) := E1 (IL);
         E1 (I)  := E1 (IJ);
      end P0;

      procedure P3 (X : in Whet_Float;
                    Y : in Whet_Float;
                    Z : in out Whet_Float);

      procedure P3 (X : in Whet_Float;
                    Y : in Whet_Float;
                    Z : in out Whet_Float) is
         Xtemp : Whet_Float := T * (Z + X);
         Ytemp : Whet_Float := T * (Xtemp + Y);
      begin
         Z := (Xtemp + Ytemp) / T2;
      end P3;

   begin  -- Small_Whetstone

      for Outer_Loop_Var in 1 .. Kilo_Whets loop

         Clear_Array;

         --  Module 6: Integer arithmetic

         IJ := (IK - IJ) * (IL - IK);
         IK := IL - (IK - IJ);
         IL := (IL - IK) * (IK + IL);
         E1 (IL - 1) := Whet_Float (IJ + IK + IL);
         E1 (IK - 1) := Sin (Whet_Float (IL));

         --  Module 8: Procedure calls

         Z := E1 (4);
         for Inner_Loop_Var in 1 .. N8 loop
            P3 (Y * Whet_Float (Inner_Loop_Var), Y + Z, Z);
         end loop;

         --  Second version of Module 6:

         IJ := IL - (IL - 3) * IK;
         IL := (IL - IK) * (IK - IJ);
         IK := (IL - IK) * IK;
         E1 (IL - 1) := Whet_Float (IJ + IK + IL);
         E1 (IK + 1) := abs (Cos (Z));

         --  Module 9: Array references

         --  Note: In the Pascal version, the global variable I is used as both
         --       the control variable of the for loop and an array index
         --       within procedure P0.  Because the for-loop control variable
         --       of Ada is strictly local, this translation uses a while loop.

         I := 1;
         while I <= N9 loop
            P0;
            I := I + 1;
         end loop;

         --  Module 11: Standard mathematical functions

         --  Note: The actual name of the natural logarithm function used here
         --       is implementation-dependent.  See the comments above.

         Z := Sqrt (Exp (Log (E1 (N9)) / T1));

         Sum := Sum + Z;

         --  Check the current value of the loop computation

         if abs (Z - Value) > Tolerance then
            Sum := 2.0 * Sum; --  Forces error at end
            IJ := IJ + 1;     --  Prevents optimization
         end if;

      end loop;

      --  Self-validation check

      if abs (Sum / Whet_Float (Kilo_Whets) - Value) >
        Tolerance * Whet_Float (Kilo_Whets) then
         raise Workload_Failure;
      end if;

   end Small_Whetstone;

end Workload;
