------------------------------------------------------------------------------
--                                                                          --
--                         GNAT RUN-TIME COMPONENTS                         --
--                                                                          --
--                       A D A . E X C E P T I O N S                        --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--          Copyright (C) 1992-2014, Free Software Foundation, Inc.         --
--                                                                          --
-- This specification is derived from the Ada Reference Manual for use with --
-- GNAT. The copyright notice above, and the license provisions that follow --
-- apply solely to the  contents of the part following the private keyword. --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 3,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.                                     --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
-- You should have received a copy of the GNU General Public License and    --
-- a copy of the GCC Runtime Library Exception along with this program;     --
-- see the files COPYING3 and COPYING.RUNTIME respectively.  If not, see    --
-- <http://www.gnu.org/licenses/>.                                          --
--                                                                          --
-- The GNAT Ada tool chain for the JVM and .NET platforms is  maintained by --
-- AdaCore - http://www.adacore.com                                         --
--                                                                          --
------------------------------------------------------------------------------

--  This is the JGNAT version of Ada.Exceptions

--  Compiling Ada exceptions for the Java Virtual Machine (JVM) is much simpler
--  than on regular microprocessors since the JVM comes equipped with an
--  exception mechanism that maps nicely onto Ada's.

--  In the regular GNAT exception scheme, for targets that do not provide
--  direct support for exception handling, GNAT transforms exception
--  declaration and raise statements.

--  When declaring an exception E, GNAT constructs E's name as a string and
--  registers the name along with the Exception_Id corresponding to E in a hash
--  table for later retrieval by routines such as Exception_Name. Because all
--  this information is readily available in the JVM and the Java API this
--  expansion activity is no longer necessary in JGNAT.

--  For an explicit raise statment, GNAT converts the raise into a call to
--  Raise_Exception, which does two important things before raising the actual
--  exception:

--     1. It saves the current exception occurrence
--     2. It performs a call to Abort_Defer

--  To ensure that Ada and Java code be fully interoperable, we must be able to
--  properly handle an exception raised inside some Java method called by Ada
--  code. This means that the above two actions that are typically performed
--  before raising an exception in GNAT have to be deferred to the exception
--  handler.  This is possible because raising an exception is an atomic action
--  on the JVM which means that until we actually catch the exception raised it
--  is impossible for the Ada run-time to abort the task where the exception is
--  being propagated. Thus we can safely defer the saving of the exception
--  occurrence and the call to Abort_Defer to the first statement in an
--  exception handler.

--  Luckily the first statement that is generated by GNAT in an exception
--  handler is a call to Abort_Undefer, which in the case of JGNAT can be
--  replaced with a call to Update_Exception (also declared in
--  System.Soft_Links). Update_Exception saves the current occurrence passed to
--  it as a parameter and then simulates the calls to Abort_Defer and
--  Abort_Undefer at once.

--  The last piece that needs to be clarified is where do we get the current
--  exception occurrence at the beginning of an exception handler to pass to
--  routine Update_Exception. The trick is to introduce a new Intrinsic routine
--  Current_Target_Exception which the JGNAT back-end replaces with the
--  exception occurrence that the JVM makes available.

with System;

package Ada.Exceptions is
   pragma Preelaborate;
   --  In accordance with Ada 2005 AI-362

   type Exception_Id is private;
   Null_Id : constant Exception_Id;

   type Exception_Occurrence is limited private;
   type Exception_Occurrence_Access is access all Exception_Occurrence;
   Null_Occurrence : constant Exception_Occurrence;

   function Exception_Name (Id : Exception_Id) return String;
   function Exception_Name (X : Exception_Occurrence) return String;
   --  Same as Exception_Name (Exception_Identity (X))

   procedure Raise_Exception (E : Exception_Id; Message : String := "");
   --  Note: it would be really nice to give a pragma No_Return for this
   --  procedure, but it would be wrong, since Raise_Exception does return if
   --  given the null exception in Ada 95 mode. However we do special case the
   --  name in the test in the compiler for issuing a warning for a missing
   --  return after this call. Program_Error seems reasonable enough in such a
   --  case. See also the routine Raise_Exception_Always in the private part.
   --  In Ada 2005 mode the expanded code raises an exception if the routine
   --  does return, providing proper Ada 2005 semantics which does not allow
   --  this routine to return and raises an exception instead.

   function Exception_Message  (X : Exception_Occurrence) return String;

   procedure Reraise_Occurrence (X : Exception_Occurrence);
   --  Note: it would be really nice to give a pragma No_Return for this
   --  procedure, but it would be wrong, since Reraise_Occurrence does return
   --  if the argument is the null exception occurrence. See also procedure
   --  Reraise_Occurrence_Always in the private part of this package.

   function  Exception_Identity (X : Exception_Occurrence) return Exception_Id;
   function  Exception_Information (X : Exception_Occurrence) return String;

   --  Note on ordering: the compiler uses the Save_Occurrence procedure, but
   --  not the function from Rtsfind, so it is important that the procedure
   --  come first, since Rtsfind finds the first matching entity.

   procedure Save_Occurrence
     (Target : out Exception_Occurrence;
      Source : Exception_Occurrence);

   function Save_Occurrence
     (Source : Exception_Occurrence)
      return   Exception_Occurrence_Access;

private

   type Exception_Id is new System.Address;
   --  This is really the equivalent of java.lang.Class

   pragma Warnings (Off);
   Null_Id : constant Exception_Id := Exception_Id (System.Null_Address);

   type Exception_Occurrence is new System.Address;
   --  This is really the equivalent of some class that derived from
   --  java.lang.RuntimeException.

   Null_Occurrence : constant Exception_Occurrence :=
                       Exception_Occurrence (System.Null_Address);
   pragma Warnings (On);

   procedure Raise_Exception_Always (E : Exception_Id; Message : String := "");
   pragma No_Return (Raise_Exception_Always);
   --  This differs from Raise_Exception only in that the caller has determined
   --  that for sure the parameter E is not null, and that therefore the call
   --  to this procedure cannot return. The expander converts Raise_Exception
   --  calls to Raise_Exception_Always if it can determine this is the case.

   procedure Reraise_Occurrence_Always (X : Exception_Occurrence);
   pragma No_Return (Reraise_Occurrence_Always);
   --  This differs from Raise_Occurrence only in that the caller guarantees
   --  that for sure the parameter X is not the null occurrence, and that
   --  therefore this procedure cannot return. The expander uses this routine
   --  in the translation of a raise statement with no parameter (reraise).

   procedure Reraise_Occurrence_No_Defer (X : Exception_Occurrence);
   pragma No_Return (Reraise_Occurrence_No_Defer);
   --  Calls to this routine are emitted by the expander when it is known that
   --  abort is already deferred. X is known not to be the Null_Occurrence.

   function Exception_Name_Simple (X : Exception_Occurrence) return String;
   --  Like Exception_Name, but returns the simple non-qualified name of the
   --  exception. This is used to implement the Exception_Name function in
   --  Current_Exceptions (the DEC compatible unit). It is called from the
   --  compiler generated code (using Rtsfind, which does not respect the
   --  private barrier, so we can place this function in the private part
   --  where the compiler can find it, but the spec is unchanged.)

   procedure Poll;
   pragma Inline (Poll);
   --  Check for asynchronous abort

   function Current_Target_Exception return Exception_Occurrence;
   pragma Import
            (Assembler, Current_Target_Exception, "current_target_exception");
   --  When called from inside an exception handler it returns the current
   --  JVM exception. In all other cases the result returned is undefined.

   -----------------------
   -- Imported Routines --
   -----------------------

   pragma Import (Java, Exception_Identity,
                  "jgnat.adalib.GNAT_libc.exception_identity");
   pragma Import (Java, Reraise_Occurrence_No_Defer,
                  "jgnat.adalib.GNAT_libc.reraise_occurrence_no_defer");

end Ada.Exceptions;
