------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--     A D A . F I N A L I Z A T I O N . H E A P _ M A N A G E M E N T      --
--                                                                          --
--                                B o d y                                   --
--                                                                          --
--          Copyright (C) 2008-2011, Free Software Foundation, Inc.         --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 3,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.                                     --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
-- You should have received a copy of the GNU General Public License and    --
-- a copy of the GCC Runtime Library Exception along with this program;     --
-- see the files COPYING3 and COPYING.RUNTIME respectively.  If not, see    --
-- <http://www.gnu.org/licenses/>.                                          --
--                                                                          --
-- GNAT was originally developed  by the GNAT team at  New York University. --
-- Extensive contributions were provided by Ada Core Technologies Inc.      --
--                                                                          --
------------------------------------------------------------------------------

--  This is a .NET/JVM-specific version

with Ada.Exceptions;           use Ada.Exceptions;

with GNAT.IO;                  use GNAT.IO;

with System;                   use System;
with System.Address_Image;
with System.Finalization_Root; use System.Finalization_Root;

package body Ada.Finalization.Heap_Management is

   ------------
   -- Attach --
   ------------

   procedure Attach
     (Collection : in out Finalization_Collection;
      Obj        : System.Finalization_Root.Root_Controlled_Ptr)
   is
      L : constant Root_Controlled_Ptr := Collection.Objects;

   begin
      --  Do not allow the allocation of controlled objects while the
      --  associated collection is being finalized.

      if Collection.Finalization_Started then
         raise Program_Error with "allocation after finalization started";
      end if;

      L.Next.Prev := Obj;
      Obj.Next    := L.Next;
      L.Next      := Obj;
      Obj.Prev    := L;
   end Attach;

   ------------
   -- Detach --
   ------------

   procedure Detach (Obj : System.Finalization_Root.Root_Controlled_Ptr) is
   begin
      if Obj.Prev /= null
        and then Obj.Next /= null
      then
         Obj.Prev.Next := Obj.Next;
         Obj.Next.Prev := Obj.Prev;
         Obj.Prev := null;
         Obj.Next := null;
      end if;
   end Detach;

   --------------
   -- Finalize --
   --------------

   overriding procedure Finalize
     (Collection : in out Finalization_Collection)
   is
      function Head (L : Root_Controlled_Ptr) return Root_Controlled_Ptr;
      --  Return the node which comes after the dummy head

      function Is_Dummy_Head (N : Root_Controlled_Ptr) return Boolean;
      --  Determine whether a node acts as a dummy head. Such nodes do not
      --  have an actual "object" attached to them and point to themselves.

      function Is_Empty_List (L : Root_Controlled_Ptr) return Boolean;
      --  Determine whether a list is empty

      ----------
      -- Head --
      ----------

      function Head (L : Root_Controlled_Ptr) return Root_Controlled_Ptr is
      begin
         return L.Next;
      end Head;

      -------------------
      -- Is_Dummy_Head --
      -------------------

      function Is_Dummy_Head (N : Root_Controlled_Ptr) return Boolean is
      begin
         --  To be a dummy head, the node must point to itself in both
         --  directions.

         return
           N.Next /= null
             and then N.Next = N
             and then N.Prev /= null
             and then N.Prev = N;
      end Is_Dummy_Head;

      -------------------
      -- Is_Empty_List --
      -------------------

      function Is_Empty_List (L : Root_Controlled_Ptr) return Boolean is
      begin
         return L = null or else Is_Dummy_Head (L);
      end Is_Empty_List;

      Curr_Ptr : Root_Controlled_Ptr;
      Ex_Occur : Exception_Occurrence;
      Next_Ptr : Root_Controlled_Ptr;
      Raised   : Boolean := False;

   --  Start of processing for Finalize

   begin
      --  Lock the collection to prevent any allocations while the objects are
      --  being finalized. The collection remains locked because the associated
      --  access type is about to go out of scope.

      Collection.Finalization_Started := True;

      while not Is_Empty_List (Collection.Objects) loop

         --  Find the real head of the collection, skipping the dummy head

         Curr_Ptr := Head (Collection.Objects);

         --  If the dummy head is the only remaining node, all real objects
         --  have already been detached and finalized.

         if Is_Dummy_Head (Curr_Ptr) then
            exit;
         end if;

         --  Store the next node now since the detachment will destroy the
         --  reference to it.

         Next_Ptr := Curr_Ptr.Next;

         --  Remove the current node from the list

         Detach (Curr_Ptr);

         begin
            --  This call is treated specially by the compiler (see Exp_Ch6.
            --  Expand_Call) and is converted into a call to Deep_Finalize.

            Finalize (Curr_Ptr.all);

         exception
            when Fin_Except : others =>
               if not Raised then
                  Raised := True;
                  Save_Occurrence (Ex_Occur, Fin_Except);
               end if;
         end;

         Curr_Ptr := Next_Ptr;
      end loop;

      --  If the finalization of a particular node raised an exception, reraise
      --  it after the remainder of the list has been finalized.

      if Raised then
         Reraise_Occurrence (Ex_Occur);
      end if;
   end Finalize;

   ----------------
   -- Initialize --
   ----------------

   overriding procedure Initialize
     (Collection : in out Finalization_Collection)
   is
   begin
      Collection.Objects := new Root_Controlled;

      --  The dummy head must point to itself in both directions

      Collection.Objects.Next := Collection.Objects;
      Collection.Objects.Prev := Collection.Objects;
   end Initialize;

   ----------
   -- pcol --
   ----------

   procedure pcol (Collection : Finalization_Collection) is
      Head_Seen : Boolean := False;
      N_Ptr     : Root_Controlled_Ptr;

   begin
      --  Output the basic contents of the collection

      --    Collection: 0x123456789
      --    Fin_Start : TRUE <or> FALSE

      Put ("Collection: ");
      Put_Line (Address_Image (Collection'Address));

      Put ("Fin_Start : ");
      Put_Line (Collection.Finalization_Started'Img);

      --  Output all chained elements. The format is the following:

      --    ^ <or> ? <or> null
      --    |Header: 0x123456789 (dummy head)
      --    |  Prev: 0x123456789
      --    |  Next: 0x123456789
      --    V

      --  ^ - the current element points back to the correct element
      --  ? - the current element points back to an erroneous element
      --  n - the current element points back to null

      --  Header - the address of the list header
      --  Prev   - the address of the list header which the current element
      --         - points back to
      --  Next   - the address of the list header which the current element
      --         - points to
      --  (dummy head) - present if dummy head

      N_Ptr := Collection.Objects;

      while N_Ptr /= null loop
         Put_Line ("V");

         --  The current node is the head. If we have already traversed the
         --  chain, the head will be encountered again since the chain is
         --  circular.

         if N_Ptr = Collection.Objects then
            if Head_Seen then
               exit;
            else
               Head_Seen := True;
            end if;
         end if;

         --  The current element points back to null. This should never happen
         --  since the list is circular.

         if N_Ptr.Prev = null then
            Put_Line ("null (ERROR)");

         --  The current element points back to the correct element

         elsif N_Ptr.Prev.Next = N_Ptr then
            Put_Line ("^");

         --  The current element points back to an erroneous element

         else
            Put_Line ("? (ERROR)");
         end if;

         --  Output the header and fields

         Put ("|Header: ");
         Put (Address_Image (N_Ptr.all'Address));

         --  Detect the dummy head

         if N_Ptr = Collection.Objects then
            Put_Line (" (dummy head)");
         else
            Put_Line ("");
         end if;

         Put ("|  Prev: ");
         if N_Ptr.Prev = null then
            Put_Line ("null");
         else
            Put_Line (Address_Image (N_Ptr.Prev.all'Address));
         end if;

         Put ("|  Next: ");
         if N_Ptr.Next = null then
            Put_Line ("null");
         else
            Put_Line (Address_Image (N_Ptr.Next.all'Address));
         end if;

         N_Ptr := N_Ptr.Next;
      end loop;
   end pcol;

end Ada.Finalization.Heap_Management;
