/* libaddr2line.c -- convert addresses to line number and function name
   Copyright 1997, 1998, 1999, 2000, 2001, 2002, 2003, 2004, 2006, 2007
   Free Software Foundation, Inc.
   Contributed by Ulrich Lauther <Ulrich.Lauther@mchp.siemens.de>

   This file is part of GNU Binutils.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 3, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, 51 Franklin Street - Fifth Floor, Boston,
   MA 02110-1301, USA.  */
#include "config.h"
#include <string.h>
#include <stdlib.h>

#include "bfd.h"
#include "libiberty.h"
#include "demangle.h"

static asymbol **cur_syms;		/* Symbol table.  */
static bfd *cur_bfd;

static void find_address_in_section (bfd *, asection *, void *);
void convert_addresses (const char *, void *addrs[], int , char *, int *);

/* These global variables are used to pass information between
   translate_addresses and find_address_in_section.  */

static bfd_vma pc;
static const char *filename;
static const char *functionname;
static unsigned int line;
static bfd_boolean found;

/* Look for an address in a section.  This is called via
   bfd_map_over_sections.  */

static void
find_address_in_section (bfd *abfd, asection *section,
			 void *data ATTRIBUTE_UNUSED)
{
  bfd_vma vma;
  bfd_size_type size;

  if (found)
    return;

  if ((bfd_get_section_flags (abfd, section) & SEC_ALLOC) == 0)
    return;

  vma = bfd_get_section_vma (abfd, section);
  if (pc < vma)
    return;

  size = bfd_get_section_size (section);
  if (pc >= vma + size)
    return;

  found = bfd_find_nearest_line (abfd, section, cur_syms, pc - vma,
				 &filename, &functionname, &line);
}

/* Initialize global data from FILE_NAME.  */

static void
convert_init (const char *file_name)
{
  long symcount;
  long storage;
  const char *target = TARGET;
  struct bfd *abfd;
  asymbol **syms;

  bfd_init ();

  if (! bfd_set_default_target (target))
    return;

  /* Open and check the bfd.  */
  abfd = bfd_openr (file_name, target);
  if (abfd == NULL)
    return;

  if (!bfd_check_format (abfd, bfd_object)
      || (bfd_get_file_flags (abfd) & HAS_SYMS) == 0)
    goto err;

  /* Read in the symbol table.  */
  storage = bfd_get_symtab_upper_bound (abfd);
  if (storage == 0)
    goto err;

  syms = (asymbol **) xmalloc (storage);
  symcount = bfd_canonicalize_symtab (abfd, syms);
  if (symcount < 0)
    {
      free (syms);
      goto err;
    }

  cur_bfd = abfd;
  cur_syms = syms;
  return;

 err:
  bfd_close (abfd);
  return;
}

void
convert_addresses (const char *file_name,
		   void *addrs[], int n_addr, char *buf, int *len)
{
  static int init = 0;
  int j;
  char *p;

  *len = 0;

  cplus_demangle_set_style (gnat_demangling);

  if (!init)
    {
      init = 1;
      convert_init (file_name);
    }

  if (cur_bfd == NULL)
    return;

  p = buf;
  *p = 0;

  /* Add a line for each address.  */
  for (j = 0; j < n_addr; j++)
    {
      pc = (bfd_vma)addrs[j];

      functionname = NULL;
      filename = NULL;
      line = 0;

      found = FALSE;
      bfd_map_over_sections (cur_bfd, find_address_in_section, NULL);

      /* Append address.  */
      sprintf (p, "%p in ", addrs[j]);
      p += strlen (p);

      if (! found)
	{
          /* No info for this PC.  Append dummy message and continue.  */
	  strcpy (p, "?? at ??:0\n");
	  p += strlen (p);
          continue;
	}

      if (functionname == NULL || *functionname == '\0')
        {
          /* No function name.  */
          strcpy (p, "??");
          p += 2;
        }
      else
        {
          const char *n;
          char *dn;

          /* Try to demangle.  */
          dn = bfd_demangle (cur_bfd, functionname, DMGL_ANSI | DMGL_PARAMS);
          if (dn == NULL)
            n = functionname;
          else
            n = dn;
          strcpy (p, n);
          p += strlen (n);
          if (dn)
            free (dn);
        }

      if (filename == NULL)
        filename = "??";
      else
        filename = lbasename (filename);

      sprintf (p, " at %s:%u\n", filename, line);
      p += strlen (p);
    }
  *len = p - buf;
}
