------------------------------------------------------------------------------
--                                                                          --
--                               GNATemulator                               --
--                                                                          --
--                       Copyright (C) 2013, AdaCore                        --
--                                                                          --
-- GNATemulator is free software; you can redistribute it  and/or modify it --
-- under terms of the GNU General Public License as published by the Free   --
-- Software Foundation; either version 2, or (at your option) any later     --
-- version. GNATemulator is distributed in the hope that it will be useful, --
-- but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHAN-  --
-- TABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public --
-- License  for more details. You  should  have  received a copy of the GNU --
-- General Public License  distributed with GNAT; see file COPYING. If not, --
-- write  to  the Free  Software  Foundation,  59 Temple Place - Suite 330, --
-- Boston, MA 02111-1307, USA.                                              --
--                                                                          --
------------------------------------------------------------------------------

with Interfaces.VxWorks; use Interfaces.VxWorks;
with System;
with Ada.Text_IO; use Ada.Text_IO;
with Ada.Synchronous_Task_Control; use Ada.Synchronous_Task_Control;

package body uart is

   ---------------
   -- Uart Ctrl --
   ---------------

   type Reserved_28 is array (0 .. 27) of Boolean;
   for Reserved_28'Size use 28;
   pragma Pack (Reserved_28);

   type UART_CTRL_Register is record
      Enable_Interrupt : Boolean;
      Interrupt_Raised : Boolean;
      Ack_Interrupt    : Boolean;
      Data_To_Read     : Boolean;
      Reserved         : Reserved_28;
   end record;

   for UART_CTRL_Register use
      record
         Reserved         at 0 range 00 .. 27;
         Data_To_Read     at 0 range 28 .. 28;
         Ack_Interrupt    at 0 range 29 .. 29;
         Interrupt_Raised at 0 range 30 .. 30;
         Enable_Interrupt at 0 range 31 .. 31;
      end record;

   for UART_CTRL_Register'Size use 32;

   pragma Suppress_Initialization (UART_CTRL_Register);

   UART_CTRL : UART_CTRL_Register;
   pragma Atomic (UART_CTRL);
   for UART_CTRL'Address use System'To_Address (16#f800_1000#);

   UART_DATA : Character;
   pragma Atomic (UART_DATA);
   for UART_DATA'Address use System'To_Address (16#f800_1004#);

   SO : Suspension_Object;

   task Receiver is
   end Receiver;

   task body Receiver is
   begin
      loop
         --  Wait for notification from immediate handler
         Suspend_Until_True (SO);

         while UART_CTRL.Data_To_Read loop
            Put (UART_DATA);
         end loop;
      end loop;
   end Receiver;

   procedure Handler (Param : System.Address) is
   begin
      UART_CTRL.Ack_Interrupt := True;
      Set_True (SO);
   end Handler;

   function IntEnable (intLevel : int) return STATUS;
   pragma Import (C, IntEnable, "intEnable");
   function IntConnect
     (vector    : Interrupt_Vector;
      handler   : VOIDFUNCPTR;
      parameter : System.Address := System.Null_Address) return STATUS;
   pragma Import (C, IntConnect, "intConnect");

   S : Status;
begin

   S := IntConnect (INUM_TO_IVEC (Interrupt), Handler'Access);
   S := IntEnable (intLevel => Interrupt);

   UART_CTRL.Enable_Interrupt := True;

end uart;
