/****************************************************************************
 *                                                                          *
 *                           A U T O E X E C . C                            *
 *                                                                          *
 *                        Copyright (C) 2012, AdaCore                       *
 *                                                                          *
 * This program is free software;  you can redistribute it and/or modify it *
 * under terms of  the GNU General Public License as  published by the Free *
 * Softwareg Foundation;  either version 3,  or (at your option)  any later *
 * version. This progran is distributed in the hope that it will be useful, *
 * but  WITHOUT  ANY  WARRANTY;   without  even  the  implied  warranty  of *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                     *
 *                                                                          *
 *                                                                          * 
 *                                                                          * 
 *                                                                          * 
 *                                                                          *
 * You should have received a copy  of the GNU General Public License and a *
 * copy of the  GCC Runtime Library Exception along  with this program; see *
 * the  files  COPYING3  and  COPYING.RUNTIME  respectively.  If  not,  see *
 * <http://www.gnu.org/licenses/>.                                          *
 *                                                                          *
 ****************************************************************************/

#include <vxWorks.h>

/* #define USE_HD */

#include <dosFsLib.h>
#include <cbioLib.h>
#include <ramDiskCbio.h>
#include <xbdRamDisk.h>
#include <drv/hdisk/ataDrv.h>
#include <stdio.h>
#include <ioLib.h>
#include <loadLib.h>
#include <unldLib.h>
#include <usrLib.h>
#include <symLib.h>
#include <taskLib.h>
#include <shellLib.h>
#include <rebootLib.h>
#include <sysLib.h>
#include <string.h>
#include <ledLib.h>
#include <private/ledLibP.h>
#include <shellInterpLib.h>
#include <errnoLib.h>
#include <taskHookLib.h>
#include <private/taskLibP.h>

#include "config.h"

#include "gnatemu_hostfs.h"

#define RAMDISK_SIZE (8 * 1024 * 1024)
#define RAMDISK_PATH "/tmp"

#define PARAMS_ADDR  (CCSBAR + 0x80000)
#define QTRACE_START (CCSBAR + 0x81000)

/* Default stack size */
static unsigned int DEFAULT_STACK_SIZE = 3 * 1024 * 1024;

/* Return basename.  */
static const char *
basename (const char *filename)
{
  const char *result = strrchr(filename, '/');

  if (result == NULL) {
    /* Windows path */
    result = strrchr(filename, '\\');
  }

  if (result == NULL) {
    result = filename;
  } else {
    result++;
  }

  return result;
}

STATUS
error_msg (const char *msg, ...)
{
  va_list args;
  va_start (args, msg);
  vprintf (msg, args);
  va_end (args);
  return ERROR;
}

STATUS
msg (STATUS status, const char *msg, ...)
{
  va_list args;
  va_start (args, msg);
  vprintf (msg, args);
  va_end (args);

  printf ("...");
  if (status == OK)
    printf ("OK\n");
  else
    printf ("ERROR (%d)\n", status);
  return status;
}

static int task_input_fd = -1;

void
set_stdin (const char *name)
{
  task_input_fd = open(name, O_RDONLY, 0);
}

void *
find_module_entry (const char *module_name)
{
  char *entry_point_addr; /* address of the program's main entry point */
  SYM_TYPE pType;              /* temp var */
  STATUS   findsym_result;     /* result of the symFindByName */
  extern SYMTAB_ID sysSymTbl;  /* system symbol table */
  char *child_unit_name = NULL;
  char *tmp;
  char *module_name_copy;

  /* First copy the module_name.  */
  module_name_copy = (char *) malloc (strlen(module_name) + 1);
  strcpy (module_name_copy, module_name);

  /* in case a filename is passed remove the extension.  */
  tmp = strrchr (module_name_copy, '.');
  if (tmp)
    *tmp = 0;

  /* if the name of the executable is a-b-c-d then the value of
     child_unit_name will be d. This is helpful to handle cases where the Ada
     main is a in a child package.  */
  child_unit_name = strrchr (module_name_copy, '-');
  if (child_unit_name != NULL)
    child_unit_name++;
  else
    child_unit_name = module_name_copy;

  /* Get entrypoint address. We are looking for several symbol names when
     looking for the main entry point. We look in this order: child_unit_name
     (Ada), module_name (Ada), or "main" (C).  */
  findsym_result = ERROR;
  if (child_unit_name)
      findsym_result = symFindByName (sysSymTbl,
                                      child_unit_name,
                                      (char **)&entry_point_addr,&pType);
  if (findsym_result == ERROR)
      findsym_result = symFindByName (sysSymTbl,
                                      (char *)module_name_copy,
                                      (char **)&entry_point_addr,&pType);
  if (findsym_result == ERROR)
      findsym_result = symFindByName (sysSymTbl,
                                      "main",
                                      (char **)&entry_point_addr,&pType);
  if (findsym_result == ERROR) {
      return NULL;
  }

  return entry_point_addr;
}

static int is_task_hook_initialized = 0;
static WIND_TCB *running_task_tcb;

static int
task_hook (WIND_TCB *tcb)
{
  if (tcb == running_task_tcb)
    running_task_tcb = NULL;
  return OK;
}

static STATUS
blocking_task_spawn (const char *name,
                     void *entry_point,
                     int stack_size)
{
  int task_id;
  int real_stack_size = stack_size;

  /* Reserve more space for exception stack - needed by the unwinder.  */
  taskKerExcStackSize = 16 * 1024;

  /* Adjust stack size */
  if (real_stack_size == 0)
    real_stack_size = DEFAULT_STACK_SIZE;

  /* check if hook has been initialized */
  if (!is_task_hook_initialized)
    {
      if (taskDeleteHookAdd (task_hook) != OK) {
          error_msg ("cannot install kernel task hook\n");
          printErrno(errnoGet());
          return ERROR;
      }
      is_task_hook_initialized = 1;
    }

  /* Launch the task */
  task_id = taskCreate ((char *)name, 100,
                        VX_FP_TASK,
                        real_stack_size,
                        (FUNCPTR)entry_point, 0,0,0,0,0,0,0,0,0,0);
  if (task_id == ERROR) {
    error_msg ("cannot spawn task %s\n", name);
    printErrno(errnoGet());
    return ERROR;
  }
  if (task_input_fd != -1) {
     if (ioTaskStdSet(task_id, 0, task_input_fd) != OK) {
         printErrno(errnoGet());
         return ERROR;
     }
  }

  /* Retrieve the running task TCB and register it for the hook.  */
  running_task_tcb = taskTcb (task_id);
  if (running_task_tcb == NULL) {
    error_msg ("cannot get TCB for task %s\n", name);
    printErrno(errnoGet());
    return ERROR;
  }

  /* Activate task */
  if (taskActivate (task_id) != OK) {
    error_msg ("cannot activate task %s\n", name);
    printErrno(errnoGet());
    return ERROR;
  }

  /* wait for task termination */
  while (running_task_tcb != NULL)
    taskDelay(10);

  return OK;
}

void
run_kernel (const char *name)
{
  int fd;
  const char *exec_name = basename (name);
  MODULE_ID  module_id = 0;
  void *entry;
  char *pText, *pData, *pBss;

  if (hfs_copy_from_host (name, exec_name) != OK)
    return;

  /* Load the program to execute */
  fd = open (exec_name, O_RDONLY, 0644);
  if (fd == ERROR)
    {
      printf ("can't open %s: 0x%x\n", exec_name, errno);
      perror(exec_name);
      return;
    }
  pText = pData = pBss = LD_NO_ADDRESS;
  module_id = loadModuleAt (fd, LOAD_GLOBAL_SYMBOLS, &pText, &pData, &pBss);
  close (fd);
  if (module_id == 0)
    {
      /* Error during load of the program */
      printf ("Error during load\n");
      printErrno(errnoGet());
      return;
    }

  printf ("<runkernel %s at addr=%p id=%d>\n", exec_name, pText, module_id);
  /* Write a trace for gnatcoverage.  */
  *(volatile unsigned int *)QTRACE_START = pText;

  entry = find_module_entry (exec_name);
  if (entry == NULL)
    {
      printf ("Cannot find entry point for %s\n", exec_name);
    }
  else
    {
      blocking_task_spawn (exec_name, entry, 0);
    }
  if (0)
    unldByModuleId (module_id, 0);
  printf ("</runkernel>\n");
}


static void
run_shell (void)
{
  printf ("Run_shell\n");
  shellGenericInit (NULL, 0, NULL, NULL,
                    TRUE, FALSE, STD_IN, STD_OUT, STD_ERR);
}

static void
run_line (char *line)
{
  if (strncmp (line, "kernel ", 7) == 0)
    {
      run_kernel (line + 7);
    }
  else if (strcmp (line, "shell") == 0)
    {
      run_shell ();
      exit (0);
    }
  else
    printf ("Bad command: %s\n", line);
}

char *cmdline = (char *)PARAMS_ADDR;

static STATUS
create_ramdisk (char *mount_path, int size)
{
  device_t xbd;

  /* First create the devive.  */
  xbd = xbdRamDiskDevCreate (512, size, FALSE, mount_path);
  if (xbd == NULLDEV) {
    error_msg("cannot create ramdisk device in %s\n", mount_path);
    printErrno(errnoGet());
    return ERROR;
  }

  /* Then format it with the selected filesystem.  */
  if (dosFsVolFormat(mount_path, DOS_OPT_BLANK|DOS_OPT_QUIET, NULL) != OK) {
    error_msg ("error while formatting ram drive\n");
    printErrno(errnoGet());
    return ERROR;
  }

  return OK;
}

void
myreset (void)
{
#ifdef M85XX_PORPLLSR
  /* e500v2 (8548 and p2010) */
  *(volatile unsigned *)(CCSBAR + 0xe0000 + 0xb0) = 2;
#else
  /* wrSbc834x */
  *(volatile unsigned *)(CCSBAR + 0x0900 + 0x18) = 0x52535445;
  *(volatile unsigned *)(CCSBAR + 0x0900 + 0x1c) = 2;
#endif
}

void
autostart (void)
{
  STATUS result;

  printf ("cmdline: %s\n", cmdline);

  result = create_ramdisk(RAMDISK_PATH, RAMDISK_SIZE);
  msg (result, "Mount ramdisk in /tmp");
  chdir (RAMDISK_PATH);

#ifdef USE_HD
  result = mount_host_dir(0, "/hda");
  msg (result, "Mount host directory in /hda");
  chdir ("/hda");
#endif

  ledModeRegister (emacsLedLibInit);

  shellInterpRegister (shellInterpCInit);

  if (cmdline[0])
    {
      run_line (cmdline);
#if 1
      taskDelay(10);
      myreset ();
#endif
      reboot (BOOT_NORMAL);
    }

  run_shell ();
  return;
}
