------------------------------------------------------------------------------
--                                                                          --
--                               GNATemulator                               --
--                                                                          --
--                    Copyright (C) 2014-2015, AdaCore                      --
--                                                                          --
-- GNATemulator is free software; you can redistribute it  and/or modify it --
-- under terms of the GNU General Public License as published by the Free   --
-- Software Foundation; either version 2, or (at your option) any later     --
-- version. GNATemulator is distributed in the hope that it will be useful, --
-- but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHAN-  --
-- TABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public --
-- License  for more details. You  should  have  received a copy of the GNU --
-- General Public License  distributed with GNAT; see file COPYING. If not, --
-- write  to  the Free  Software  Foundation,  59 Temple Place - Suite 330, --
-- Boston, MA 02111-1307, USA.                                              --
--                                                                          --
------------------------------------------------------------------------------

with Ada.Containers.Indefinite_Ordered_Maps;

with Ada.Characters.Handling; use Ada.Characters.Handling;
with Ada.Directories;         use Ada.Directories;
with Ada.Strings.Unbounded;   use Ada.Strings.Unbounded;
with Ada.Text_IO;             use Ada.Text_IO;

with GNATCOLL.Traces;   use GNATCOLL.Traces;
with GNATCOLL.Projects; use GNATCOLL.Projects;
with GNATCOLL.VFS;      use GNATCOLL.VFS;
with Driver_Target;

with Errors; use Errors;
with Ada.Strings.Equal_Case_Insensitive;

package body Project is

   Emulator_Package      : aliased String := "emulator";
   Emulator_Package_List : aliased String_List :=
                             (1 => Emulator_Package'Access);

   type Attribute is (Board, Debug_Port, Switches);

   subtype List_Attribute is
     Attribute range Switches .. Switches;
   subtype String_Attribute is
     Attribute range Board .. Debug_Port;

   function "+" (A : String_Attribute) return Attribute_Pkg_String;
   function "+" (A : List_Attribute) return Attribute_Pkg_List;
   --  Build identifiers for attributes in package Coverage

   package Scv_Maps is
     new Ada.Containers.Indefinite_Ordered_Maps
       (Key_Type     => String,
        Element_Type => String);
   Scv_Map : Scv_Maps.Map;
   --  All defined scenario variables

   Env      : Project_Environment_Access;
   Prj_Tree : Project_Tree_Access;

   procedure Initialize;
   --  Initialize project environment. Target is the target prefix, or NULL
   --  for the native case.

   ---------
   -- "+" --
   ---------

   function "+" (A : String_Attribute) return Attribute_Pkg_String is
   begin
      return Build (Emulator_Package, A'Img);
   end "+";

   function "+" (A : List_Attribute) return Attribute_Pkg_List is
   begin
      return Build (Emulator_Package, A'Img);
   end "+";

   ----------------------
   -- Add_Scenario_Var --
   ----------------------

   procedure Add_Scenario_Var (Key, Value : String) is
   begin
      Scv_Map.Include (Key, Value);
   end Add_Scenario_Var;

   --------------------------
   -- Compute_Project_View --
   --------------------------

   procedure Compute_Project_View is
   begin
      if Prj_Tree /= null then
         Prj_Tree.Recompute_View;
      end if;
   end Compute_Project_View;

   ----------------
   -- Initialize --
   ----------------

   procedure Initialize is
      use Scv_Maps;
   begin
      Initialize (Env);

      --  Register attributes of package Coverage

      for A in Attribute'Range loop
         declare
            Err : constant String :=
                    Register_New_Attribute
                      (Name    => A'Img,
                       Pkg     => Emulator_Package,
                       Is_List => A in List_Attribute);
         begin
            if Err /= "" then
               Put_Line (Err);
            end if;
         end;
      end loop;

      --  Set scenario variables

      for Scv_C in Scv_Map.Iterate loop
         Change_Environment (Env.all, Key (Scv_C), Element (Scv_C));
      end loop;
   end Initialize;

   -----------------------
   -- Load_Root_Project --
   -----------------------

   procedure Load_Root_Project (Prj_Name : String) is
   begin
      if Prj_Tree /= null then
         Fatal_Error ("only one root project can be specified");
      end if;

      --  Allow activation of GNATcoll debug traces via configuration file,
      --  prior to initializing the project subsystem.

      GNATCOLL.Traces.Parse_Config_File (Filename => No_File);

      pragma Assert (Env = null);
      Initialize;
      pragma Assert (Env /= null);

      Prj_Tree := new Project_Tree;
      Prj_Tree.Load
        (Root_Project_Path => Create (+Prj_Name),
         Env               => Env,
         Packages_To_Check => Emulator_Package_List'Access,
         Recompute_View    => False,
         Errors            => Print_Error'Access);
   end Load_Root_Project;

   --------------
   -- Switches --
   --------------

   function Switches return String_List_Access is
   begin
      if Prj_Tree /= null then
         return Attribute_Value (Prj_Tree.Root_Project, +Switches);
      else
         return null;
      end if;
   end Switches;

   -----------
   -- Board --
   -----------

   function Board return String is
   begin
      if Prj_Tree /= null then
         return Attribute_Value (Prj_Tree.Root_Project, +Board);
      else
         return "";
      end if;
   end Board;

   ----------------
   -- Debug_Port --
   ----------------

   function Debug_Port return String is
      Default_Port : constant String := "1234";
   begin
      if Prj_Tree /= null then
         return Attribute_Value (Prj_Tree.Root_Project, +Debug_Port,
                                 Default => Default_Port);
      else
         return Default_Port;
      end if;
   end Debug_Port;

   ------------------------
   -- Board_From_Runtime --
   ------------------------

   function Board_From_Runtime return String is

      function Ends_With (Str : String; Pattern : String) return Boolean
      is
         From : Integer := Str'Last - Pattern'Length + 1;
         To   : Integer := Str'Last;
      begin
         if Str'Length < Pattern'Length then
            return False;
         else
            return Ada.Strings.Equal_Case_Insensitive
              (Str (From .. To), Pattern);
         end if;
      end Ends_With;

   begin
      if Prj_Tree = null then
         return "";
      end if;

      declare
         RTS : String := Get_Runtime (Root_Project (Prj_Tree.all));
      begin
         if RTS /= "" then
            for Driver of Driver_Target.Driver_Table loop
               if Ends_With (RTS, Driver.RTS_Keyword.all) then
                  return Driver.Board.all;
               end if;
            end loop;
         end if;
      end;
      return "";
   end Board_From_Runtime;

end Project;
