/****************************************************************************
 *                                                                          *
 *                         B U S _ I N T E R F A C E                        *
 *                                                                          *
 *                                  B o d y                                 *
 *                                                                          *
 *                     Copyright (C) 2012-2018, AdaCore                     *
 *                                                                          *
 * This program is free software;  you can redistribute it and/or modify it *
 * under terms of  the GNU General Public License as  published by the Free *
 * Softwareg Foundation;  either version 3,  or (at your option)  any later *
 * version. This progran is distributed in the hope that it will be useful, *
 * but  WITHOUT  ANY  WARRANTY;   without  even  the  implied  warranty  of *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                     *
 *                                                                          *
 *                                                                          * 
 *                                                                          * 
 *                                                                          * 
 *                                                                          *
 * You should have received a copy  of the GNU General Public License and a *
 * copy of the  GCC Runtime Library Exception along  with this program; see *
 * the  files  COPYING3  and  COPYING.RUNTIME  respectively.  If  not,  see *
 * <http://www.gnu.org/licenses/>.                                          *
 *                                                                          *
 ****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#ifdef _WIN32
#include <winsock2.h>
#include <windows.h>
#include <ws2tcpip.h>
#else
#include <arpa/inet.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <netinet/in.h>

/* shared memory */
#include <sys/mman.h>
#include <sys/stat.h>
#include <fcntl.h>
#endif

#include "gnat-bus-interface.h"

#include "bus_interface.h"
#include "packet_handlers.h"

#define DEVICE_CHECK_RETURN(return_value)                        \
if (dev == NULL) {                                               \
    fprintf(stderr, "%s error: "                                 \
            "Device not properly initialized\n", __func__);      \
    return return_value;                                         \
 }                                                               \


/* Devices */

Device *allocate_device(void)
{
    return calloc(sizeof(Device), 1);
}

void cleanup_device(Device *dev)
{
    gnatbus_lock_device(dev);
    if (dev->fd) {
        close(dev->fd);
        dev->fd = 0;
    }
    gnatbus_unlock_device(dev);
}

int set_device_info(Device           *dev,
                    void             *opaque,
                    uint32_t          vendor_id,
                    uint32_t          device_id,
                    DeviceEndianness  endianness,
                    const char       *name,
                    const char       *desc)
{
    DEVICE_CHECK_RETURN(-1);

    if (name == NULL) {
        fprintf(stderr, "%s: %s error: device name missing\n",
                dev->register_packet.name, __func__);
        return -1;
    }

    GnatBusPacket_Register_Init(&dev->register_packet);

    dev->opaque = opaque;
    dev->register_packet.bus_version = GNATBUS_VERSION;
    dev->register_packet.vendor_id = vendor_id;
    dev->register_packet.device_id = device_id;
    dev->register_packet.endianness = endianness;

    /* Copy device name */
    strncpy(dev->register_packet.name, name, NAME_LENGTH - 1);

    /* Set trailing null character */
    dev->register_packet.name[NAME_LENGTH - 1] = '\0';

    /* Copy device description */
    strncpy(dev->register_packet.desc, desc, DESC_LENGTH - 1);

    /* Set trailing null character */
    dev->register_packet.desc[NAME_LENGTH - 1] = '\0';

    /* Zeros the number of io memory */
    dev->register_packet.nr_iomem = 0;

    /* Zeros the number of shared memories */
    dev->register_packet.nr_shared_mem = 0;

    /* This implies that set_device_info must be called before any other
     * function.
     */
    gnatbus_device_init_lock(dev);
    return 0;
}

int register_callbacks(Device      *dev,
                       io_read_fn  *io_read,
                       io_write_fn *io_write,
                       reset_fn    *device_reset,
                       init_fn     *device_init,
                       exit_fn     *device_exit)
{
    DEVICE_CHECK_RETURN(-1);

    dev->io_read      = io_read;
    dev->io_write     = io_write;
    dev->device_reset = device_reset;
    dev->device_init  = device_init;
    dev->device_exit  = device_exit;

    return 0;
}

int register_io_memory(Device *dev, uint64_t base, uint64_t size)
{
    DEVICE_CHECK_RETURN(-1);

    if (dev->register_packet.nr_iomem >= MAX_IOMEM) {
        fprintf(stderr, "%s: %s error: Maximum I/O memory number reached\n",
                dev->register_packet.name, __func__);
        return -1;
    }

    dev->register_packet.iomem[dev->register_packet.nr_iomem].base = base;
    dev->register_packet.iomem[dev->register_packet.nr_iomem].size = size;
    dev->register_packet.nr_iomem++;

    return 0;
}

int register_shared_memory(Device *dev, uint64_t base, uint64_t size,
                           const char *name)
{
    DEVICE_CHECK_RETURN(-1);

#if defined(_WIN32) || defined(_WIN64)
    /* Not supported on Microsoft yet. */
    fprintf(stderr, "%s: %s error: Shared Memory is not supported on "
            "Microsoft Windows platform yet.",
            dev->register_packet.name, __func__);
    return -1;
#else /* Linux */
    struct PACKED SharedMemory *shared_mem_pkt;
    struct DeviceSharedMemory *shm;

    if (dev->register_packet.nr_shared_mem >= GNATBUS_MAX_SHARED_MEM) {
        fprintf(stderr, "%s: %s error: Maximum shared memory number "
                "reached\n", dev->register_packet.name, __func__);
        return -1;
    }

    if (strlen(name) > GNATBUS_SHARED_MEM_NAME_MAX) {
      fprintf(stderr, "%s: %s error: The name is too long\n",
              dev->register_packet.name, __func__);
      return -1;
    }

    shm = &dev->shm[dev->shared_mem_cnt];

    shm->shared_memory_fd = shm_open(name, O_RDWR | O_CREAT | O_EXCL,
                                           S_IRUSR | S_IWUSR);
    if (shm->shared_memory_fd < 0) {
      /* In this case try to unlink the shm and retry.. */
      shm_unlink(name);
      shm->shared_memory_fd = shm_open(name, O_RDWR | O_CREAT | O_EXCL,
                                             S_IRUSR | S_IWUSR);
      if (shm->shared_memory_fd < 0) {
        /* This is still not working.. That means something odd happening. */
        fprintf(stderr, "%s: %s error: shm_open failed",
                dev->register_packet.name, __func__);
        perror("\n");
        return -1;
      }
    }

    if (ftruncate(shm->shared_memory_fd, size) < 0) {
      fprintf(stderr, "ftruncate(%s, 0x%8.8x) failed", name, size);
      perror("\n");
      return -1;
    }

    shm->mmap_ptr = mmap(NULL, size,
                               PROT_READ | PROT_WRITE,
                               MAP_SHARED,
                               shm->shared_memory_fd,
                               0);

    if (shm->mmap_ptr == MAP_FAILED) {
      fprintf(stderr, "mmap(%s) failed", name);
      perror("\n");
      close(shm->shared_memory_fd);
      return -1;
    }

    /* We can close the fd right now */
    close(shm->shared_memory_fd);
    dev->shared_mem_cnt += 1;

    /*
     * We have multiple solution here.. Theorically the socket can share a
     * file descriptor.. But it seems too beautiful to be portable. So let's
     * just share the name and reopen it on the GNATEmulator side.
     */
    shared_mem_pkt =
         &dev->register_packet.shared_mem[dev->register_packet.nr_shared_mem];
    shared_mem_pkt->base = base;
    shared_mem_pkt->size = size;
    memcpy(shared_mem_pkt->name, name, GNATBUS_SHARED_MEM_NAME_MAX);
    dev->register_packet.nr_shared_mem++;
    return 0;
#endif /* !_WIN32 && !_WIN64 */
}

void *shm_map(Device *dev, int id)
{
  if (id > dev->shared_mem_cnt) {
    return NULL;
  } else {
    return dev->shm[id].mmap_ptr;
  }
}

void shm_sync(Device *dev, int id)
{
#if !defined(_WIN32) && !defined(_WIN64)
  if (id <= dev->shared_mem_cnt) {
    msync(dev->shm[id].mmap_ptr, dev->shm[id].size, MS_SYNC);
  }
#endif /* !_WIN32 && !_WIN64 */
}

#ifdef _WIN32
static void socket_cleanup(void)
{
    WSACleanup();
}
#endif

static int send_register_packet(Device *dev)
{
    GnatBusPacket_Endianness *resp = NULL;
    GnatBusPacket_Error      *err  = NULL;
    GnatBusPacket            *pck  = NULL;
    int                       error;
    uint32_t i;

    resp = (GnatBusPacket_Endianness *)
      send_and_wait_resp(dev, (GnatBusPacket_Request *)&dev->register_packet);

    if (resp != NULL && resp->parent.type == GnatBusResponse_Endianness) {
        fprintf(stderr, "Receive response from Bus Master: Device init ok\n");
        dev->target_endianness = resp->endianness;
        error = 0;
    } else {
        pck = (GnatBusPacket *)resp;
        err = (GnatBusPacket_Error *)resp;
        if (err != NULL && err->parent.type == GnatBusResponse_Error) {
            fprintf(stderr, "Error status in Bus Master response (code:%d)\n",
                    err->error_code);
        } else if (pck != NULL) {
            fprintf(stderr, "Invalid packet from Bus Master "
                    "(size:%d, type:%d)\n", pck->size, pck->type);
        } else {
            fprintf(stderr, "No packet from Bus Master\n");
        }
        error = -1;
        close (dev->fd);
    }

    free(resp);

#if !defined(_WIN32) && !defined(_WIN64)
    /*
     * At this moment we know that GNATEmulator has processed the register
     * packet and mapped all the shared memories.. Then if there is a crash
     * thoses shared memories will be automatically removed and won't stay
     * on the host.
     */
    for (i = 0; i < dev->register_packet.nr_shared_mem; i++) {
      if (shm_unlink(dev->register_packet.shared_mem[i].name) < 0) {
        fprintf(stderr, "shm_unlink failed: %s",
                dev->register_packet.shared_mem[i].name);
        perror("");
      }
    }
#endif /* Linux */
    return error;
}

int socket_init(void)
{
#ifdef _WIN32
    WSADATA Data;
    int ret, err;

    ret = WSAStartup(MAKEWORD(2,2), &Data);
    if (ret != 0) {
        err = WSAGetLastError();
        fprintf(stderr, "WSAStartup: %d\n", err);
        return -1;
    }
    atexit(socket_cleanup);
#endif
    return 0;
}

void psocket_error(const char *msg)
{
#ifdef _WIN32
    int err;
    err = WSAGetLastError();
    fprintf(stderr, "%s: winsock error %d\n", msg, err);
#else
    perror(msg);
#endif
}

int register_device(Device *dev, int port)
{
    return register_device_tcp(dev, port);
}

int register_device_tcp(Device *dev, int port)
{
    int                  error;
    int                  serversock;
    int                  optval    = 1;
    int                  c_sock    = -1;
    struct sockaddr_in   server, client;
    unsigned int         clientlen = sizeof(client);

    if (dev == NULL) {
        fprintf(stderr, "%s: %s error: invalid device\n",
                dev->register_packet.name, __func__);
        return -1;
    }

    if (dev->register_packet.bus_version == 0) {
        fprintf(stderr, "%s error: Device not initialized. "
                "Did you call set_device_info()?\n", __func__);
        return -1;
    }

    if (socket_init() < 0) {
        return -1;
    }

    /* Create the TCP socket */
    serversock = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
#ifdef _WIN32
    if (serversock == INVALID_SOCKET) {
#else
    if (serversock < 0) {
#endif
        psocket_error("Failed to create socket");
        return -1;
    }
    setsockopt(serversock, SOL_SOCKET, SO_REUSEADDR, (const char *)&optval,
			   sizeof(optval));

    /* Construct the server sockaddr_in structure */
    memset(&server, 0, sizeof(server));         /* Clear struct */
    server.sin_family      = AF_INET;           /* Internet/IP */
    server.sin_addr.s_addr = htonl(INADDR_ANY); /* Incoming addr */
    server.sin_port        = htons(port);       /* server port */

    /* Bind the server socket */
    if (bind(serversock, (struct sockaddr *) &server, sizeof(server)) < 0) {
        psocket_error("Failed to bind the server socket");
        close(serversock);
        return -1;
    }
    /* Listen on the server socket */
    if (listen(serversock, 1) < 0) {
        psocket_error("Failed to listen on server socket");
        close(serversock);
        return -1;
    }


    /* Wait for synchronous channel connection */
    c_sock = accept(serversock, (struct sockaddr *) &client, &clientlen);
    close(serversock);
    if (c_sock < 0) {
        psocket_error("Failed to accept connection");
        return -1;
    }

    fprintf(stderr, "Bus Master connected: %s\n", inet_ntoa(client.sin_addr));

    dev->fd = c_sock;

    return send_register_packet(dev);
}

#ifndef _WIN32
int register_device_named(Device *dev, const char *name)
{
    GnatBusPacket_Error *resp   = NULL;
    int                  error;
    int                  serversock;
    int                  optval = 1;
    int                  c_sock = -1;
    struct sockaddr_un   address;
    socklen_t            address_length;

    if (dev == NULL) {
        fprintf(stderr, "%s: %s error: invalid device\n",
                dev->register_packet.name, __func__);
        return -1;
    }

    if (dev->register_packet.bus_version == 0) {
        fprintf(stderr, "%s error: Device not initialized. "
                "Did you call set_device_info()?\n", __func__);
        return -1;
    }

    if (name == NULL || name[0] != '@') {
        fprintf(stderr, "%s: %s error: invalid name '%s'"
                " (should start with @)\n",
                dev->register_packet.name, __func__, name);
        return -1;
    }

    if (socket_init() < 0) {
        return -1;
    }

    /* Create the UNIX domain socket */
    serversock = socket(PF_UNIX, SOCK_STREAM, 0);
    if (serversock < 0) {
        psocket_error("Failed to create socket");
        return -1;
    }

    /* start with a clean address structure */
    memset(&address, 0, sizeof(struct sockaddr_un));

    address.sun_family = AF_UNIX;
    /* Abstract UDS */
    snprintf(address.sun_path + 1, sizeof(address.sun_path) - 1,
             "/gnatbus/%s", name + 1);

    /* Bind the server socket */
    if(bind(serversock, (struct sockaddr *) &address,
            sizeof(struct sockaddr_un)) < 0) {
        psocket_error("Failed to bind the server socket");
        close(serversock);
        return -1;
    }

    /* Listen on the server socket */
    if (listen(serversock, 1) < 0) {
        psocket_error("Failed to listen on server socket");
        close(serversock);
        return -1;
    }

    address_length = sizeof(address);
    c_sock = accept(serversock, (struct sockaddr *) &address, &address_length);
    close(serversock);
    if (c_sock < 0) {
        psocket_error("Failed to accept connection");
        return -1;
    }

    fprintf(stderr, "Bus Master connected: %s\n", name);

    dev->fd = c_sock;

    return send_register_packet(dev);
}

#else  /* ! _WIN32 */

int register_device_named(Device *dev, const char *name)
{
    BOOL   fConnected = FALSE;
    DWORD  dwThreadId = 0;
    char openname[256];

    if (name == NULL || name[0] != '@') {
        fprintf(stderr, "%s: %s error: invalid name '%s'"
                " (should start with @)\n",
                dev->register_packet.name, __func__, name);
        return -1;
    }
    snprintf(openname, sizeof(openname), "\\\\.\\pipe\\gnatbus\\%s", name + 1);

    dev->is_pipe = 1;
    dev->hPipe = CreateNamedPipe(
        openname,                        // pipe name
        PIPE_ACCESS_DUPLEX,              // read/write access
        PIPE_TYPE_BYTE |                 // pipe mode
        PIPE_READMODE_BYTE |
        PIPE_WAIT,
        PIPE_UNLIMITED_INSTANCES,        // max. instances
        sizeof(GnatBusPacket_Write) + 8, // output buffer size
        sizeof(GnatBusPacket_Write) + 8, // input buffer size
        0,                               // client time-out
        NULL);                           // default security attribute

    if (dev->hPipe == INVALID_HANDLE_VALUE)
    {
        printf("CreateNamedPipe failed, GLE=%d.\n", GetLastError());
        return -1;
    }

    fConnected = ConnectNamedPipe(dev->hPipe, NULL) ?
        TRUE : (GetLastError() == ERROR_PIPE_CONNECTED);

    if ( ! fConnected) {
        printf("ConnectNamedPipe failed, GLE=%d.\n", GetLastError());
        CloseHandle(dev->hPipe);
        return -1;
    }

    fprintf(stderr, "Bus Master connected: %s\n", name);

    return send_register_packet(dev);
}
#endif /* ! _WIN32 */

int device_loop(Device *dev)
{
    GnatBusPacket *packet = NULL;

    DEVICE_CHECK_RETURN(-1);

    while (1) {
        packet = receive_packet(dev);
        if (packet == NULL) {
            break;
        }
        process_packet(dev, packet);
        free(packet);
    }

    /*
     * Do a little cleanup before exiting as we might have an external IRQ
     * pending in an other thread.
     */
    gnatbus_lock_device(dev);
    close(dev->fd);
    dev->fd = 0;
    gnatbus_unlock_device(dev);
    return 0;
}

/* IRQ */

/* set_IRQ is thread safe. */
static int set_IRQ(Device *dev, uint8_t line, uint8_t level)
{
    GnatBusPacket_SetIRQ setIRQ;

    DEVICE_CHECK_RETURN(-1);

    GnatBusPacket_SetIRQ_Init(&setIRQ);
    setIRQ.line  = line;
    setIRQ.level = level;

    return send_packet(dev, (GnatBusPacket *)&setIRQ);
}

int IRQ_raise(Device *dev, uint8_t line)
{
    DEVICE_CHECK_RETURN(-1);

    return set_IRQ(dev, line, 1);
}

int IRQ_lower(Device *dev, uint8_t line)
{
    DEVICE_CHECK_RETURN(-1);

    return set_IRQ(dev, line, 0);
}

int IRQ_pulse(Device *dev, uint8_t line)
{
    DEVICE_CHECK_RETURN(-1);

    return set_IRQ(dev, line, 2);

}

/* DMA */

int dma_read(Device *dev, void *dest, uint64_t addr, uint64_t size)
{
    DEVICE_CHECK_RETURN(-1);

    GnatBusPacket_Read    read;
    GnatBusPacket_Data   *resp;
    uint32_t              ret = 0;

    GnatBusPacket_Read_Init(&read);

    read.address = addr;
    read.length  = size;

    resp = (GnatBusPacket_Data *)
        send_and_wait_resp(dev, (GnatBusPacket_Request *)&read);

    if (resp == NULL
        || resp->parent.type != GnatBusResponse_Data
        || resp->length != size) {
        /* bad response */
        ret = -1;
    } else {
        memcpy(dest, resp->data, size);
    }

    free(resp);
    return ret;
}

int dma_write(Device *dev, const void *src, uint64_t addr, uint64_t size)
{
    GnatBusPacket_Write  *write;
    GnatBusPacket_Error  *resp;
    uint32_t              ret = 0;

    DEVICE_CHECK_RETURN(-1);

    write = malloc(sizeof(GnatBusPacket_Write) + size);

    if (write == NULL) {
        abort();
    }

    GnatBusPacket_Write_Init(write);
    write->parent.parent.size += size;
    write->address = addr;
    write->length  = size;
    memcpy(write->data, src, size);

    resp = (GnatBusPacket_Error *)
        send_and_wait_resp(dev, (GnatBusPacket_Request *)write);

    free(write);

    if (resp == NULL
        || resp->parent.type != GnatBusResponse_Error
        || resp->error_code != 0) {
        /* bad response */
        ret = -1;
    } else {
        ret = 0;
    }

    free(resp);
    return ret;
}

/* Time */

uint64_t get_time(Device *dev)
{
    GnatBusPacket_GetTime  gettime;
    GnatBusPacket_Time    *resp;
    uint64_t               ret = 0;

    DEVICE_CHECK_RETURN(-1);

    GnatBusPacket_GetTime_Init(&gettime);

    resp = (GnatBusPacket_Time *)
        send_and_wait_resp(dev, (GnatBusPacket_Request *)&gettime);

    if (resp == NULL
        || resp->parent.type != GnatBusResponse_Time) {
        /* bad response */
        ret = (uint64_t)-1;
    } else {
        ret = resp->time;
    }
    free(resp);

    return ret;
}

int add_event(Device        *dev,
              uint64_t       expire,
              uint32_t       event_id,
              EventCallback  event)
{
    GnatBusPacket_RegisterEvent reg;

    GnatBusPacket_RegisterEvent_Init(&reg);

    reg.expire_time = expire;
    reg.event_id    = event_id;
    reg.event       = 0;
    reg.event       = (uint64_t)event;

    return send_packet(dev, (GnatBusPacket *)&reg);
}

TargetEndianness target_endianness(Device *dev)
{
    DEVICE_CHECK_RETURN(TargetEndianness_Unknown);

    return dev->target_endianness;
}

int shutdown_request(Device *dev)
{
    GnatBusPacket_Shutdown shutdown;

    GnatBusPacket_Shutdown_Init(&shutdown);

    return send_packet(dev, (GnatBusPacket *)&shutdown);
}
