------------------------------------------------------------------------------
--                                                                          --
--                          B U S I N T E R F A C E                         --
--                                                                          --
--                                  S p e c                                 --
--                                                                          --
--                     Copyright (C) 2012-2018, AdaCore                     --
--                                                                          --
-- This program is free software;  you can redistribute it and/or modify it --
-- under terms of  the GNU General Public License as  published by the Free --
-- Softwareg Foundation;  either version 3,  or (at your option)  any later --
-- version. This progran is distributed in the hope that it will be useful, --
-- but  WITHOUT  ANY  WARRANTY;   without  even  the  implied  warranty  of --
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                     --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
--                                                                          --
-- You should have received a copy  of the GNU General Public License and a --
-- copy of the  GCC Runtime Library Exception along  with this program; see --
-- the  files  COPYING3  and  COPYING.RUNTIME  respectively.  If  not,  see --
-- <http://www.gnu.org/licenses/>.                                          --
--                                                                          --
------------------------------------------------------------------------------

with Bus_Types; use Bus_Types;
with Interfaces.C;
with Interfaces.C.Strings;
with System;

package BusInterface is

   type Internal_Bus_Device is access Integer;
   type Opaque_Ptr          is access Integer;

   --------------------------------
   -- Callbacks types definition --
   --------------------------------

   type IO_Read_Access is access function
     (Opaque  : Opaque_Ptr;
      Address : Bus_Address;
      Length  : Bus_Address)
     return Bus_Data;
   pragma Convention (C, IO_Read_Access);

   type IO_Write_Access is access procedure
     (Opaque  : Opaque_Ptr;
      Address : Bus_Address;
      Length  : Bus_Address;
      Value   : Bus_Data);
   pragma Convention (C, IO_Write_Access);

   type Device_Init_Access is access procedure (Opaque : Opaque_Ptr);
   pragma Convention (C, Device_Init_Access);

   type Device_Reset_Access is access procedure (Opaque : Opaque_Ptr);
   pragma Convention (C, Device_Reset_Access);

   type Device_Exit_Access is access procedure (Opaque : Opaque_Ptr);
   pragma Convention (C, Device_Exit_Access);

   type Event_Callback_Access is access procedure (Opaque      : Opaque_Ptr;
                                                   Event_Id    : Id;
                                                   Expire_Time : Bus_Time);
   pragma Convention (C, Event_Callback_Access);

   --  Import Bus interface from C implementation

   function Allocate_Device return Internal_Bus_Device;
   pragma Import (C, Allocate_Device, "allocate_device");

   procedure Cleanup_Device (Dev : Internal_Bus_Device);
   pragma Import (C, Cleanup_Device, "cleanup_device");

   procedure Set_Device_Info (Device               : Internal_Bus_Device;
                              Opaque               : Opaque_Ptr;
                              Vendor_Id, Device_Id : Id;
                              Endianness           : Device_Endianness;
                              Name, Description    : String);

   procedure Register_IO_Memory (Device          : Internal_Bus_Device;
                                 Address, Length : Bus_Address);
   pragma Import (C, Register_IO_Memory, "register_io_memory");

   procedure Register_Shared_Memory (Device : Internal_Bus_Device;
                                     Address, Length : Bus_Address;
                                     Name : String);

   function SHM_Map (Device : Internal_Bus_Device;
                     Id : Integer) return System.Address;
   pragma Import (C, SHM_Map, "shm_map");

   procedure SHM_Sync (Device : Internal_Bus_Device;
                       Id : Integer);
   pragma Import (C, SHM_Sync, "shm_sync");

   procedure Register_Callbacks (Device       : Internal_Bus_Device;
                                 IO_Read      : IO_Read_Access;
                                 IO_Write     : IO_Write_Access;
                                 Device_Init  : Device_Init_Access;
                                 Device_Reset : Device_Reset_Access;
                                 Device_Exit  : Device_Exit_Access);
   pragma Import (C, Register_Callbacks, "register_callbacks");

   function Register_Device_Tcp
     (Device : Internal_Bus_Device;
      Port    : Bus_Port) return Interfaces.C.int;
   pragma Import (C, Register_Device_Tcp, "register_device_tcp");

   function Register_Device_Named
     (Device : Internal_Bus_Device;
      Name   : Interfaces.C.Strings.chars_ptr) return Interfaces.C.int;
   pragma Import (C, Register_Device_Named, "register_device_named");

   procedure Device_Loop (Device : Internal_Bus_Device);
   pragma Import (C, Device_Loop, "device_loop");

   procedure IRQ_Raise (Device : Internal_Bus_Device; Line : IRQ_Line);
   pragma Import (C, IRQ_Raise, "IRQ_raise");
   procedure IRQ_Lower (Device : Internal_Bus_Device; Line : IRQ_Line);
   pragma Import (C, IRQ_Lower, "IRQ_lower");
   procedure IRQ_Pulse (Device : Internal_Bus_Device; Line : IRQ_Line);
   pragma Import (C, IRQ_Pulse, "IRQ_pulse");

   function Get_Time (Device : Internal_Bus_Device) return Bus_Time;
   pragma Import (C, Get_Time, "get_time");

   procedure Add_Event (Device   : Internal_Bus_Device;
                        Expire   : Bus_Time;
                        Event_Id : Id;
                        Event    : Event_Callback_Access);
   pragma Import (C, Add_Event, "add_event");

   procedure DMA_Read (Device : Internal_Bus_Device;
                       Dest   : out Interfaces.C.char_array;
                       Addr   : Bus_Address;
                       Size   : Bus_Data);
   pragma Import (C, DMA_Read, "dma_read");

   procedure DMA_Write (Device : Internal_Bus_Device;
                        Src    : Interfaces.C.char_array;
                        Addr   : Bus_Address;
                        Size   : Bus_Data);
   pragma Import (C, DMA_Write, "dma_write");

   function Get_Target_Endianness (Device : Internal_Bus_Device)
                                  return Interfaces.C.int;
   pragma Import (C, Get_Target_Endianness, "target_endianness");

   procedure Shutdown_Request (Device : Internal_Bus_Device);
   pragma Import (C, Shutdown_Request, "shutdown_request");

end BusInterface;
