/****************************************************************************
 *                                                                          *
 *                      G N A T E M U _ H O S T F S . C                     *
 *                                                                          *
 *                      Copyright (C) 2012-2013, AdaCore                    *
 *                                                                          *
 * This program is free software;  you can redistribute it and/or modify it *
 * under terms of  the GNU General Public License as  published by the Free *
 * Softwareg Foundation;  either version 3,  or (at your option)  any later *
 * version. This progran is distributed in the hope that it will be useful, *
 * but  WITHOUT  ANY  WARRANTY;   without  even  the  implied  warranty  of *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                     *
 *                                                                          *
 * As a special exception under Section 7 of GPL version 3, you are granted *
 * additional permissions  described in the GCC  Runtime Library Exception, *
 * version 3.1, as published by the Free Software Foundation.               *
 *                                                                          *
 * You should have received a copy  of the GNU General Public License and a *
 * copy of the  GCC Runtime Library Exception along  with this program; see *
 * the  files  COPYING3  and  COPYING.RUNTIME  respectively.  If  not,  see *
 * <http://www.gnu.org/licenses/>.                                          *
 *                                                                          *
 ****************************************************************************/

#include <vxWorksCommon.h>
#include <ioLib.h>
#include <stdio.h>
#include <errnoLib.h>
#include <usrLib.h>
#include "config.h"

#include "gnatemu_hostfs.h"

#define HOSTFS_START (CCSBAR + 0x82000)

/* Syscall IDs */
#define HOSTFS_SYSCALL_OPEN  1
#define HOSTFS_SYSCALL_READ  2
#define HOSTFS_SYSCALL_WRITE 3
#define HOSTFS_SYSCALL_CLOSE 4

static uint32_t
hfs_generic(uint32_t syscall_id, uint32_t arg1, uint32_t arg2, uint32_t arg3, uint32_t arg4, uint32_t arg5)
{
    uint32_t *hostfs_register = (uint32_t *)HOSTFS_START;

    hostfs_register[1] = arg1;
    hostfs_register[2] = arg2;
    hostfs_register[3] = arg3;
    hostfs_register[4] = arg4;
    hostfs_register[5] = arg5;

    /* Write syscall_id to launch syscall */
    hostfs_register[0] = syscall_id;
    return hostfs_register[1];
}

uint32_t hfs_open(const char *pathname, uint32_t flags, uint32_t mode)
{
    return hfs_generic(HOSTFS_SYSCALL_OPEN, (uint32_t) pathname, flags, mode, 0, 0);
}

uint32_t hfs_read(uint32_t fd, void *buf, uint32_t count)
{
    return hfs_generic(HOSTFS_SYSCALL_READ, fd, (uint32_t) buf, count, 0, 0);
}

uint32_t hfs_write(uint32_t fd, const void *buf, uint32_t count)
{
    return hfs_generic(HOSTFS_SYSCALL_WRITE, fd, (uint32_t) buf, count, 0, 0);
}

uint32_t hfs_close(uint32_t fd)
{
    return hfs_generic(HOSTFS_SYSCALL_CLOSE, fd, 0, 0, 0, 0);
}

uint32_t hfs_copy_from_host(const char *host_filename, const char *local_filename)
{
    int fd;
    int status;
    uint32_t hfs_fd = 0;
    int size = 0;
    int total_size = 0;
    char buf[1024];

    printf("copy host:%s -> local:%s\n", host_filename, local_filename);

    hfs_fd = hfs_open (host_filename, HOSTFS_O_RDONLY | HOSTFS_O_BINARY, 0644);
    if (hfs_fd == -1)
    {
        printf ("hostfs_open error %s:%d\n", host_filename, hfs_fd);
        return ERROR;
    }

    fd = open (local_filename, O_CREAT | O_WRONLY, 0644);
    if (fd == ERROR)
    {
        printf ("can't open %s: %x\n", local_filename, errno);
        perror(local_filename);
        return ERROR;
    }

    do {
        size = hfs_read(hfs_fd, buf, sizeof(buf));
        if (size < 0) {
            printf ("can't write %s: %d\n", host_filename, size);
            return ERROR;
        }
        status = write(fd, buf, size);
        if (status < 0)
        {
            printf ("can't write %s: %d\n", local_filename, status);
            printErrno(errnoGet());
            return ERROR;
        }
        total_size += size;
    } while (size == sizeof (buf));

    hfs_close(hfs_fd);
    close (fd);

    printf("%d byte(s) transferred\n", total_size);

    return OK;
}
