------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                          G N A T . O S _ L I B                           --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--          Copyright (C) 1995-2013, Free Software Foundation, Inc.         --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 3,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.                                     --
--                                                                          --
-- As a special exception under Section 7 of GPL version 3, you are granted --
-- additional permissions described in the GCC Runtime Library Exception,   --
-- version 3.1, as published by the Free Software Foundation.               --
--                                                                          --
-- You should have received a copy of the GNU General Public License and    --
-- a copy of the GCC Runtime Library Exception along with this program;     --
-- see the files COPYING3 and COPYING.RUNTIME respectively.  If not, see    --
-- <http://www.gnu.org/licenses/>.                                          --
--                                                                          --
-- GNAT was originally developed  by the GNAT team at  New York University. --
-- Extensive contributions were provided by Ada Core Technologies Inc.      --
--                                                                          --
------------------------------------------------------------------------------

--  Operating system interface facilities

--  This package contains types and procedures for interfacing to the
--  underlying OS. It is used by the GNAT compiler and by tools associated
--  with the GNAT compiler, and therefore works for the various operating
--  systems to which GNAT has been ported. This package will undoubtedly grow
--  as new services are needed by various tools.

--  This package tends to use fairly low-level Ada in order to not bring in
--  large portions of the RTL. For example, functions return access to string
--  as part of avoiding functions returning unconstrained types.

--  Except where specifically noted, these routines are portable across all
--  GNAT implementations on all supported operating systems.

--  Note: this package is in the System hierarchy so that it can be directly
--  be used by other predefined packages. User access to this package is via
--  a renaming of this package in GNAT.OS_Lib (file g-os_lib.ads).

with System;

package GNAT.OS_Lib is

   ----------------
   -- File Stuff --
   ----------------

   --  These routines give access to the open/creat/close/read/write level of
   --  I/O routines in the typical C library (these functions are not part of
   --  the ANSI C standard, but are typically available in all systems). See
   --  also package Interfaces.C_Streams for access to the stream level
   --  routines.

   --  Note on file names. If a file name is passed as type String in any of
   --  the following specifications, then the name is a normal Ada string and
   --  need not be NUL-terminated. However, a trailing NUL character is
   --  permitted, and will be ignored (more accurately, the NUL and any
   --  characters that follow it will be ignored).

   type File_Descriptor is new Integer;
   --  Corresponds to the int file handle values used in the C routines

   Standin  : constant File_Descriptor := 0;
   Standout : constant File_Descriptor := 1;
   Standerr : constant File_Descriptor := 2;
   --  File descriptors for standard input output files

   Invalid_FD : constant File_Descriptor := -1;
   --  File descriptor returned when error in opening/creating file

   type Mode is (Binary, Text);
   for Mode'Size use Integer'Size;
   for Mode use (Binary => 0, Text => 1);
   --  Used in all the Open and Create calls to specify if the file is to be
   --  opened in binary mode or text mode. In systems like Unix, this has no
   --  effect, but in systems capable of text mode translation, the use of
   --  Text as the mode parameter causes the system to do CR/LF translation
   --  and also to recognize the DOS end of file character on input. The use
   --  of Text where appropriate allows programs to take a portable Unix view
   --  of DOS-format files and process them appropriately.

   function Open_Read
     (Name  : String;
      Fmode : Mode) return File_Descriptor;
   --  Open file Name for reading, returning file descriptor File descriptor
   --  returned is Invalid_FD if file cannot be opened.

   function Open_Read_Write
     (Name  : String;
      Fmode : Mode) return File_Descriptor;
   --  Open file Name for both reading and writing, returning file descriptor.
   --  File descriptor returned is Invalid_FD if file cannot be opened.

   function Create_File
     (Name  : String;
      Fmode : Mode) return File_Descriptor;
   --  Creates new file with given name for writing, returning file descriptor
   --  for subsequent use in Write calls. If the file already exists, it is
   --  overwritten. File descriptor returned is Invalid_FD if file cannot be
   --  successfully created.

   function Create_Output_Text_File (Name : String) return File_Descriptor;
   --  Creates new text file with given name suitable to redirect standard
   --  output, returning file descriptor. File descriptor returned is
   --  Invalid_FD if file cannot be successfully created.

   function Create_New_File
     (Name  : String;
      Fmode : Mode) return File_Descriptor;
   --  Create new file with given name for writing, returning file descriptor
   --  for subsequent use in Write calls. This differs from Create_File in
   --  that it fails if the file already exists. File descriptor returned is
   --  Invalid_FD if the file exists or cannot be created.

   procedure Close (FD : File_Descriptor; Status : out Boolean);
   --  Close file referenced by FD. Status is False if the underlying service
   --  failed. Reasons for failure include: disk full, disk quotas exceeded
   --  and invalid file descriptor (the file may have been closed twice).

   procedure Close (FD : File_Descriptor);
   --  Close file referenced by FD. This form is used when the caller wants to
   --  ignore any possible error (see above for error cases).

   procedure Delete_File (Name : String; Success : out Boolean);
   --  Deletes file. Success is set True or False indicating if the delete is
   --  successful.

   function Read
     (FD : File_Descriptor;
      A  : System.Address;
      N  : Integer) return Integer;
   --  Read N bytes to address A from file referenced by FD. Returned value is
   --  count of bytes actually read, which can be less than N at EOF.

   function Write
     (FD : File_Descriptor;
      A  : System.Address;
      N  : Integer) return Integer;
   --  Write N bytes from address A to file referenced by FD. The returned
   --  value is the number of bytes written, which can be less than N if a
   --  disk full condition was detected.

   Seek_Cur : constant := 1;
   Seek_End : constant := 2;
   Seek_Set : constant := 0;
   --  Used to indicate origin for Lseek call

   procedure Lseek
     (FD     : File_Descriptor;
      offset : Long_Integer;
      origin : Integer);
   --  Sets the current file pointer to the indicated offset value, relative
   --  to the current position (origin = SEEK_CUR), end of file (origin =
   --  SEEK_END), or start of file (origin = SEEK_SET).

   subtype C_File_Name is System.Address;
   --  This subtype is used to document that a parameter is the address of a
   --  null-terminated string containing the name of a file.

   --  All the following functions need comments ???

   function Open_Read
     (Name  : C_File_Name;
      Fmode : Mode) return File_Descriptor;

   function Open_Read_Write
     (Name  : C_File_Name;
      Fmode : Mode) return File_Descriptor;

   function Create_Output_Text_File
     (Name : C_File_Name) return File_Descriptor;

   function Create_File
     (Name  : C_File_Name;
      Fmode : Mode) return File_Descriptor;

   function Create_New_File
     (Name  : C_File_Name;
      Fmode : Mode) return File_Descriptor;

   procedure Delete_File (Name : C_File_Name; Success : out Boolean);

end GNAT.OS_Lib;
