------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                          G N A T . O S _ L I B                           --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                     Copyright (C) 1995-2013, AdaCore                     --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 3,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.                                     --
--                                                                          --
-- As a special exception under Section 7 of GPL version 3, you are granted --
-- additional permissions described in the GCC Runtime Library Exception,   --
-- version 3.1, as published by the Free Software Foundation.               --
--                                                                          --
-- You should have received a copy of the GNU General Public License and    --
-- a copy of the GCC Runtime Library Exception along with this program;     --
-- see the files COPYING3 and COPYING.RUNTIME respectively.  If not, see    --
-- <http://www.gnu.org/licenses/>.                                          --
--                                                                          --
-- GNAT was originally developed  by the GNAT team at  New York University. --
-- Extensive contributions were provided by Ada Core Technologies Inc.      --
--                                                                          --
------------------------------------------------------------------------------

with Ada.Unchecked_Conversion;
with Interfaces;

with Host_FS_Params;

package body GNAT.OS_Lib is

   type HFS_u32 is new Interfaces.Unsigned_32;
   --  This is the base type for all data in HostFS

   function To_HFS_u32 is new Ada.Unchecked_Conversion
     (Source => C_File_Name, Target => HFS_u32);

   function To_HFS_u32 is new Ada.Unchecked_Conversion
     (Source => File_Descriptor, Target => HFS_u32);

   function To_FD is new Ada.Unchecked_Conversion
     (Source => HFS_u32, Target => File_Descriptor);

   function To_HFS_u32 is new Ada.Unchecked_Conversion
     (Source => Long_Integer, Target => HFS_u32);

   function To_HFS_u32 is new Ada.Unchecked_Conversion
     (Source => Integer, Target => HFS_u32);

   function To_Integer is new Ada.Unchecked_Conversion
     (Source => HFS_u32, Target => Integer);

   subtype Syscall is HFS_u32;

   HFS_SYSCALL_OPEN   : constant Syscall := 1;
   HFS_SYSCALL_READ   : constant Syscall := 2;
   HFS_SYSCALL_WRITE  : constant Syscall := 3;
   HFS_SYSCALL_CLOSE  : constant Syscall := 4;
   HFS_SYSCALL_UNLINK : constant Syscall := 5;
   HFS_SYSCALL_LSEEK  : constant Syscall := 6;

   subtype Flag is HFS_u32;

   HFS_O_RDONLY : constant Flag := 2 ** 0;
   HFS_O_WRONLY : constant Flag := 2 ** 1;
   HFS_O_CREAT  : constant Flag := 2 ** 2;
   HFS_O_RDWR   : constant Flag := 2 ** 3;
   HFS_O_APPEND : constant Flag := 2 ** 4;
   HFS_O_TRUNC  : constant Flag := 2 ** 5;
   HFS_O_BINARY : constant Flag := 2 ** 6;
   HFS_O_EXCL   : constant Flag := 2 ** 7;

   -----------------------
   -- Local Subprograms --
   -----------------------

   function HFS_Wrapper (Syscall_Id : Syscall;
                         arg1, arg2, arg3, arg4, arg5 : HFS_u32)
                         return HFS_u32;

   -----------
   -- Close --
   -----------

   procedure Close (FD : File_Descriptor) is
      Ret : HFS_u32;
      pragma Unreferenced (Ret);
   begin
      Ret := HFS_Wrapper (HFS_SYSCALL_CLOSE, To_HFS_u32 (FD), 0, 0, 0, 0);
   end Close;

   procedure Close (FD : File_Descriptor; Status : out Boolean) is
      Ret : HFS_u32;
   begin
      Ret := HFS_Wrapper (HFS_SYSCALL_CLOSE, To_HFS_u32 (FD), 0, 0, 0, 0);
      Status := (Ret = 0);
   end Close;

   -----------------
   -- Create_File --
   -----------------

   function Create_File
     (Name  : C_File_Name;
      Fmode : Mode) return File_Descriptor
   is
      Ret   : HFS_u32;
      Flags : Flag := HFS_O_WRONLY or HFS_O_CREAT or HFS_O_TRUNC;
   begin

      if Fmode = Binary then
         Flags := Flags or HFS_O_BINARY;
      end if;

      Ret := HFS_Wrapper (HFS_SYSCALL_OPEN, To_HFS_u32 (Name), Flags, 8#666#,
                          0, 0);

      return To_FD (Ret);
   end Create_File;

   function Create_File
     (Name  : String;
      Fmode : Mode) return File_Descriptor
   is
      C_Name : String (1 .. Name'Length + 1);

   begin
      C_Name (1 .. Name'Length) := Name;
      C_Name (C_Name'Last)      := ASCII.NUL;
      return Create_File (C_Name (C_Name'First)'Address, Fmode);
   end Create_File;

   ---------------------
   -- Create_New_File --
   ---------------------

   function Create_New_File
     (Name  : C_File_Name;
      Fmode : Mode) return File_Descriptor
   is
      Ret   : HFS_u32;
      Flags : Flag := HFS_O_WRONLY or HFS_O_CREAT or HFS_O_EXCL;
   begin

      if Fmode = Binary then
         Flags := Flags or HFS_O_BINARY;
      end if;

      Ret := HFS_Wrapper (HFS_SYSCALL_OPEN, To_HFS_u32 (Name), Flags, 8#666#,
                          0, 0);

      return To_FD (Ret);
   end Create_New_File;

   function Create_New_File
     (Name  : String;
      Fmode : Mode) return File_Descriptor
   is
      C_Name : String (1 .. Name'Length + 1);

   begin
      C_Name (1 .. Name'Length) := Name;
      C_Name (C_Name'Last)      := ASCII.NUL;
      return Create_New_File (C_Name (C_Name'First)'Address, Fmode);
   end Create_New_File;

   -----------------------------
   -- Create_Output_Text_File --
   -----------------------------

   function Create_Output_Text_File
     (Name : C_File_Name) return File_Descriptor
   is
      Ret   : HFS_u32;
      Flags : constant Flag :=
        HFS_O_WRONLY or HFS_O_CREAT or HFS_O_EXCL or HFS_O_TRUNC;
   begin

      Ret := HFS_Wrapper (HFS_SYSCALL_OPEN, To_HFS_u32 (Name), Flags, 8#666#,
                          0, 0);

      return To_FD (Ret);
   end Create_Output_Text_File;

   function Create_Output_Text_File (Name : String) return File_Descriptor is
      C_Name : String (1 .. Name'Length + 1);

   begin
      C_Name (1 .. Name'Length) := Name;
      C_Name (C_Name'Last)      := ASCII.NUL;
      return Create_Output_Text_File (C_Name (C_Name'First)'Address);
   end Create_Output_Text_File;

   -----------------
   -- Delete_File --
   -----------------

   procedure Delete_File (Name : C_File_Name; Success : out Boolean) is
      Ret : HFS_u32;
   begin
      Ret := HFS_Wrapper (HFS_SYSCALL_UNLINK, To_HFS_u32 (Name), 0, 0, 0, 0);
      Success := (Ret = 0);
   end Delete_File;

   procedure Delete_File (Name : String; Success : out Boolean) is
      C_Name : String (1 .. Name'Length + 1);

   begin
      C_Name (1 .. Name'Length) := Name;
      C_Name (C_Name'Last)      := ASCII.NUL;

      Delete_File (C_Name'Address, Success);
   end Delete_File;

   -----------------
   -- HFS_Generic --
   -----------------

   HFS_Registers : array (0 .. 5) of HFS_u32;
   pragma Volatile (HFS_Registers);
   for HFS_Registers'Address use Host_FS_Params.HFS_Reg_Addr;

   function HFS_Wrapper (Syscall_Id : Syscall;
                         arg1, arg2, arg3, arg4, arg5 : HFS_u32)
                         return HFS_u32
   is
   begin
      HFS_Registers (1) := arg1;
      HFS_Registers (2) := arg2;
      HFS_Registers (3) := arg3;
      HFS_Registers (4) := arg4;
      HFS_Registers (5) := arg5;

      --  Writing to register 0 triggers the syscall
      HFS_Registers (0) := Syscall_Id;

      return HFS_Registers (1);
   end HFS_Wrapper;

   -----------
   -- Lseek --
   -----------

   procedure Lseek
     (FD     : File_Descriptor;
      offset : Long_Integer;
      origin : Integer)
   is
      Ret : HFS_u32;
      pragma Unreferenced (Ret);
   begin
      Ret := HFS_Wrapper (HFS_SYSCALL_LSEEK, To_HFS_u32 (FD),
                         To_HFS_u32 (offset), To_HFS_u32 (origin), 0, 0);
   end Lseek;

   ---------------
   -- Open_Read --
   ---------------

   function Open_Read
     (Name  : C_File_Name;
      Fmode : Mode) return File_Descriptor
   is
      Ret   : HFS_u32;
      Flags : Flag := HFS_O_RDONLY;
   begin

      if Fmode = Binary then
         Flags := Flags or HFS_O_BINARY;
      end if;

      Ret := HFS_Wrapper (HFS_SYSCALL_OPEN, To_HFS_u32 (Name), Flags, 8#666#,
                          0, 0);

      return To_FD (Ret);
   end Open_Read;

   function Open_Read
     (Name  : String;
      Fmode : Mode) return File_Descriptor
   is
      C_Name : String (1 .. Name'Length + 1);
   begin
      C_Name (1 .. Name'Length) := Name;
      C_Name (C_Name'Last)      := ASCII.NUL;
      return Open_Read (C_Name (C_Name'First)'Address, Fmode);
   end Open_Read;

   ---------------------
   -- Open_Read_Write --
   ---------------------

   function Open_Read_Write
     (Name  : C_File_Name;
      Fmode : Mode) return File_Descriptor
   is
      Ret   : HFS_u32;
      Flags : Flag := HFS_O_RDWR;
   begin

      if Fmode = Binary then
         Flags := Flags or HFS_O_BINARY;
      end if;

      Ret := HFS_Wrapper (HFS_SYSCALL_OPEN, To_HFS_u32 (Name), Flags, 8#666#,
                          0, 0);

      return To_FD (Ret);
   end Open_Read_Write;

   function Open_Read_Write
     (Name  : String;
      Fmode : Mode) return File_Descriptor
   is
      C_Name : String (1 .. Name'Length + 1);
   begin
      C_Name (1 .. Name'Length) := Name;
      C_Name (C_Name'Last)      := ASCII.NUL;
      return Open_Read_Write (C_Name (C_Name'First)'Address, Fmode);
   end Open_Read_Write;

   ----------
   -- Read --
   ----------

   function Read
     (FD : File_Descriptor;
      A  : System.Address;
      N  : Integer) return Integer
   is
   begin
      return To_Integer (HFS_Wrapper (HFS_SYSCALL_READ, To_HFS_u32 (FD),
                         To_HFS_u32 (A), To_HFS_u32 (N), 0, 0));
   end Read;

   -----------
   -- Write --
   -----------

   function Write
     (FD : File_Descriptor;
      A  : System.Address;
      N  : Integer) return Integer
   is
   begin
      return To_Integer (HFS_Wrapper (HFS_SYSCALL_WRITE, To_HFS_u32 (FD),
                         To_HFS_u32 (A), To_HFS_u32 (N), 0, 0));
   end Write;

end GNAT.OS_Lib;
