------------------------------------------------------------------------------
--                                                                          --
--                               GNATemulator                               --
--                                                                          --
--                     Copyright (C) 2010-2016, AdaCore                     --
--                                                                          --
-- GNATemulator is free software; you can redistribute it  and/or modify it --
-- under terms of the GNU General Public License as published by the Free   --
-- Software Foundation; either version 2, or (at your option) any later     --
-- version. GNATemulator is distributed in the hope that it will be useful, --
-- but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHAN-  --
-- TABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public --
-- License  for more details. You  should  have  received a copy of the GNU --
-- General Public License  distributed with GNAT; see file COPYING. If not, --
-- write  to  the Free  Software  Foundation,  59 Temple Place - Suite 330, --
-- Boston, MA 02111-1307, USA.                                              --
--                                                                          --
------------------------------------------------------------------------------

with Ada.Unchecked_Conversion;
with Ada.Text_IO;      use Ada.Text_IO;
with Ada.Command_Line; use Ada.Command_Line;
with Ada.Directories;  use Ada.Directories;
with Ada.Strings.Fixed; use Ada.Strings.Fixed;

with Interfaces;

with GNAT.Strings; use GNAT.Strings;
with GNAT.OS_Lib;

with Driver_Type; use Driver_Type;
with Driver_Target; use Driver_Target;
with Driver_Constants; use Driver_Constants;
with GNAT.Command_Line;
with Project; use Project;
with Errors; use Errors;

with Ada.Containers.Doubly_Linked_Lists;
with Ada.Strings.Equal_Case_Insensitive;

procedure GNATEmu is
   --  Variables set by the command line.

   Verbose : Boolean := False;
   --  Be verbose.

   Executable : String_Access := null;
   --  Executable to run. Ensure that the value is not initialized to null.
   --  otherwise this can cause issue when using options such as --version
   --  for which no executable is passed.

   TFTP_Root_Default : constant String := ".";
   --  Default TFTP root directory

   TFTP_Root : String_Access := new String'(TFTP_Root_Default);
   --  TFTP root directory

   Vx_Kernel : String_Access := new String'("");
   --  VxWorks kernel to use

   Hostfwd : String_Access := new String'("");
   --  Default Host forward

   Driver : Driver_Target_Type := Driver_Table (1);
   --  Board driver (default: first in the table)

   Project_File : String_Access := null;
   --  Name of the project file, if any

   Default_Board : Boolean := True;

   package String_Lists is new Ada.Containers.Doubly_Linked_Lists
     (String_Access);
   use String_Lists;

   procedure Help;

   procedure Run_Command (Command : String; Options : List);
   --  Spawn command with options

   procedure Select_Board (Board : String);

   function Starts_With (Str : String; Pattern : String) return Boolean;

   function String_After
     (Str     : String;
      Pattern : String;
      Prefix  : String := "";
      Default : String := "")
      return String;

   procedure Append (Table : in out List; Str : String);

   procedure Remove (Table : in out List; Cur : in out Cursor);
   --  Remove an element from the list and set Cur to the next one, if any

   ------------
   -- Append --
   ------------

   procedure Append (Table : in out List; Str : String) is
   begin
      Append (Table, new String'(Str));
   end Append;

   -------------
   -- Remove  --
   -------------

   procedure Remove (Table : in out List; Cur : in out Cursor) is
      Next_Cur : Cursor := Next (Cur);
   begin
      Table.Delete (Cur);
      Cur := Next_Cur;
   end Remove;

   ----------
   -- Help --
   ----------

   procedure Help is

      procedure Put_Option
        (Name : String;
         Help : String;
         Alternate_Name  : String := "");

      procedure Put_Option
        (Name : String;
         Help : String;
         Alternate_Name  : String := "")
      is
         Size : Natural := 0;
      begin
         if Alternate_Name /= "" then
            Put ("  " & Name & ", " & Alternate_Name);
         else
            Put ("  " & Name);
         end if;
         Put (" ");
         Set_Col (24);
         Put_Line (Help);
      end Put_Option;

   begin
      Put_Line ("Usage: " & Command_Name & " [OPTIONS] FILE");
      Put_Line ("Options are:");
      Put_Option ("-v", "Be verbose", "--verbose");
      Put_Option ("-h", "Display this help", "--help");
      Put_Option ("-Pproj or -P proj", "Use GNAT Project File proj");
      Put_Option ("-Xnm=val",
                  "Specify an external reference for Project Files");
      Put_Option ("--version", "Display version");
      Put_Option ("--serial=null", "Redirect 1st serial port to null file");
      Put_Option ("--serial=stdio", "Redirect 1st serial port to stdio");
      Put_Option ("--serial=file:FILENAME",
                  "Redirect 1st serial port to a file (write only)");
      Put_Option ("--serial=tcp:HOST:PORT[,server]",
                  "Redirect 1st serial port to HOST:PORT via tcp.");
      Put_Option ("--serialN", "Idem as --serial for the Nth serial port");
      Put_Option ("--tftp-root=path",
                  "Set root directory of tftp server (default: "
                  & TFTP_Root_Default & ")");
      Put_Option ("--wdb[=HOST_PORT]",
                  "Set WDB redirection listen UDP port (default: 17185)");
      Put_Option ("--gdb[=PORT]",
                  "Allow gdb connection on port PORT (default port is 1234)");
      Put_Option
        ("-g",
         "Allow default debug (i.e --wdb or --gdb --freeze-on-startup");
      Put_Option ("--freeze-on-startup", "Freeze emulation on startup");
      Put_Option
        ("--gnatbus=HOST:PORT[,HOST:PORT]", "Connect a GNATBus device");
      Put_Option ("--add-memory=name=MEM_NAME,size=MEM_SIZE," &
                    "addr=MEM_ADDR[,read-only=on|off]",
                  "Add a memory bank to the emulated address space.");
      Put_Option ("",
                  "Address can be in hexadecimal (0xF0000000) and");
      Put_Option ("",
                  "size accepts (K)ilo, (M)ega, (G)iga, (T)era postfix.");
      Put_Option ("--show-memory-map", "Display the emulated address space");
      Put_Option ("--emulator-help", "Display available Qemu options");
      Put_Option ("--eargs", "Start a group of Qemu options");
      Put_Option ("--eargs-end", "End a group of Qemu options");
      if Driver.Target.all = "ppc-vx6" then
         Put_Option ("--kernel=FILE", "vxworks kernel to use");
      end if;
      if Driver_Table'Length /= 1 then
         Put_Option ("--board=BOARD_NAME", "Supported boards are:");
         for Index in Driver_Table'Range loop
            Put_Option ("", Driver_Table (Index).Board.all);
         end loop;

      end if;
   end Help;

   -----------------
   -- Run_Command --
   -----------------

   procedure Run_Command (Command : String; Options : List) is
      use Ada.Directories;

      --  Compute location of the gnatemu executable. We want to be sure that
      --  an absolute path is returned thus the call to locate_exec_on_path.
      Driver_Dir : constant String := Containing_Directory
        (GNAT.OS_Lib.Locate_Exec_On_Path (Command_Name).all);

      --  The Libexec_Dir is used to locate the qemu executable. It's a
      --  location relative to the gnatemu driver.
      Libexec_Dir : constant String := Driver_Dir &
        "/../libexec/gnatemulator/" & Driver.Target.all & "/" & Version;

      Args_List  : String_List (1 .. Integer (Options.Length));
      Success    : Boolean;
      Prg        : String_Access;

      type Token_Substitution is record
         Find, Replace_By : String_Access;
      end record;

      Args_Tokens : constant array (Positive range <>) of Token_Substitution :=
        ((Find        => new String'("$share_dir"),
          Replace_By  => new String'(Driver_Dir & "/../share/")),
         (Find        => new String'("$tftp_root"),
          Replace_By  => TFTP_Root),
         (Find        => new String'("$hostfwd"),
          Replace_By  => Hostfwd),
         (Find        => new String'("$kernel"),
          Replace_By  => Vx_Kernel)
        );

      Tok : Token_Substitution;

      procedure Substitute (Args : in out String_List;
                            Pattern : String;
                            Subst : String);
      --  If PATTERN is present in ARGS, replace it by a string
      --  containing SUBST instead of PATTERN.

      procedure Substitute (Args : in out String_List;
                            Pattern : String;
                            Subst : String) is
         T_Index    : Natural;
      begin
         for J in Args'Range loop
            T_Index := Index (Args (J).all, Pattern);

            if T_Index /= 0 then
               Args (J) := new String'(Replace_Slice (Args (J).all,
                                                      T_Index,
                                                      T_Index + Pattern'Length,
                                                      Subst));
            end if;
         end loop;
      end Substitute;

   begin
      declare
         Index : Natural := 1;
      begin
         for Opt of Options loop
            Args_List (Index) := Opt;
            Index := Index + 1;
         end loop;
      end;

      --  Find executable

      --  Try the directory where this executable is.
      Prg := new String'(Compose (Libexec_Dir, Command));
      if not GNAT.OS_Lib.Is_Executable_File (Prg.all) then
         Free (Prg);
      end if;

      if Prg = null then
         --  Try the path.
         Prg := GNAT.OS_Lib.Locate_Exec_On_Path (Command);
      end if;

      if Prg = null then
         Fatal_Error ("cannot find " & Command & " on your path");
      end if;

      --  Copy arguments and expand argument macros

      for K in Args_Tokens'Range loop
         Tok := Args_Tokens (K);
         Substitute (Args_List, Tok.Find.all, Tok.Replace_By.all);
      end loop;

      if Executable /= null then
         Substitute (Args_List, "$exe",
                     GNAT.OS_Lib.Normalize_Pathname (Executable.all));
         Substitute (Args_List, "$base_exe",
                     Simple_Name (Executable.all));
         Substitute (Args_List, "$dir_exe",
                     Containing_Directory (Executable.all));
      end if;

      if Verbose then
         Put ("exec: ");
         Put (Prg.all);
         for J in Args_List'Range loop
            Put (' ');
            Put (Args_List (J).all);
         end loop;
         New_Line;
      end if;

      --  Run

      GNAT.OS_Lib.Spawn (Prg.all, Args_List, Success);
      if not Success then
         if Verbose then
            Error ("Execution of " & Prg.all & " failed");
         end if;
         raise Exec_Error;
      end if;
   end Run_Command;

   ------------------
   -- Select_Board --
   ------------------

   procedure Select_Board (Board : String) is
      Found : Boolean := False;
   begin
      for Index in Driver_Table'Range loop
         if Ada.Strings.Equal_Case_Insensitive
           (Driver_Table (Index).Board.all, Board)
         then
            Driver := Driver_Table (Index);
            Found := True;
            Default_Board := False;
            exit;
         end if;
      end loop;
      if not Found then
         Error ("unkown board: " & Board);
         Put (Standard_Error, "Supported boards are:");
         for Index in Driver_Table'Range loop
            Put (" " & Driver_Table (Index).Board.all);
         end loop;
         New_Line (Standard_Error);
         return;
      end if;
   end Select_Board;

   function Starts_With (Str : String; Pattern : String) return Boolean
   is
   begin
      if Str'Length < Pattern'Length then
         return False;
      else
         return Str (Str'First .. Pattern'Length + Str'First - 1) = Pattern;
      end if;
   end Starts_With;

   function String_After
     (Str     : String;
      Pattern : String;
      Prefix  : String := "";
      Default : String := "")
      return String
   is
   begin
      if Starts_With (Str, Pattern) and then Str'Length > Pattern'Length then
         return Prefix & Str (Str'First + Pattern'Length .. Str'Last);
      else
         return Default;
      end if;
   end String_After;

   Switches         : List;
   Cur              : Cursor;
   Optind           : Natural := 1;
   Qemu_Args        : List;
   Serial_Args      : array (1 .. 9) of String_Access := (1 => new String'(""),
                                                          others => null);
   Serial_Last      : Integer := 1;
   Has_Serial_Redir_To_Stdio : Boolean := False;
   In_Eargs_Section : Boolean := False;
   --  First undecoded parameter.

   Need_Executable : constant Boolean :=
     Driver.Target.all /= "ppc-vx6"
       and then
     Driver.Target.all /= "e500v2-vx6"
       and then
     Driver.Target.all /= "ppc-vx7";
   --  Executable not required on Vx6/7
begin

   --  Convert args into a linked list so we can remove some of them later
   loop
      exit when Optind > Argument_Count;
      Switches.Append (new String'(Argument (Optind)));
      Optind := Optind + 1;
   end loop;

   Set_Exit_Status (Success);

   pragma Assert (Driver.Run_Command /= null);

   --  First pass for projects related switches (those swiches will be removed)
   Cur := Switches.First;
   loop
      exit when Cur = No_Element;

      declare
         Arg : constant String := Element (Cur).all;
      begin

         if Arg = "-P" then
            Optind := Optind + 1;
            if Next (Cur) = No_Element then
               Fatal_Error ("'-P' requires an argument");
            end if;

            Project_File := new String'(Element (Next (Cur)).all);

            Remove (Switches, Cur);
            Remove (Switches, Cur);
         elsif Starts_With (Arg, "-P")  then
            Project_File := new String'(Arg (Arg'First + 2 .. Arg'Last));

            Remove (Switches, Cur);
         elsif Starts_With (Arg, "-X")  then

            --  Get name and value from "-X<name>=<value>"

            declare
               Name_First, Name_Last, Value_First : Positive;
            begin
               Name_First := Arg'First + 2;
               Name_Last := Name_First - 1;
               while Name_Last < Arg'Last
                 and then Arg (Name_Last + 1) /= '='
               loop
                  Name_Last := Name_Last + 1;
               end loop;

               Value_First := Name_Last + 2;

               Add_Scenario_Var
                 (Key   => Arg (Name_First .. Name_Last),
                  Value => Arg (Value_First .. Arg'Last));
            end;

            Remove (Switches, Cur);
         else
            Cur := Next (Cur);
         end if;
      end;
   end loop;

   if Project_File /= null then
      Load_Root_Project (Project_File.all);
   end if;

   Project.Compute_Project_View;

   --  Get switches from project
   declare
      Attr : String_List_Access := Project.Switches;
   begin
      if Attr /= null then
         for Index in 0 .. Attr'Last - 1 loop
            Switches.Prepend (new String'(Attr (Attr'Last - Index).all));
         end loop;
         Free (Attr);
      end if;
   end;

   --  Get board name from project
   declare
      Attr : String := Project.Board;
   begin
      if Attr /= "" then
         Switches.Prepend (new String'("--board=" & Attr));
      end if;
   end;

   Cur := Switches.First;
   loop
      if In_Eargs_Section then
         exit when Cur = No_Element;
      else
         exit when Cur = No_Element or else
           not Starts_With (Element (Cur).all, "-");
      end if;

      declare
         Arg : constant String := Element (Cur).all;
      begin
         if In_Eargs_Section then
            if Arg = "--eargs-end" then
               In_Eargs_Section := False;
            else
               Append (Qemu_Args, Arg);
            end if;
         else
            exit when Arg'Length = 0 or else Arg (Arg'First) /= '-';

            if Arg = "-v" or else Arg = "--verbose" then
               Verbose := True;
            elsif Arg = "-h" or else Arg = "--help" then
               Help;
               return;
            elsif Arg = "--emulator-help" then
               Append (Qemu_Args, "-nographic");
               Append (Qemu_Args, "-help");
               Run_Command (Driver.Run_Command.all, Qemu_Args);
               return;
            elsif Arg = "--version" then
               Put_Line ("GNATEmulator " & Version
                         & " (" & Version_Date & ") for "
                         & Driver.Target_String.all & ", using: "
                         & Driver.Run_Command.all);
               Append (Qemu_Args, "-version");
               Append (Qemu_Args, "-nographic");
               Run_Command (Driver.Run_Command.all, Qemu_Args);
               return;
            elsif Arg = "--eargs" then
               In_Eargs_Section := True;
            elsif Starts_With (Arg, "--wdb") then
               declare
                  Host_Port : constant String :=
                    String_After (Arg, "--wdb=", Default => "17185");
               begin
                  Hostfwd := new String'(",hostfwd=udp:127.0.0.1:"
                                         & Host_Port
                                         & "-192.168.0.2:17185");
               end;
            elsif Starts_With (Arg, "--serial") then

               declare
                  Tmp : constant String := String_After (Arg, "--serial");
                  Serial_Number : Positive := 1;
                  Index : Natural := Tmp'First;
               begin
                  if Tmp'Length > 0 and then Tmp (Index) in '1' .. '9' then
                     Serial_Number := Integer'Value (Tmp (Index .. Index));
                     Index := Index + 1;
                  end if;

                  if Serial_Number not in Serial_Args'Range or else
                    (Tmp'Length > 1 and then Tmp (Index) in '0' .. '9')
                  then
                     Error ("Invalid --serial[n] option. n should be in (" &
                              Serial_Args'First'Img & " .." &
                              Serial_Args'Last'Img & " )");
                     return;
                  end if;

                  if Index + 1 > Tmp'Last or else Tmp (Index) /= '=' then
                     Error ("--serial requires an argument");
                     return;
                  end if;

                  declare
                     Serial_Arg : String_Access := new String'(String_After
                       (Tmp, Tmp (Tmp'First .. Index)));
                  begin
                     if Serial_Arg.all = "stdio" then
                        if Has_Serial_Redir_To_Stdio then
                           Error ("only one redirection to stdio is allowed");
                           return;
                        end if;
                        Serial_Arg := new String'("mon:stdio");
                        Has_Serial_Redir_To_Stdio := True;
                     end if;
                     Serial_Args (Serial_Number) := Serial_Arg;

                     if Serial_Number > Serial_Last then
                        Serial_Last := Serial_Number;
                     end if;
                  end;

               end;
            elsif Starts_With (Arg, "--tftp-root") then
               declare
                  Path : constant String := String_After (Arg, "--tftp-root=");
               begin
                  if Path'Length = 0 then
                     Error ("--tftp-root requires an argument");
                     return;
                  end if;

                  TFTP_Root := new String'(Path);
               end;
            elsif Starts_With (Arg, "--gnatbus") then
               declare
                  Devices : constant String :=
                    String_After (Arg, "--gnatbus=");
               begin
                  if Devices'Length = 0 then
                     Error ("--gnatbus requires an argument");
                     return;
                  end if;

                  Append (Qemu_Args, "-gnatbus");
                  Append (Qemu_Args, Devices);
               end;
            elsif Starts_With (Arg, "--add-memory") then
               declare
                  Mem_Arg : constant String :=
                    String_After (Arg, "--add-memory=");
               begin
                  if Mem_Arg'Length = 0 then
                     Error ("--add-memory requires an argument");
                     return;
                  end if;

                  Append (Qemu_Args, "-add-memory");
                  Append (Qemu_Args, Mem_Arg);
               end;
            elsif Starts_With (Arg, "--show-memory-map") then
               Append (Qemu_Args, "-monitor-cmd");
               Append (Qemu_Args, "info mtree");
            elsif Starts_With (Arg, "--gdb") then
               declare
                  Port : constant String :=
                    String_After (Arg, "--gdb=", "tcp::",
                                  "tcp::" & Debug_Port);
               begin
                  Append (Qemu_Args, "-gdb");
                  Append (Qemu_Args, Port);
               end;
            elsif Arg = "-g" then
               if Driver.Target.all = "ppc-vx653"
                 or else
                  Driver.Target.all = "ppc-vx6"
                 or else
                  Driver.Target.all = "e500v2-vx6"
                 or else
                  Driver.Target.all = "ppc-vx7"
               then
                  Hostfwd := new String'(",hostfwd=udp:127.0.0.1:17185"
                                           & "-192.168.0.2:17185");
               else
                  Append (Qemu_Args, "-gdb");
                  Append (Qemu_Args, "tcp::" & Debug_Port);
                  Append (Qemu_Args, "-S");
               end if;
            elsif Arg = "--freeze-on-startup" then
               Append (Qemu_Args, "-S");
            elsif Starts_With (Arg, "--kernel") then
               declare
                  Path : constant String := String_After (Arg, "--kernel=");
               begin
                  if Path'Length = 0 then
                     Error ("--kernel requires an argument");
                     return;
                  end if;
                  if Driver.Target.all /= "ppc-vx6"
                    and then
                     Driver.Target.all /= "e500v2-vx6"
                      and then
                     Driver.Target.all /= "ppc-vx7"
                  then
                     Error ("--kernel available only for VxWorks targets");
                     return;
                  end if;

                  Vx_Kernel := new String'(Path);
               end;
            elsif Starts_With (Arg, "--board") then
               declare
                  Board_Name : constant String
                    := String_After (Arg, "--board=");
               begin
                  if Board_Name /= "" then
                     Select_Board (Board_Name);
                  end if;
               end;
            else
               Error ("unknown option " & Arg);
               GNAT.Command_Line.Try_Help;
               return;
            end if;
         end if;
         Cur := Next (Cur);
      end;
   end loop;

   --  If no board was explicitly selected (from project or command line),
   --  we will try to guess board name from the run-time.
   if Default_Board and then Driver_Table'Length > 1 then
      declare
         Board : String := Board_From_Runtime;
      begin
         if Board /= "" then
            if Verbose then
               Put_Line ("Board '" & Board &
                         "' automaticaly selected from run-time.");
               Put_Line ("Use '--board=' switch to override");
            end if;
            Select_Board (Board);
         end if;
      end;
   end if;

   --  Handle redirections of serial ports
   for Index in Serial_Args'First .. Serial_Last loop
      Append (Qemu_Args, "-serial");
      if Serial_Args (Index) = null
        or else Serial_Args (Index).all = ""
      then
         if not Has_Serial_Redir_To_Stdio and Index = 1 then
            Append (Qemu_Args, "mon:stdio");
         else
            Append (Qemu_Args, "null");
         end if;
      else
         Append (Qemu_Args, Serial_Args (Index).all);
      end if;
   end loop;

   if In_Eargs_Section then
         Error ("missing --eargs-end");
   end if;

   if Cur = No_Element then
      if Need_Executable then
         Error ("missing executable/kernel filename");
         return;
      end if;
   else
      Executable := new String'(Element (Cur).all);
      Cur := Next (Cur);
   end if;

   if Cur /= No_Element then
      Error ("executable/kernel filename"
               & " must be the last command line argument");
      return;
   end if;

   if Executable /= null
     and then not GNAT.OS_Lib.Is_Regular_File (Executable.all)
   then
      Error ("cannot find executable " & Executable.all);
      return;
   end if;

   --  Add default Qemu arguments
   for Index in Driver.Run_Options'Range loop
      Append (Qemu_Args, Driver.Run_Options (Index).all);
   end loop;

   --  Specify Vx6 executable to use
   if Driver.Target.all = "ppc-vx6"
     or else
       Driver.Target.all = "e500v2-vx6"
      or else
       Driver.Target.all = "ppc-vx7"
   then
      if Vx_Kernel'Length = 0 then
         Error ("--kernel required for VxWorks");
         return;
      end if;

      if Executable /= null then
         Append (Qemu_Args, "-append");
         Append (Qemu_Args, "kernel " & Executable.all);
      end if;
   end if;

   --  Run qemu
   Run_Command (Driver.Run_Command.all, Qemu_Args);

exception
   when Exec_Error =>
      Set_Exit_Status (Failure);
end GNATEmu;
