/****************************************************************************
 *                                                                          *
 *                         B U S _ I N T E R F A C E                        *
 *                                                                          *
 *                                  S p e c                                 *
 *                                                                          *
 *                     Copyright (C) 2012-2014, AdaCore                     *
 *                                                                          *
 * This program is free software;  you can redistribute it and/or modify it *
 * under terms of  the GNU General Public License as  published by the Free *
 * Softwareg Foundation;  either version 3,  or (at your option)  any later *
 * version. This progran is distributed in the hope that it will be useful, *
 * but  WITHOUT  ANY  WARRANTY;   without  even  the  implied  warranty  of *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                     *
 *                                                                          *
 * As a special exception under Section 7 of GPL version 3, you are granted *
 * additional permissions  described in the GCC  Runtime Library Exception, *
 * version 3.1, as published by the Free Software Foundation.               *
 *                                                                          *
 * You should have received a copy  of the GNU General Public License and a *
 * copy of the  GCC Runtime Library Exception along  with this program; see *
 * the  files  COPYING3  and  COPYING.RUNTIME  respectively.  If  not,  see *
 * <http://www.gnu.org/licenses/>.                                          *
 *                                                                          *
 ****************************************************************************/

#ifndef _BUS_INTERFACE_H_
#define _BUS_INTERFACE_H_

#include <stdint.h>
#include <stdlib.h>

#ifdef _WIN32
#include <windows.h>
#endif

#include "gnat-bus-interface.h"

#ifdef __GNUC__
#define DEPRECATED(func) func __attribute__ ((deprecated))
#elif defined(_MSC_VER)
#define DEPRECATED(func) __declspec(deprecated) func
#else
#pragma message("WARNING: You need to implement DEPRECATED for this compiler")
#define DEPRECATED(func) func
#endif


typedef uint32_t target_addr_t;

/**
 * @brief   IO read callback
 *
 * This callback is called each time the emulated CPU executes a load
 * instruction in one the IO areas registered by this device.
 *
 * @param opaque  Opaque pointer set during initialization.
 * @param addr    Address of read request (absolute address).
 * @param size    Size of read request in bytes (1, 2 or 4).
 *
 * @retval ret    Value returned to load instruction.
 */
typedef uint32_t io_read_fn(void *opaque, target_addr_t addr, uint32_t size);

/**
 * @brief   IO write callback
 *
 * This callback is called each time the emulated CPU executes a store
 * instruction in one the IO areas registered by this device.
 *
 * @param opaque  Opaque pointer set during initialization.
 * @param addr    Address of write request (absolute address).
 * @param size    Size of write request in bytes (1, 2 or 4).
 * @param val     Value from store instruction.
 */
typedef void     io_write_fn(void          *opaque,
                             target_addr_t  addr,
                             uint32_t       size,
                             uint32_t       val);

/**
 * @brief   Reset callback
 *
 * Called at each system reset.
 *
 * @param opaque  Opaque pointer set during initialization.
 */
typedef void reset_fn(void *opaque);

/**
 * @brief   Initialization callback
 *
 * Called once during system initialization.
 *
 * @param opaque  Opaque pointer set during initialization.
 */
typedef void init_fn(void *opaque);

/**
 * @brief   Exit callback
 *
 * Called once after system shutdown.
 *
 * @param opaque  Opaque pointer set during initialization.
 */
typedef void exit_fn(void *opaque);

/* Devices */

typedef struct Device {
    GnatBusPacket_Register register_packet;

    int fd;

#ifdef _WIN32
    HANDLE hPipe;
    int    is_pipe;
#endif

    TargetEndianness target_endianness;

    void *opaque;

    io_read_fn  *io_read;
    io_write_fn *io_write;

    reset_fn *device_reset;
    init_fn  *device_init;
    exit_fn  *device_exit;
} Device;

/**
 * @brief   Allocate a new device.
 *
 * @retval  NULL  In case of memory allocation failure.
 * @retval  ptr   A pointer to the device.
 */
Device *allocate_device(void);

/**
 * @brief   Cleanup the device
 *
 * Does not free allocated memory.
 *
 * @param dev  A pointer to the device.
 */
void cleanup_device(Device *dev);

/**
 * @brief   Set basic information about the device.
 *
 * @param dev        A pointer to the device.
 * @param opaque     Opaque pointer provided to all callbacks (optional).
 * @param vendor_id  Vendor identifier (optional).
 * @param device_id  Device identifier (optional).
 * @param endianness Device endianness (BigEndian, LittleEndian, NativeEndian).
 * @param name       Device name.
 * @param desc       Short description of the device (optional).
 *
 * @retval 0 On success.
 */
int set_device_info(Device           *dev,
                    void             *opaque,
                    uint32_t          vendor_id,
                    uint32_t          device_id,
                    DeviceEndianness  endianness,
                    const char       *name,
                    const char       *desc);

/**
 * @brief   Set callbacks for IO and reset requests.
 *
 * @param dev           A pointer to the device.
 * @param io_read       IO read callback.
 * @param io_write      IO write callback.
 * @param device_reset  System reset callback.
 * @param device_init   System initialization callback.
 * @param device_exit   System shutdown callback.
 *
 * @retval 0 On success.
 */
int register_callbacks(Device      *dev,
                       io_read_fn  *io_read,
                       io_write_fn *io_write,
                       reset_fn    *device_reset,
                       init_fn     *device_init,
                       exit_fn     *device_exit);

/**
 * @brief   Register a memory mapped IO area.
 *
 * @param dev   A pointer to the device.
 * @param base  Base address of the area.
 * @param size  Size of the area in bytes.
 *
 * @retval 0 On success.
 */
int register_io_memory(Device        *dev,
                       target_addr_t  base,
                       target_addr_t  size);

/**
 * @deprecated Please use register_device_tcp().
 */
int DEPRECATED(register_device(Device *dev,
                               int     port));


/**
 * @brief   Start the device and wait for GNATemu connection.
 *
 * Uses TCP connection to communicate with GNATemu.
 *
 * @param dev  A pointer to the device.
 * @param port TCP port for this device.
 *
 * @retval 0 On success.
 */
int register_device_tcp(Device *dev,
                        int     port);

/**
 * @brief   Start the device and wait for GNATemu connection.
 *
 * Uses named connection to communicate with GNATemu.
 *
 * @param dev  A pointer to the device.
 * @param name Device connection name (starting with '@').
 *
 * @retval 0 On success.
 */
int register_device_named(Device     *dev,
                          const char *name);

/**
 * @brief   Execute device loop.
 *
 * This is a blocking function that waits for bus packet and process them. The
 * function returns when the connection is closed.
 *
 * @param dev  A pointer to the device.
 *
 * @retval 0 On success.
 */
int device_loop(Device *dev);

/* Time */

#define SECOND      (1000000000ULL) /**< One second in emulation time */
#define MILLISECOND (1000000ULL)    /**< One millisecond in emulation time */
#define NANOSECOND  (1000ULL)       /**< One nanosecond in emulation time */

/**
 * @brief   Type of event callbacks.
 *
 * @param opaque       Opaque pointer set during initialization.
 * @param event_id     Identifier of expired event.
 * @param expire_time  Expected expiration time for this event.
 */
typedef void (*EventCallback)(void     *opaque,
                              uint32_t  event_id,
                              uint64_t  expire_time);

/**
 * @brief   Get emulation time.
 *
 * @param dev  A pointer to the device.
 * @param event_id     Identifier of expired event.
 * @param expire_time  Expected expiration time for this event.
 *
 * @retval 0  Elapsed time since beginning of emulation.
 */
uint64_t get_time(Device *dev);

/**
 * @brief   Add a new event.
 *
 * @param dev       A pointer to the device.
 * @param expire    Expected expiration time for this event.
 * @param event_id  Event identifier.
 * @param event     Callback for this event.
 *
 * @retval 0 On success.
 */
int add_event(Device        *dev,
              uint64_t       expire,
              uint32_t       event_id,
              EventCallback  event);

/* IRQ */

/**
 * @brief   Raise IRQ line.
 *
 * @param dev   A pointer to the device.
 * @param line  IRQ line.
 *
 * @retval 0 On success.
 */
int IRQ_raise(Device *dev, uint8_t line);

/**
 * @brief   Lower IRQ line.
 *
 * @param dev   A pointer to the device.
 * @param line  IRQ line.
 *
 * @retval 0 On success.
 */
int IRQ_lower(Device *dev, uint8_t line);

/**
 * @brief   Raise and lower IRQ line.
 * @ingroup IRQ
 *
 * @param dev   A pointer to the device.
 * @param line  IRQ line.
 *
 * @retval 0 On success.
 */
int IRQ_pulse(Device *dev, uint8_t line);

/* DMA */

/**
 * @brief   Read from emulated memory.
 *
 * @param dev   A pointer to the device.
 * @param dest  Buffer to store data from memory.
 * @param addr  Base address of DMA operation.
 * @param size  Size of DMA operation in bytes.
 *
 * @retval 0 On success.
 */
int dma_read(Device *dev, void *dest, target_addr_t addr, uint32_t size);

/**
 * @brief   Write to emulated memory.
 *
 * @param dev   A pointer to the device.
 * @param dest  Data to write in memory.
 * @param addr  Base address of DMA operation.
 * @param size  Size of DMA operation in bytes.
 *
 * @retval 0 On success.
 */
int dma_write(Device *dev, const void *src, target_addr_t addr, uint32_t size);

/**
 * @brief   Return target endianness.
 *
 * @param dev   A pointer to the device.
 *
 * @retval GnatBusEndianness_Unknown, GnatBusEndianness_BigEndian or
 *         GnatBusEndianness_LittleEndian.
 */
TargetEndianness target_endianness(Device *dev);

/**
 * @brief   Send a shutdown request to the target.
 *
 * @param dev   A pointer to the device.
 *
 * @retval 0 On success.
 */
int shutdown_request(Device *dev);

#endif /* ! _BUS_INTERFACE_H_ */
