/****************************************************************************
 *                                                                          *
 *                         G N A T e m u l a t o r                          *
 *                                                                          *
 *                     Copyright (C) 2012-2014, AdaCore                     *
 *                                                                          *
 * This program is free software;  you can redistribute it and/or modify it *
 * under terms of  the GNU General Public License as  published by the Free *
 * Softwareg Foundation;  either version 3,  or (at your option)  any later *
 * version. This progran is distributed in the hope that it will be useful, *
 * but  WITHOUT  ANY  WARRANTY;   without  even  the  implied  warranty  of *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.                     *
 *                                                                          *
 * As a special exception under Section 7 of GPL version 3, you are granted *
 * additional permissions  described in the GCC  Runtime Library Exception, *
 * version 3.1, as published by the Free Software Foundation.               *
 *                                                                          *
 * You should have received a copy  of the GNU General Public License and a *
 * copy of the  GCC Runtime Library Exception along  with this program; see *
 * the  files  COPYING3  and  COPYING.RUNTIME  respectively.  If  not,  see *
 * <http://www.gnu.org/licenses/>.                                          *
 *                                                                          *
 ****************************************************************************/

#include <stdio.h>
#include <bus_interface.h>

#define BASE_ADDRESS 0x80001000

typedef struct
{
    uint32_t  ctrl_reg;
    int       index_in_str;
    Device   *dev;
} UART_ctrl;


static char get_next_char(UART_ctrl *uart)
{
    static const char *msg = "Hello World\n";

    uart->index_in_str = uart->index_in_str % 12;

    return msg[uart->index_in_str++];
}

static uint32_t io_read(void          *opaque,
                        target_addr_t  addr,
                        uint32_t       size)
{
    UART_ctrl *uart = (UART_ctrl*)opaque;
    uint32_t value = 0;;

    printf("IO read at 0x%08x\n", addr);

    switch (addr - BASE_ADDRESS) {
    case 0:
        /* To simplify we always set the bit Data_To_Read */
        value = 0x2 | uart->ctrl_reg;
        break;
    case 4:
        value = get_next_char(uart);
        break;
    default:
        printf("Read unknown register at 0x%x\n", addr - BASE_ADDRESS);
    }

    printf("return value 0x%08x\n", value);
    return value;
}

static void io_write(void          *opaque,
                     target_addr_t  addr,
                     uint32_t       size,
                     uint32_t       val)
{
    UART_ctrl *uart = (UART_ctrl*)opaque;

    printf("IO Write at 0x%08x size 0x%x value 0x%x\n", addr, size, val);

    switch (addr - BASE_ADDRESS) {
    case 0:
        uart->ctrl_reg = val;
        break;
    default:
        printf("Write unknown register at 0x%x\n", addr - BASE_ADDRESS);
    }
}

static void device_event(void     *opaque,
                         uint32_t  event_id,
                         uint64_t  expire_time)
{
    UART_ctrl *uart = (UART_ctrl*)opaque;

    printf("Event number %u triggered\n", event_id);

    /* Raise IRQ 4 to signal that we have data */
    IRQ_raise(uart->dev, 4);
}

static void device_reset(void *opaque)
{
    UART_ctrl *uart = (UART_ctrl*)opaque;

    uart->ctrl_reg = 0;
    uart->index_in_str = 0;

    /* Set an event at T = 3s */
    add_event(uart->dev, 3 * SECOND, 0x1 /* event_id */, device_event);

    printf("%s\n", __func__);
}

static void device_init(void *opaque)
{
    printf("%s\n", __func__);
}

static void device_exit(void *opaque)
{
    printf("%s\n", __func__);
}

int main(void)
{
    UART_ctrl uart;

    uart.dev = allocate_device();

    if (uart.dev == NULL) {
        printf("Device allocation failed\n");
        return 1;
    }

    if (set_device_info(uart.dev, &uart /* opaque */ , 0x1010, 0x1010,
                        DeviceEndianness_NativeEndian, "UART",
                        "Simple UART device") != 0) {
        printf("Set_device_info failed\n");
        return 1;
    }

    if (register_callbacks(uart.dev, io_read, io_write, device_reset,
                           device_init, device_exit) != 0) {
        printf("register_callbacks failed\n");
        return 1;
    }

    if (register_io_memory(uart.dev, BASE_ADDRESS, 8) != 0) {
        printf("register_io_memory failed\n");
        return 1;
    }

    if (register_device_tcp(uart.dev, 8032 /* TCP port */) != 0) {
        printf("register_device failed\n");
        return 1;
    }

    if (device_loop(uart.dev) != 0) {
        printf("device_loop failed\n");
        return 1;
    }

    cleanup_device(uart.dev);
    free(uart.dev);
}
