with Ada.Text_IO;            use Ada.Text_IO;
with GNAT.String_Split;      use GNAT.String_Split;
with Ada.Strings.Maps;       use Ada.Strings;
with Ada.Characters.Latin_1; use Ada.Characters;
with Ada.Exceptions;         use Ada.Exceptions;
with Ada.Strings.Fixed;
with Gnatwrapper_Utils;      use Gnatwrapper_Utils;
with GNATwrapper_Installation;

package body Known_Build_Specs is

   Field_Separator : constant Maps.Character_Set := Maps.To_Set (Latin_1.HT);

   --  returns file named Name with absolute path to the location of the file
   --  within the "com.adacore.gnatbench.windriver_x.y.z" plugin folder.
   function Path_To (Name : String) return String;

   -------------
   -- Mapping --
   -------------

   function Map return Vector is
      Result         : Vector;
      Mapping_File   : File_Type;
      Slices         : Slice_Set;
      File_With_Path : constant String := Path_To (Map_File_Name);

      --  note we need to read the file each time this function is called, so
      --  that a simple substitution of a new mapping file will suffice to
      --  enable new tool mappings without requiring installation of a new
      --  version of GNATbench.
   begin
      if not Is_Regular_File (File_With_Path) then
         Put_Line (Standard_Error,
                   "Could not locate spec:tool mapping file '" &
                   File_With_Path & "'");
         return Empty_Vector;
      end if;

      Open (Mapping_File, In_File, File_With_Path);

      while not End_Of_Line (Mapping_File) loop
         Create (Slices, Get_Line (Mapping_File), Field_Separator, Multiple);
         Result.Append (
            Descriptor'(Pattern => new String'(Slice (Slices, 1)),
                        Prefix  => new String'(Slice (Slices, 2))));
      end loop;

      Close (Mapping_File);

      return Result;
   exception
      when Ada.Text_IO.Name_Error =>
         Put_Line (Standard_Error,
                   "Could not open spec:tool mapping file '" &
                   File_With_Path & "'");
         return Empty_Vector;

      when GNAT.String_Split.Index_Error =>
         Put_Line (Standard_Error, "Invalid spec:tool mapping file format.");
         return Empty_Vector;

      when Error : others =>
         Put_Line (Standard_Error,
                   "Unexpected exception " &
                   Exception_Name (Error) &
                   " when handling the spec:tool mapping file.");
         return Empty_Vector;
   end Map;

   -------------
   -- Path_To --
   -------------

   function Path_To (Name : String) return String is
      use Gnatwrapper_Installation;
   begin
      return GNATbench_Root &
             "eclipse" & Directory_Separator &
             "plugins" & Directory_Separator &
             Windriver_Plugin_Folder & Directory_Separator
             & Name;
   end Path_To;

end Known_Build_Specs;
