with GNAT.Expect;      use GNAT.Expect;
with Ada.Text_IO;  use Ada.Text_IO;
with Gnat.OS_Lib;
with Gnatwrapper_Status_Codes;  use Gnatwrapper_Status_Codes;

package body Gnatwrapper_Utils is

   ------------
   -- Launch --
   ------------

   procedure Launch (Command_With_Path : in String_Access;
                     Args              : in String_List)
   is
      Make_Result : Integer;
   begin
      Make_Result := GNAT.OS_Lib.Spawn (Command_With_Path.all, Args);
      if Make_Result /= 0 then
         GNAT.OS_Lib.OS_Exit (Make_Result);
      end if;
   end Launch;

   ----------------------
   -- Get_GNAT_Version --
   ----------------------

   procedure Get_GNAT_Version
     (Gnatls_Name  : in  String_Access;
      GNAT_Version : out String_Access)
   is
      --  see procedure Compute_Predefined_Paths in package Projects.Registry
      --  (the body) located gps\kernel\src_info\projects-registry.adb

      Current : String_Access := new String'("");
      Args    : constant String_List_Access := new String_List (1..1);

      Result  : Expect_Match;
      Success : Boolean;
      Fd      : Process_Descriptor_Access;

   begin
      begin
         Success := True;

         declare
            Gnatls_Path : String_Access :=
               Locate_Exec_On_Path (Gnatls_Name.all);
         begin
            if Gnatls_Path = null then
               Success := False;
            else
               Args (1) := new String'("-v");
               Fd := new Process_Descriptor;
               Non_Blocking_Spawn
                 (Fd.all,
                  Gnatls_Path.all,
                  Args(1..1),
                  Buffer_Size => 0, Err_To_Out => True);
               Free (Gnatls_Path);
            end if;
         end;
      exception
         when others =>
            Success := False;
      end;

      if not Success then
         Put_Line
           (Standard_Error,
            "Could not invoke the gnatls command");
         GNAT.OS_Lib.OS_Exit (GNATLS_Error);
      end if;

      Expect (Fd.all, Result, "GNATLS .+(\n| )Copyright", Timeout => 5_000);

      declare
         S : constant String := Strip_CR (Expect_Out_Match (Fd.all));
      begin
         GNAT_Version := new String'(S (S'First + 11 .. S'First + 15));
      end;

   exception
      when Process_Died =>
         Free (Current);
         Close (Fd.all);
         Put_Line (Standard_Error, "Could not get GNAT version");
         GNAT.OS_Lib.OS_Exit (GNAT_Version_Failure);
   end Get_GNAT_Version;

   ----------------------------------
   -- Correct_Executable_File_Name --
   ----------------------------------

   procedure Correct_Executable_File_Name
      (File_Name         : in out String_Access;
       Executable_Suffix : in     String_Access;
       Spec              : in     Build_Spec)
   is
      --  This routine corrects the file name extension of the executable
      --  from what the project facility reports into what is actually used
      --  by the compiler.  This is necessary because the project facility
      --  indicates the host's executable extension unless the optional
      --  attribute Builder'Executable_Suffix is specified in the user's
      --  project file.

      --  If the attribute *is* specified we need do nothing. Otherwise, the
      --  approach is to strip off the host-oriented extension, if any, and
      --  then append one that depends on the build spec specified as a
      --  parameter to gnatwrapper. We have to be careful to correctly handle
      --  names that include dots that do not precede extensions, and we must
      --  also handle the case in which the host name does not have an
      --  extension.

      --  the index of the right-most '.' in File_Name, if any
      Dot_Pos          : Natural := 0;

      --  These are the known host executable file name extensions to use when
      --  looking for extensions in a candidate executable name.
      Known_Extensions : constant String_List :=
                           (new String'(""),
                            new String'(".exe"));

   begin
      --  if the suffix is defined, the project facility will return the
      --  right file name so there is nothing to correct.
      if Executable_Suffix /= null then
         return;
      end if;

      --  Locate the right-most '.' in the file name, which may or may not be
      --  part of an extension.
      for K in reverse File_Name.all'Range loop
         if File_Name (K) = '.' then
            Dot_Pos := K;
            exit;
         end if;
      end loop;

      if Dot_Pos /= 0 then
         --  Found a possible extension. We test it to see if it is a real,
         --  recognized extension and not just a part of the name itself. If it
         --  is a known extension we remove it because it came from the project
         --  facility.

         declare
            Proposed_Extension : constant String :=
              File_Name (Dot_Pos .. File_Name'Last);
            Basename           : constant String :=
              File_Name (File_Name'First .. Dot_Pos - 1);
         begin
            for K in Known_Extensions'Range loop
               if Proposed_Extension = Known_Extensions (K).all then
                  --  We have found a known extension at the end of the name.
                  --  Remove it.

                  File_Name := new String'(Basename);
                  exit;
               end if;
            end loop;
         end;
      end if;

      --  At this point, File_Name.all has no extension. It may have a dot
      --  inside the name, but not for an extension.  Now we can append the
      --  proper extension based on the build spec.
      File_Name := new String'(File_Name.all & Spec.Executable_File_Extension);
   end Correct_Executable_File_Name;

   --------------
   -- Strip_CR --
   --------------

   function Strip_CR (Text : String) return String is
      To       : String (1 .. Text'Length);
      Index_To : Positive := 1;
   begin
      for Index in Text'Range loop
         if Text (Index) /= ASCII.CR then
            To (Index_To) := Text (Index);
            Index_To := Index_To + 1;
         end if;
      end loop;

      return To (1 .. Index_To - 1);
   end Strip_CR;

end Gnatwrapper_Utils;
