with Ada.Strings.Fixed;
with GNAT.Expect;      use GNAT.Expect;
with Ada.Text_IO;  use Ada.Text_IO;
with Gnat.OS_Lib;
with Gnatwrapper_Status_Codes;  use Gnatwrapper_Status_Codes;
with Gnatwrapper_Utils;  use Gnatwrapper_Utils;

package body GNATwrapper_Installation is

   procedure Get_GNATbench_Root (Location : out String_Access);

   procedure Get_Windriver_Plugin_Folder (Location : out String_Access);

   GNATbench_Root_Located    : Boolean := False;
   GNATbench_Root_Location   : String_Access;

   Windriver_Plugin_Located  : Boolean := False;
   Windriver_Plugin_Location : String_Access;


   --------------------
   -- GNATbench_Root --
   --------------------

   function GNATbench_Root return String is
   begin
      if not GNATbench_Root_Located then
         Get_GNATbench_Root (GNATbench_Root_Location);
         GNATbench_Root_Located := True;
      end if;
      return GNATbench_Root_Location.all;
   end GNATbench_Root;

   -----------------------------
   -- Windriver_Plugin_Folder --
   -----------------------------

   function Windriver_Plugin_Folder return String is
   begin
      if not Windriver_Plugin_Located then
         Get_Windriver_Plugin_Folder (Windriver_Plugin_Location);
         Windriver_Plugin_Located := True;
      end if;
      return Windriver_Plugin_Location.all;
   end Windriver_Plugin_Folder;

   ------------------------
   -- Get_GNATbench_Root --
   ------------------------

   procedure Get_GNATbench_Root (Location : out String_Access) is
      GNATwrapper_Path : String_Access;
      Pos : Integer;
      use Ada.Strings.Fixed;
   begin
      GNATwrapper_Path := Locate_Exec_On_Path ("gnatwrapper");
      Pos := Index (Source  => GNATwrapper_Path.all,
                    Pattern => "bin");
      Location := new String'(GNATwrapper_Path (1 .. Pos - 1));

      if Directory_Separator = '\' then
         for K in Location'Range loop
            if Location (K) = '/' then
               Location (K) := '\';
            end if;
         end loop;
      end if;
   end Get_GNATbench_Root;

   ---------------------------------
   -- Get_Windriver_Plugin_Folder --
   ---------------------------------

   procedure Get_Windriver_Plugin_Folder (Location : out String_Access) is
      Current : String_Access := new String'("");
      Args    : String_List_Access;
      Result  : Expect_Match;
      Success : Boolean;
      Fd      : Process_Descriptor_Access;
      Command : String_Access;
   begin
      if Directory_Separator = '\' then
         --  on windows

         Command := new String'("cmd.exe");
         Args := new String_List (1 .. 5);
         Args (1) := new String'("/c");
         Args (2) := new String'("dir");
         Args (3) := new String'("/A:D");
         Args (4) := new String'("/B");
         Args (5) := new String'(GNATbench_Root & "eclipse\plugins");
      else
         Command := new String'("ls");
         Args := new String_List (1 .. 2);
         Args (1) := new String'("-1");
         Args (2) := new String'(GNATbench_Root & "eclipse/plugins");
      end if;

      begin
         Success := True;
         Fd := new Process_Descriptor;
         Non_Blocking_Spawn
           (Fd.all,
            Command.all,
            Args (1 .. Args'Last),
            Buffer_Size => 0, Err_To_Out => True);
      exception
         when others =>
            Success := False;
      end;

      if not Success then
         Put_Line
           (Standard_Error,
            "Could not invoke the directory list command '" &
            Command.all & "'");
         GNAT.OS_Lib.OS_Exit (GNATLS_Error);
      end if;

      Expect (Fd.all,
              Result,
              "com.adacore.gnatbench.windriver_([0-9]|\.)+(.)?",
              Timeout => 5_000);

      Location := new String'(Strip_CR (Expect_Out_Match (Fd.all)));
   exception
      when Process_Died =>
         Free (Current);
         Close (Fd.all);
         Put_Line (Standard_Error, "Could not get target plugin folder");
         GNAT.OS_Lib.OS_Exit (GNAT_Version_Failure);
   end Get_Windriver_Plugin_Folder;


end GNATwrapper_Installation;
