with Ada.Command_Line; use Ada.Command_Line;
with Ada.Text_IO;      use Ada.Text_IO;
with GNAT.OS_Lib;      use GNAT.OS_Lib;
with Ada.Exceptions;   use Ada.Exceptions;

with Ada.Containers.Vectors;

with Gnatwrapper_Status_Codes; use Gnatwrapper_Status_Codes;
with Gnatwrapper_Utils;        use Gnatwrapper_Utils;
with Build_Specs;              use Build_Specs;
with GNAT_Project;
with Ragged_Strings;
with Scenario_Variables;

procedure GNATWrapper is

   Version               : constant String := "2.3.0-20090317";

   Project               : GNAT_Project.File;

   Target                : Build_Spec;

   --  the Workbench destination when copying the GNAT-produced executable file
   Output_File           : String_Access;

   --  the name of the unit used as the main procedure
   Selected_Main_Unit    : String_Access;

   --  the name of the project file passed to gnatwrapper
   Project_File_Name     : String_Access;

   --  project file names are last in the args so this is also a
   --  flag used to catch extra args
   Project_File_Name_Set : Boolean := False;

   --  the specifc build spec passed to gnatwrapper, eg "SIMNTgnu"
   Build_Spec_Name       : String_Access;

   --  are we cleaning instead of building?
   Cleaning_Project      : Boolean := False;

   --  the version of GNAT current installed, so we can check against the value
   --  of First_GNAT_Version_with_eS_Switch
   GNAT_Version          : String_Access;

   --  GNAT versions prior to 6.0 did not recognize -eS switch that writes
   --  errors to Standard_Error
   First_GNAT_Version_with_eS_Switch : constant Character := '6';

   --  The args that are required for gnatmake, such as "-P gpr-file"
   Common_Args   : Ragged_Strings.Vector;

   --  The optional -margs args passed directly to gnatmake
   Optional_Args : Ragged_Strings.Vector;

   --  All the scenario variable settings in the .gb_project file
   User_Scenario_Settings : constant Scenario_Variables.Settings.Vector :=
      Scenario_Variables.From_File;

   --  All the scenario variable settings in the .gb_project file represented
   --  as a list of strings in the form -Xname=value
   Scenario_Var_Args : constant Ragged_Strings.Vector :=
      Scenario_Variables.As_Switches (User_Scenario_Settings);

   --  Convert a Vector to a String_List for sake of Launch procedure
   function As_String_List (This : Ragged_Strings.Vector)
      return String_List;

   procedure Note_Makefile_Error (Msg : in String);

   procedure Note_Installation_Error (Tool_Name : in String);

   procedure Parse_Switches;

   function Invalid_Arg (Arg_Num : Natural) return Boolean;

   --------------------
   -- Parse_Switches --
   --------------------

   procedure Parse_Switches is
      Current_Arg : Positive := 1;
   begin
      loop
         exit when Current_Arg > Argument_Count;
         if Argument (Current_Arg) = "-m" then
            Current_Arg := Current_Arg + 1;
            if Invalid_Arg (Current_Arg) then
               Note_Makefile_Error ("no main unit specified with -m switch");
               GNAT.OS_Lib.OS_Exit (Switch_Error);
            end if;
            Selected_Main_Unit := new String'(Argument (Current_Arg));

         elsif Argument (Current_Arg) = "-s" then
            Current_Arg := Current_Arg + 1;
            if Invalid_Arg (Current_Arg) then
               Note_Makefile_Error ("no build spec specified with -s switch");
               GNAT.OS_Lib.OS_Exit (Switch_Error);
            end if;

            Build_Spec_Name := new String'(Argument (Current_Arg));

         elsif Argument (Current_Arg) = "-o" then
            Current_Arg := Current_Arg + 1;
            if Invalid_Arg (Current_Arg) then
               Note_Makefile_Error ("no output file specified with -o switch");
               GNAT.OS_Lib.OS_Exit (Switch_Error);
            end if;
            Output_File := new String'(Argument (Current_Arg));

         elsif Argument (Current_Arg) = "-margs" then
            while Current_Arg <= Argument_Count loop
               Optional_Args.Append (new String'(Argument(Current_Arg)));
               Current_Arg := Current_Arg + 1;
            end loop;

         elsif Argument (Current_Arg) = "-v" then
            Ada.Text_IO.Put_Line (Version);
            GNAT.OS_Lib.OS_Exit (GNATwrapper_Status_Codes.Success);

         elsif Argument (Current_Arg) = "-c" then
            Cleaning_Project := True;

         else
            if not Project_File_Name_Set then
               Project_File_Name := new String'(Argument (Current_Arg));
               Project_File_Name_Set := True;
            else
               Note_Makefile_Error ("invalid switch " & Argument (Current_Arg));
            end if;
         end if;
         Current_Arg := Current_Arg + 1;
      end loop;
   end Parse_Switches;

   --------------------
   -- As_String_List --
   --------------------

   function As_String_List (This : Ragged_Strings.Vector) return String_List is
      Result : String_List (1 .. Integer (This.Length));

      use Ragged_Strings;

      C : Cursor := This.First;
   begin
      for K in 1 .. Result'Length loop
         Result (K) := Element (C);
         Next (C);
      end loop;
      return Result;
   end As_String_List;

   -------------------------
   -- Note_Makefile_Error --
   -------------------------

   procedure Note_Makefile_Error (Msg : in String) is
   begin
      Put_Line (Standard_Error, "Error in makefile: " & Msg & '.');
      Put_Line (Standard_Error,
                "Project makefile fragment 'ada.makefile' " &
                "has been corrupted.");
   end Note_Makefile_Error;

   -----------------------------
   -- Note_Installation_Error --
   -----------------------------

   procedure Note_Installation_Error (Tool_Name : in String) is
   begin
      Put_Line (Standard_Error,
         "The GNAT Pro toolchain is not in your path (cannot find the " &
         Tool_Name & " command).");
      Put_Line (Standard_Error,
         "Possible installation error or corruption.");
   end Note_Installation_Error;

   -----------------
   -- Invalid_Arg --
   -----------------

   function Invalid_Arg (Arg_Num : Natural) return Boolean is
   begin
      return Arg_Num > Argument_Count or else Argument (Arg_Num)(1) = '-';
   end Invalid_Arg;

   --------------
   -- Do_Build --
   --------------

   procedure Do_Build is
      --  the gnatmake tool name with the full path included
      Gnatmake_Command  : String_Access;

      --  the value of the Executable attribute in the project file that
      --  specifies the file name of the executable produced by the compiler
      Executable_Name   : String_Access;

      Executable_Suffix : String_Access;
      Successful_Copy   : Boolean;

      use type Ragged_Strings.Vector;
   begin
      Gnatmake_Command := Locate_Exec_On_Path (Target.Gnatmake_Name.all);
      if Gnatmake_Command = null then
         Note_Installation_Error (Target.Gnatmake_Name.all);
         GNAT.OS_Lib.OS_Exit (No_Toolchain);
      end if;

      Get_GNAT_Version (Target.Gnatls_Name, GNAT_Version);

      if GNAT_Version (GNAT_Version'First) >= First_GNAT_Version_with_eS_Switch then
         --  The arg "-eS" causes gnatmake to write to stdout instead of
         --  stderr, so that Workbench does not display normal output as errors.
         Common_Args.Append (new String'("-eS"));
      end if;

      Common_Args.Append (new String'("-P"));
      Common_Args.Append (new String'(Project.Name.all));

      Project.Get_Exe (Executable_Name, Selected_Main_Unit);

      --  Get the suffix, if specified, to use when correcting the name
      Project.Get_Exe_Suffix (Executable_Suffix);

      Common_Args.Append (Selected_Main_Unit);

      --  run the gnatmake command, halting make with error code if fails
      Launch (Gnatmake_Command,
              As_String_List (Common_Args & Optional_Args & Scenario_Var_Args));

      --  Now that GNAT has produced the executable we can correct its file
      --  name. We need to correct the extension in the name returned by the
      --  project facility because the project facility is, by default,
      --  appending the extension of the host rather than what the cross
      --  compiler actually used. If Executable_Suffix *is* defined in the
      --  project file the project facility will use it, but that attribute is
      --  optional.

      Correct_Executable_File_Name (Executable_Name, Executable_Suffix, Target);

      --  Copy the executable to the location expected by Workbench
      Put_Line ("copying " & Executable_Name.all & " to " & Output_File.all);
      Copy_File
        (Executable_Name.all,
         Output_File.all,
         Successful_Copy,
         Overwrite);

      if not Successful_Copy then
         Put_Line
           (Standard_Error,
            "Copying Ada object to Workbench object directory failed!");
         GNAT.OS_Lib.OS_Exit (Copy_Failure);
      end if;
   end Do_Build;

   --------------
   -- Do_Clean --
   --------------

   procedure Do_Clean is
      --  the gnatclean tool name with the full path included
      Gnatclean_Command : String_Access;

      use type Ragged_Strings.Vector;
   begin
      Gnatclean_Command := Locate_Exec_On_Path (Target.Gnatclean_Name.all);
      if Gnatclean_Command = null then
         Note_Installation_Error (Target.Gnatclean_Name.all);
         GNAT.OS_Lib.OS_Exit (No_Toolchain);
      end if;

      Common_Args.Append (new String'("-P"));
      Common_Args.Append (new String'(Project.Name.all));

      --  run the gnatclean command, halting make with error code if fails
      Launch (Gnatclean_Command, As_String_List (Common_Args & Scenario_Var_Args));
   end Do_Clean;


begin
   Parse_Switches;
   --  Note that the program may exit (normally or in error) based on the
   --  switches parsed. For example, the switch "-v" will cause the version
   --  number to be printed and then the program exits.

   if Project_File_Name = null then
      Note_Makefile_Error ("no project file specified");
      GNAT.OS_Lib.OS_Exit (Switch_Error);
   end if;

   if Build_Spec_Name = null then
      Note_Makefile_Error ("no build spec specified with -s switch");
      GNAT.OS_Lib.OS_Exit (Switch_Error);
   end if;

   Target.Initialize (Build_Spec_Name.all);

   Project.Initialize (Project_File_Name.all);
   Project.Apply (User_Scenario_Settings);

   if Cleaning_Project then
      Do_Clean;
   else
      if Output_File = null then
         Note_Makefile_Error ("no output file specified with -o switch");
         GNAT.OS_Lib.OS_Exit (Switch_Error);
      end if;

      Do_Build;
   end if;
exception
   when E : others =>
      Put_Line
        (Standard_Error,
         "Unexpected exception in GNATwrapper! Please report the following:");
      Put_Line (Standard_Error, Exception_Information(E));
      GNAT.OS_Lib.OS_Exit (Unknown_Error);
      --  we still use OS_Exit so the makefile will stop executing
end GNATWrapper;
