-----------------------------------------------------------------------
--                             GNATbench                             --
--                                                                   --
--                 Copyright (C) 2006-2009, AdaCore                  --
--                                                                   --
-----------------------------------------------------------------------

with Prj.Ext;

with Gnatwrapper_Status_Codes; use Gnatwrapper_Status_Codes;
with Ada.Text_IO;              use Ada.Text_IO;
with GNATCOLL.VFS;
with GNATCOLL.Filesystem; use GNATCOLL.Filesystem;

package body GNAT_Project is

   use Projects;

   ----------------
   -- Initialize --
   ----------------

   procedure Initialize (This : in out File; GPR_File_Name : String) is
      Actually_Loaded : Boolean;
      Successful_Load : Boolean;
   begin
      This.File_Name := new String'(GPR_File_Name);
      This.Registry.Load (Root_Project_Path  =>
                            GNATCOLL.VFS.Create
                              (Filesystem_String (GPR_File_Name)),
                          Errors             => Put_Line'Access,
                          New_Project_Loaded => Actually_Loaded,
                          Status             => Successful_Load);
      if not Successful_Load then
         Put_Line
           (Standard_Error, "Failed to load project file " & GPR_File_Name);
         GNAT.OS_Lib.OS_Exit (Malformed_Project_File);
      end if;
      Projects.Registry.Recompute_View (This.Registry, Put_Line'Access);
   end Initialize;

   -----------
   -- Apply --
   -----------

   procedure Apply
      (This  : in out File;
       These :        Scenario_Variables.Settings.Vector)
   is
      use Scenario_Variables.Settings;

      C        : Cursor;
      Scenario : Scenario_Variables.Setting;
   begin
      C := These.First;
      while Has_Element (C) loop
         Scenario := Element (C);

         --  we are using the lower level Prj interface here, rather than the
         --  higher level Project.Scenario_Variable abstraction, because we
         --  have a low-level view of scenario variables here: just a tuple
         --  containing strings for the external name and the value.  Should be
         --  fine bcause this is exactly what the higher level call does.

         Prj.Ext.Add (External_Name => Scenario.Name.all,
                      Value         => Scenario.Value.all);

         Next (C);
      end loop;
      --  now we recompute the view so the above will take effect
      Projects.Registry.Recompute_View (This.Registry, Put_Line'Access);
   end Apply;

   ----------
   -- Name --
   ----------

   function Name (This : File) return String_Access is
   begin
      return This.File_Name;
   end Name;

   -------------
   -- Get_Exe --
   -------------

   procedure Get_Exe
      (This                 : in out File;
       Executable_File_Name :    out String_Access;
       Selected_Main_Name   : in out String_Access)

   --  If the gnatwrapper invocation specifies the main unit (via the -m
   --  switch) Selected_Main_Name will not be null so we verify that it is a
   --  main specified in the project. Otherwise there must be only one main
   --  unit specified by the project, and if so, Selected_Main_Name will be set
   --  to that one. The value of Executable_File_Name is set to the name,
   --  including full path, for the executable generated for the given main
   --  unit.

   is
      Root  : constant Project_Type := Get_Root_Project (This.Registry);
      Mains : constant Argument_List :=
        Get_Attribute_Value (Root, Main_Attribute);
   begin
      if Mains'Length = 0 then
         Put_Line
           (Standard_Error,
            "Error in project properties: no main units specified.");
         Put_Line (Standard_Error,
                   "Use 'Edit GPR Properties' to specify one, or edit " &
                   "the GNAT project file and use the 'Main' attribute.");
         GNAT.OS_Lib.OS_Exit (No_Specific_Main);
      end if;

      if Selected_Main_Name /= null then
         if not Is_Main_File (Root, Selected_Main_Name.all) then
            Put_Line
              (Standard_Error,
               "Error in makefile: this project defines no main unit named " &
               Selected_Main_Name.all);
            GNAT.OS_Lib.OS_Exit (No_Main_Unit);
         end if;
      else
         if Mains'Length /= 1 then
            Put
              (Standard_Error,
               "Error in project properties: multiple main units: ");
            for M in Mains'Range loop
               Put (Standard_Error, Mains (M).all);
               if M /= Mains'Last then
                  Put (Standard_Error, ", ");
               else
                  New_Line (Standard_Error);
               end if;
            end loop;
            Put_Line
              (Standard_Error,
               "Use 'Edit GPR Properties' to specify just one, or edit " &
               "the GNAT project file and use the 'Main' attribute.");
            GNAT.OS_Lib.OS_Exit (No_Specific_Main);
         end if;

         Selected_Main_Name := Mains (Mains'First);
      end if;

      --  set Executable_File_Name using Selected_Main_Name, which is
      --  definitely not null now (if it ever was).  But first we must remove
      --  the source file name extension.

      declare
         Base_Pos : constant Natural := Delete_File_Suffix
           (Filesystem_String (Selected_Main_Name.all), Root);
         Basename : constant String := Selected_Main_Name.all
           (Selected_Main_Name'First .. Base_Pos);
      begin
         if Is_Absolute_Path (Selected_Main_Name.all) then
            Executable_File_Name := new String'(Basename);
         else
            --  prepend the path
            Executable_File_Name :=
              new String'
                (String (Executables_Directory (Root)) & "/" & Basename);
         end if;
      end;
   end Get_Exe;

   --------------------
   -- Get_Exe_Suffix --
   --------------------

   procedure Get_Exe_Suffix
      (This   : in out File;
       Suffix : out String_Access)
   is
      Setting : constant String := Get_Attribute_Value
         (Project   => Get_Root_Project (This.Registry),
          Attribute => Build (Builder_Package, "Executable_Suffix"),
          Default   => "");
   begin
      if Setting = "" then
         Suffix := null;
      else
         Suffix := new String'(Setting);
      end if;
   end Get_Exe_Suffix;

begin
   Projects.Registry.Initialize;
end GNAT_Project;
