with Ada.Text_IO;       use Ada.Text_IO;
with Ada.Strings.Fixed; use Ada.Strings.Fixed;
with GNAT.Regpat;       use GNAT.Regpat;
with GNAT.OS_Lib;
with Gnatwrapper_Status_Codes;
with Known_Build_Specs;  use Known_Build_Specs;

package body Build_Specs is

   --  Determine the names of the compiler etc. based on the build spec name.
   --  This is done via a mapping of regular expressions for build spec names to
   --  their corresponding tool name prefixes.

   procedure Determine_Commands (This : in out Build_Spec);

   ----------------
   -- Initialize --
   ----------------

   procedure Initialize (This: in out Build_Spec;  Name : in String) is
   begin
      This.Name := new String'(Name);
      This.Is_RTP := Tail (Name, 4) = "_RTP";
      Determine_Commands (This);
   end Initialize;

   ------------------------
   -- Determine_Commands --
   ------------------------

   procedure Determine_Commands (This : in out Build_Spec) is
      Known_Specs : constant Mapping.Vector := Known_Build_Specs.Map;
      C           : Mapping.Cursor;
   begin
      C := Known_Specs.First;
      while Mapping.Has_Element (C) loop
         declare
            Tool    : Descriptor      := Mapping.Element (C);
            Matcher : Pattern_Matcher := Compile (Tool.Pattern.all);
         begin
            if Match (Matcher, This.Name.all) then

               This.Gnatmake  := new String'(Tool.Prefix.all & "-gnatmake");
               This.Gnatls    := new String'(Tool.Prefix.all & "-gnatls");
               This.Gnatclean := new String'(Tool.Prefix.all & "-gnatclean");

               return;
            end if;
         end;
         Mapping.Next (C);
      end loop;

      --  we don't have a match for the build spec so we exit the program with
      --  an error message

      Put_Line
        (Standard_Error,
         "Error in makefile: unrecognized build spec '" &
         This.Name.all & "'");

      GNAT.OS_Lib.OS_Exit (Gnatwrapper_Status_Codes.Switch_Error);
   end Determine_Commands;

   ----------
   -- Name --
   ----------

   function Name (This : Build_Spec) return String_Access is
   begin
      return This.Name;
   end Name;

   ------------
   -- Is_RTP --
   ------------

   function Is_RTP (This : Build_Spec) return Boolean is
   begin
      return This.Is_RTP;
   end Is_RTP;

   -------------------
   -- Gnatmake_Name --
   -------------------

   function Gnatmake_Name (This : Build_Spec) return String_Access is
   begin
      return This.Gnatmake;
   end Gnatmake_Name;

   --------------------
   -- Gnatclean_Name --
   --------------------

   function Gnatclean_Name (This : Build_Spec) return String_Access is
   begin
      return This.Gnatclean;
   end Gnatclean_Name;

   -----------------
   -- Gnatls_Name --
   -----------------

   function Gnatls_Name (This : Build_Spec) return String_Access is
   begin
      return This.Gnatls;
   end Gnatls_Name;

   -------------------------------
   -- Executable_File_Extension --
   -------------------------------

   function Executable_File_Extension (This : Build_Spec) return String is
   begin
      if This.Is_RTP then
         return ".vxe";
      else
         --  DKM
         return ".out";
      end if;
   end Executable_File_Extension;

end Build_Specs;

