/*******************************************************************************
 * Copyright (C) 2006-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *******************************************************************************/

package com.adacore.gnatbench.windriver.internal.wizards.newProject;

import java.io.ByteArrayInputStream;
import java.lang.reflect.InvocationTargetException;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.ui.INewWizard;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.actions.WorkspaceModifyOperation;

import com.adacore.gnatbench.core.internal.GNATbenchCorePlugin;
import com.adacore.gnatbench.core.internal.GNATbenchProjectProperties;
import com.adacore.gnatbench.core.internal.builder.GNATbenchProjectNature;
import com.adacore.gnatbench.windriver.internal.SupportedTargets;
import com.windriver.ide.util.IBuildSpecAccessor;

abstract public class AbstractNewAdaProject extends Wizard implements INewWizard {

	protected AdaMainPage AdaMain;
	protected IProject newProject;


	public void addPages() {
		AdaMain = new AdaMainPage("Ada Main Suprogram Settings");
		addPage(AdaMain);
	} // addPages


	class NewProjectListener implements IResourceChangeListener {
		private IProject newProject = null;

		public void resourceChanged(IResourceChangeEvent event) {
			IResourceDelta root = event.getDelta();
			IResourceDelta[] projectDeltas = root.getAffectedChildren();
			for (int i = 0; i < projectDeltas.length; i++) {
				IResourceDelta delta = projectDeltas[i];
				IResource resource = delta.getResource();
				if (delta.getKind() == IResourceDelta.ADDED) {
					newProject = (IProject)resource;
				} // if
			} // for
		} // resourceChanged

		public IProject getNewProject() {
			return newProject;
		} // getNewProject

	} // NewProjectListener


	abstract protected IProject getNewProject();


	public boolean performFinish() {
		if (newProject == null) {
			// we need to run the wizard, either because this is the first time
			// or because it was cancelled by the user in an earlier invocation

			// run the WRS "new downloadable module project" wizard
			newProject = getNewProject();

			if (newProject == null) {
				// the user cancelled out of the WRS wizard
				return false;
			} // if

			IAdaptable a = (IAdaptable) newProject;
			IBuildSpecAccessor buildSpec = (IBuildSpecAccessor) a.getAdapter(IBuildSpecAccessor.class);
			String defaultBuildSpec = buildSpec.getDefaultBuildSpec();
			if (!SupportedTargets.includes(defaultBuildSpec)) {
				displayInfo("Unsupported Target",
						    "Warning: " + defaultBuildSpec + " is not recognized as a supported AdaCore target.\n" +
						    "Before you build, you may need to change the active build spec.");
			} // if

		} // if

		WorkspaceModifyOperation op = new WorkspaceModifyOperation() {
			protected void execute(IProgressMonitor monitor)
					throws CoreException {

				try {
					GNATbenchProjectNature.addNature(newProject);
				} catch (CoreException e) {
					displayError("Create New Ada Project", "Adding nature failed");
					GNATbenchCorePlugin.getDefault().logError(null, e);
					return;
				} // try

				if (!setGPRproperties()) {
					return;
				} // if

				if (!createGPRfile(newProject.getName(), AdaMain.unitName())) {
					return;
				} // if

				if (!setupProject(newProject, AdaMain.unitName())) {
					return;
				} // if

				// now we create the ada main program file with the specified
				// unit name unless they don't want us to generate one, in
				// which case they will uncheck the box

				if (AdaMain.createFile()) {
					if (!createMainFile(AdaMain.unitName())) {
						return;
					} // if
				} // if
			}
		};

		// run the project mod operation
		try {
			getContainer().run(false, true, op);
		} catch (InterruptedException e) {
			return false;
		} catch (InvocationTargetException e) {
			displayError("Create New Ada Project", "Project creation failed");
			GNATbenchCorePlugin.getDefault().logError(null, e);
			return false;
		} // try

		return true;

	} // performFinish


	protected boolean setGPRproperties() {

		GNATbenchProjectProperties props = GNATbenchProjectProperties
				.getPropertiesFor(newProject);

		props.setRootProject(true);

		String projPath = newProject.getName() + ".gpr";
		// the gpr file has the same name as the project itself
		props.setGPRPath(projPath);

		props.saveProjectFile();

		return true;
	} // setGPRproperties


	protected boolean setupProject(final IProject project, final String mainUnitName) {
		IFolder folder = newProject.getFolder(new Path ("debug_objs"));

		try {
			folder.create(true, true, null);
		} catch (CoreException e) {
			displayError("Create GNAT Project",
					"Creating object direcory for project " + project.getName() + " failed");
			GNATbenchCorePlugin.getDefault().logError(null, e);
			return false;
		} // try

		folder = newProject.getFolder(new Path ("production_objs"));
		try {
			folder.create(true, true, null);
		} catch (CoreException e) {
			displayError("Create GNAT Project",
					"Creating object direcory for project " + project.getName() + " failed");
			GNATbenchCorePlugin.getDefault().logError(null, e);
			return false;
		} // try

		return true;
	} // setupProject


	abstract protected boolean createGPRfile(String projectName, String mainUnitName);


	protected boolean createMainFile(final String mainUnitName) {
		String mainFileName = new String (mainUnitName + ".adb");
		IFile file = newProject.getFile(new Path (mainFileName));

		if (!file.exists()) {
			String newMainContent =
				"procedure " + mainUnitName + " is\n" +
				"begin\n" +
			    "   null; -- remove this null statement and enter your code here\n" +
				"end " + mainUnitName + ";";

			try {
				file.create(
						new ByteArrayInputStream(newMainContent.getBytes()),
						true, null);
			} catch (CoreException e) {
				displayError("Create GNAT project",
						"Adding main program file '" + mainFileName +
						"' to project " + newProject.getName() + " failed");
				GNATbenchCorePlugin.getDefault().logError(null, e);
				return false;
			} // try
		} // if
		return true;
	} // createMainfile


	public void init(IWorkbench workbench, IStructuredSelection selection) {
	} // init


	protected void displayInfo(String title, String msg) {
		MessageDialog.openInformation(getShell(), title, msg);
	} // displayInfo


	protected void displayError(String title, String msg) {
		MessageDialog.openError(getShell(), title, msg);
	} // displayError

} // NewAdaKernelModuleProject
