/*****************************************************************************
 * Copyright (C) 2007-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *****************************************************************************/

package com.adacore.gnatbench.ui.internal.wizards;

import java.io.ByteArrayInputStream;
import java.lang.reflect.InvocationTargetException;
import java.util.Iterator;
import java.util.LinkedList;

import org.eclipse.core.filesystem.IFileStore;
import org.eclipse.core.filesystem.URIUtil;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.IWorkspaceRunnable;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.operation.IRunnableContext;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.ui.progress.UIJob;

import com.adacore.gnatbench.core.GNATbenchCoreException;
import com.adacore.gnatbench.core.GNATbenchSession;
import com.adacore.gnatbench.core.internal.GNATbenchCorePlugin;
import com.adacore.gnatbench.core.internal.GNATbenchProjectProperties;
import com.adacore.gnatbench.core.internal.GPRResourceSpy;
import com.adacore.gnatbench.core.internal.builder.GNATbenchIncrementalBuilder;
import com.adacore.gnatbench.core.internal.builder.GNATbenchProjectNature;
import com.adacore.gnatbench.core.internal.gpswrappers.GPRProject;
import com.adacore.gnatbench.core.internal.make.IMakefile;
import com.adacore.gnatbench.core.internal.projects.GNATProjectRegistry;
import com.adacore.gnatbench.core.projects.IGNATProjectView;
import com.adacore.gnatbench.library.LibraryMonitor;
import com.adacore.gnatbench.library.LibrarySemaphore;
import com.adacore.gnatbench.library.Output.Output_Proc;
import com.adacore.gnatbench.library.Projects.Editor.Editor_Package;
import com.adacore.gnatbench.library.GNATCOLL.Filesystem.Filesystem_String;
import com.adacore.gnatbench.library.GNATCOLL.VFS.VFS_Package;
import com.adacore.gnatbench.ui.internal.utils.DisplayUtils;
import com.adacore.gnatbench.library.Standard.AdaString;

public class GNATImportOperation {

	private static final int NOT_SET = -1;
	private static final int YES = 0;
	private static final int NO = 1;
	private static final int ALWAYS = 2;
	private static final int NEVER = 3;
	private int fOverwritePolicy = NOT_SET;

	private GPRResourcesManager fResourceManager = null;

	boolean fIsDone = false;

	private GNATProjectRegistry fRegistry;

	/**
	 * Start an import operation. If the import didn't work, then the registry
	 * is returned to that analysis on the cause of the error can be made.
	 *
	 * @param doLinkFiles When true, the import is done by link - otherwise
	 * it's done by copy
	 * @param existingProjectName The name of the project where the root
	 * project has to be imported. If null, a project will be created.
	 */
	public GNATProjectRegistry start(final boolean doLinkFiles,
			final String existingProjectName,
			IRunnableContext container,
			final String rootGPRPath,
			final IProject rootProject,
			final boolean isWindRiverVersion,
			boolean fork) {

		final IWorkspaceRunnable operation = new IWorkspaceRunnable () {
			public void run(IProgressMonitor monitor)
					throws CoreException {
				LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

				try {
					GPRResourceSpy.setActivated(false);

					IProject project;

					fRegistry = (GNATProjectRegistry) GNATbenchSession
							.getDefault()
							.getProjectRegistryFactory()
							.createAnonymousRegistry(
									GNATbenchCorePlugin.getDefault()
											.getEFSRegistry().getUniqueStore(
													URIUtil.toURI(rootGPRPath)));

					if (!fRegistry.isValid()) {
						GNATbenchCorePlugin.getDefault().logError(
								"Cannot import project with errors", null);
						fIsDone = true;
						return;
					}

					GPRProject gprProject = fRegistry.getRootProject();

					fResourceManager = new GPRResourcesManager(gprProject,
							monitor);

					if (rootProject == null) {
						IFileStore rootFileStore = GNATbenchCorePlugin
								.getDefault().getEFSRegistry().getUniqueStore(
										URIUtil.toURI(rootGPRPath));

						project = createImportedProject(gprProject,
								existingProjectName, rootFileStore,
								doLinkFiles, fResourceManager,
								new SubProgressMonitor(monitor, 0));
					} else {
						project = rootProject;
					}

					importProject(project, GNATbenchCorePlugin.getDefault()
							.getEFSRegistry().getUniqueStore(
									URIUtil.toURI(rootGPRPath)), doLinkFiles,
							isWindRiverVersion, monitor);

					fRegistry.unload();
					fRegistry = null;
					GPRResourceSpy.setActivated(true);

				} catch (GNATbenchCoreException e) {
					GNATbenchCorePlugin.getDefault().logError(null, e);
				} finally {
					LibrarySemaphore.stopGPSWork(libMonitor);

					fIsDone = true;
				}
			}

		};

		try {
			container.run(fork, true, new IRunnableWithProgress () {
				public void run(IProgressMonitor monitor)
						throws InvocationTargetException, InterruptedException {
					try {
						ResourcesPlugin.getWorkspace().run(operation,
								ResourcesPlugin.getWorkspace().getRoot(),
								IWorkspace.AVOID_UPDATE, monitor);
					} catch (CoreException e) {
						GNATbenchCorePlugin.getDefault().logError(null, e);

						fIsDone = true;
					}
				}});
		} catch (Throwable e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
			DisplayUtils.displayError("GNAT Project Import Wizard",
					"Project importing did not complete successfully");

			fIsDone = true;
		}

		return fRegistry;
	}

	/**
	 * Returns the list of project to import for a given project defined by its
	 * path. During this process, necessary eclipse projects are created. If
	 * eclipseProjectName is not null, then it will be taken as the created
	 * eclipse project, otherwise a name base on the GPR name will be created.
	 *
	 * @param gprProjectPath
	 * @param eclipseProjectName
	 * @param doLinkFile
	 * @param monitor
	 * @return A list of ProjectToImport
	 * @throws CoreException
	 */
	private LinkedList<ProjectToImport> getListOfProjectsToImport(
			IFileStore gprProjectStore, IProject project, boolean doLinkFile,
			IProgressMonitor monitor) throws CoreException,
			GNATbenchCoreException {

		LinkedList <ProjectToImport> result = new LinkedList <ProjectToImport>();

		monitor.beginTask("import project " + gprProjectStore.getName(), 0);

		GPRProject gprProject = fRegistry.getProjectFromURI(gprProjectStore
				.toURI());

		result.add(new ProjectToImport(project, gprProject, fResourceManager,
				null, doLinkFile));

		// Get dependencies and import them when needed

		LinkedList <IGNATProjectView> deps = gprProject.getDependencies(true);

		// The sub monitor will handle this project and sub projects

		for (IGNATProjectView dep : deps) {
			IPath depPath = URIUtil.toPath(dep.getProjectFile().toURI());

			if (dep.getProjectFile().equals(gprProjectStore)) {
				continue;
			}

			IProject dependencie = GPRProject
					.getEclipseProjectForGPRPath(depPath.toOSString());

			 if (dependencie == null) {
				GPRProject depGprProject = fRegistry.getProjectFromLocation(depPath);

				if (isGNATLibraryProject(depGprProject)) {
					// this is a project in gnat/lib, such as win32ada, so
					// we skip it.
					continue;
				}

				GPRResourcesManager depResourceManager =
					new GPRResourcesManager(depGprProject, monitor);

				IProject depProject = createImportedProject(depGprProject,
						null, dep.getProjectFile(), doLinkFile,
						depResourceManager,
						new SubProgressMonitor(monitor, 0));

				result.add(new ProjectToImport(depProject, depGprProject,
						depResourceManager, project, doLinkFile));
			}
		}

		return result;
	}

	/**
	 * Create the links from the project to Eclipse.
	 *
	 * @param eclipseProjectName
	 *            is taken to import the project into an existing Eclipse
	 *            project, otherwise it will guess a name an create a new
	 *            project.
	 * @param gprProject
	 * @param project
	 * @throws GPSError
	 * @throws CoreException
	 */
	private void importProject(IProject rootProject, IFileStore gprProjectStore,
			boolean doLinkFile, boolean isWindRiverVersion,
			IProgressMonitor monitor)
			throws CoreException, GNATbenchCoreException {

		final int MONITOR_UNITS = 1000;

		IPath gprProjectPath = URIUtil.toPath(gprProjectStore.toURI());

		LinkedList<ProjectToImport> listOfProjects = getListOfProjectsToImport(
				gprProjectStore, rootProject, doLinkFile,
				new NullProgressMonitor());

		for (ProjectToImport element : listOfProjects) {
			element.setup(monitor);
		}

		monitor.beginTask("import project " + gprProjectStore.getName(),
				listOfProjects
				.size() * MONITOR_UNITS);

		for (ProjectToImport projectToImport : listOfProjects) {
			importFilesFromProject(projectToImport, doLinkFile,
					new SubProgressMonitor(monitor, MONITOR_UNITS));

			GNATbenchProjectProperties props = GNATbenchProjectProperties
			.getPropertiesFor(projectToImport.getEclipseProject());

			props.setRootProject(false);
			props.saveProjectFile();
		}

		if (!doLinkFile) {
			//  End of the copy processing: we first fix project names

			for (ProjectToImport projectToImport : listOfProjects) {
				projectToImport.moveGPRAndRename(projectToImport.fEclipseProject
						.getName());
			}

			//	Then we save projects.

			for (ProjectToImport projectToImport : listOfProjects) {
				IFile projectFile = projectToImport.getEclipseProject()
						.getFile(
								projectToImport.fEclipseProject.getName()
										.toLowerCase()
										+ ".gpr");

				projectToImport.fGPRProject.save(projectFile);

				projectToImport.fEclipseProject.refreshLocal(
						IProject.DEPTH_ONE, monitor);
			}
		}

		GNATbenchProjectProperties props = GNATbenchProjectProperties
				.getPropertiesFor(GPRProject
						.getEclipseProjectForGPRName(GPRProject
								.extractNameFromPath(gprProjectPath
										.toOSString())
								+ ".gpr"));

		props.setRootProject(true);
		props.saveProjectFile();

		//  The project path may have changed, in case of a copy or if the
		//  linked path has been normalized. This call ensures that the unique
		//  association path <-> kernel is done with the latest version of the
		//  path as known by the properties, in order to avoid dangling
		//  kernels.

		//  XXX: Since we now use an anonymous registry, this should not be
		//  needed anymore.
		// fRegistry.changeProjectPath(props.getGPROSPath());

		if (!isWindRiverVersion) {
			createGNATProMakefile(GPRProject
					.getEclipseProjectForGPRName(GPRProject
							.extractNameFromPath(gprProjectPath.toOSString())
							+ ".gpr"), monitor);
		}

		// XXX: Same.
		// fRegistry.recompute();

		monitor.done();
	}

	/**
	 * Return a project able to receive the project given in parameter.
	 *
	 * @param gprProject
	 * @param eclipseProjectName
	 * @param gprProjectPath
	 * @param doLinkFiles
	 * @param monitor
	 * @return
	 * @throws GPSError
	 * @throws CoreException
	 */
	private IProject createImportedProject(GPRProject gprProject,
			String eclipseProjectName, IFileStore gprProjectStore,
			boolean doLinkFiles, GPRResourcesManager folderManager,
			IProgressMonitor monitor) throws CoreException,
			GNATbenchCoreException {
		IProject project;

		if (eclipseProjectName == null) {
			project = ResourcesPlugin.getWorkspace().getRoot()
			   .getProject(gprProject.getName());
		} else {
			project = ResourcesPlugin.getWorkspace().getRoot()
			   .getProject(eclipseProjectName);
		} // if

		return project;
	}

	/**
	 * Import the files from a gprProject into an Eclipse project
	 *
	 * @param gprPath
	 * @param gprProject
	 * @param project
	 * @param doLinkFile
	 * @param monitor
	 * @throws GPSError
	 * @throws CoreException
	 */
	private void importFilesFromProject(final ProjectToImport project,
			boolean doLinkFile, final IProgressMonitor monitor)
			throws CoreException, GNATbenchCoreException {

		final int ONE_DIRECTORY_ITEMS = 100;

		monitor.beginTask("Coping directories", project.getResourceManager()
				.getPathList().size()
				* ONE_DIRECTORY_ITEMS);

		if (doLinkFile) {

			for (PathAssociation element : project
					.getResourceManager().getPathList()) {

				if (monitor.isCanceled())
					break;

				try {
					String folderName;

					IPath localPath = element.getLocalPath();
					IPath fullOSPath = element.getOSPath();

					if (localPath.getDevice() != null) {
						folderName = localPath.toOSString().substring(
								localPath.getDevice().length());
					} else {
						folderName = localPath.toOSString();
					}

					if (folderName.length() > 0) {
						if (folderName.charAt(folderName.length() - 1) == '/'
								|| folderName.charAt(folderName
										.length() - 1) == '\\') {

							// Remove the last separator from the name,
							// we don't want to display it.
							folderName = folderName.substring(0,
									folderName.length() - 1);
						}
					}

					// Replace the folder separator because everything
					// must be inside the same folder.
					// TODO : This is no longer the case in 3.2, to be
					// updated at this time
					folderName = folderName.replaceAll("\\\\|/", " ")
					+ "/";

					if (folderName.equals("/")) {
						folderName = fullOSPath.removeFirstSegments(
								fullOSPath.segmentCount() - 1)
								.toString();
					}

					IFolder folder = project.getEclipseProject()
					.getFolder(folderName);

					if (!cancelBecauseExisting(folder, monitor)) {
						try {
							folder.createLink(fullOSPath,
									IFolder.ALLOW_MISSING_LOCAL, null);
						} catch (CoreException e) {
							//  This happen typically when the linked folder is
							//  a parent of the workspace. In these cases,
							//  we'll just ignore the link. This is most often
							//  due to an undefined directory in the project
							//  file anyway.
						}
					}

					monitor.worked(1);

				} catch (IllegalArgumentException e1) {
					// TODO: This error is raised when linking to a
					// directory  that is a subdirectory of another
					// linked directory. It is not clear wether
					// this is an Eclipse bug or a known limitation.
					GNATbenchCorePlugin.getDefault().logError(null, e1);
				}
			}

			IPath gprPath = new Path(project.getGPRProject()
					.getFile().getOSPath());
			IFile gprFile = project.getEclipseProject()
			.getFile(gprPath.lastSegment());

			if (!gprFile.exists()) {
				gprFile.createLink(gprPath, IResource.FORCE,
						monitor);
			}
		} else {
			final LinkedList<IFileStore> dirs = project.fGPRProject
					.getSourceDirectories(false);

			final PathList paths = new PathList();

			final IFolder objects = project.fEclipseProject.getFolder(paths
					.getUniqueName("obj"));
			final IFolder exec = project.fEclipseProject.getFolder(paths
					.getUniqueName("exec"));
			final LinkedList<IFileStore> sources = project.fGPRProject
					.getSourceFiles(false);
			final String [] srcDirs = new String [dirs.size()];

			if (!objects.exists ()) {
				objects.create(true, true, monitor);
			}

			if (!exec.exists ()) {
				exec.create(true, true, monitor);
			}

			int indDir = 0;

			for (IFileStore dir : dirs) {
				IFolder newFolder = project.fEclipseProject
				.getFolder(paths.getEclipsePath(URIUtil.toPath(
						dir.toURI()).toOSString()));

				if (!newFolder.exists()) {
					newFolder.create(true, true, monitor);
				}

				srcDirs[indDir] = project.fEclipseProject.getLocation().append(
						newFolder.getProjectRelativePath().lastSegment())
						.toOSString();

				indDir = indDir + 1;
			}

			for (IFileStore srcFile : sources) {
				IPath srcPath = URIUtil.toPath(srcFile.toURI ());

				IPath newPath = paths.getEclipsePath(
						srcPath.removeLastSegments(1).removeTrailingSeparator()
						.toOSString()).append(srcPath.lastSegment());

				IFile targetFile = project.fEclipseProject.getFile(newPath);

				if (!cancelBecauseExisting(targetFile, monitor)) {
					IFile tmp = project.fEclipseProject.getFile("tmp");
					tmp.createLink(srcPath, IFile.ALLOW_MISSING_LOCAL, monitor);
					tmp.copy(newPath, true, monitor);
					tmp.delete (true, monitor);
				}
			}

			project.fGPRProject.setAttributeValue("Source_Dirs", "", "",
					srcDirs);
			project.fGPRProject.setAttributeValue("Object_Dir", "", "",
					project.fEclipseProject.getLocation().append(
							objects.getProjectRelativePath()).toOSString());
			project.fGPRProject.setAttributeValue("Exec_Dir", "", "",
					project.fEclipseProject.getLocation().append(
							exec.getProjectRelativePath()).toOSString());
			project.moveGPRAndRename(project.fEclipseProject
					.getName() + "_tmp");

			final GNATbenchProjectProperties props = GNATbenchProjectProperties
					.getPropertiesFor(project.fEclipseProject);

			props.setGPRPath(project.fEclipseProject.getName().toLowerCase()
					+ ".gpr");

			props.saveProjectFile();
		}

		monitor.done();
	}

	/**
	 * This class is used to store an os path and the corresponding one in the
	 * eclipse worskpace.
	 */
	private class PathAssociation {
		private IPath fOsPath;
		private IPath fLocalPath;

		public PathAssociation (IPath osPath, IPath localPath) {
			fOsPath = osPath;
			fLocalPath = localPath;
		}

		public IPath getLocalPath () {
			return fLocalPath;
		}

		public IPath getOSPath () {
			return fOsPath;
		}
	}

	/**
	 * This class manages resource list from a project, and is able to give the
	 * information required to import this list into the workspace.
	 */
	private class GPRResourcesManager {

		private LinkedList <PathAssociation> fPathList = new LinkedList <PathAssociation> ();
		LinkedList <IGNATProjectView> fDependencies;

		/**
		 * This constructor will extract from the gprProject the list of
		 * resources that needs to be imported.
		 *
		 * @param gprProject
		 * @param monitor
		 * @throws CoreException
		 * @throws GPSError
		 */
		public GPRResourcesManager(GPRProject gprProject,
				IProgressMonitor monitor) throws GNATbenchCoreException, CoreException {
			LinkedList <IFileStore> directories;

			directories = gprProject.getSourceDirectories (false);
			directories.addAll(gprProject.getObjectDirectories(false));

			int minCommonFolders = -1;

			IPath basePath = null;

			if (directories.size() > 0 ) {
				basePath = URIUtil.toPath(directories.get(0).toURI());
			}

			for (IFileStore dir : directories) {
				IPath fullOSPath = URIUtil.toPath(dir.toURI());

				int nbCommonFolders = basePath
						.matchingFirstSegments(fullOSPath);

				if (nbCommonFolders < minCommonFolders
						|| minCommonFolders == -1) {
					minCommonFolders = nbCommonFolders;
				}
			}

			for (IFileStore dir : directories) {
				IPath fullPath = URIUtil.toPath(dir.toURI());
				IPath newPath = fullPath.removeFirstSegments(minCommonFolders);

				boolean addPath = true;
				LinkedList<PathAssociation> elementsToRemove = new LinkedList<PathAssociation>();

				for (Iterator <PathAssociation> iter2 = fPathList.iterator(); iter2.hasNext();) {
					PathAssociation path = (PathAssociation) iter2.next();

					int common = newPath.matchingFirstSegments(path
							.getLocalPath());

					if (common == path.getLocalPath().segmentCount()) {
						addPath = false;
						break;
					} else if (common == newPath.segmentCount()) {
						elementsToRemove.add(path);
					}
				}

				if (addPath) {
					fPathList.add(new PathAssociation(fullPath, newPath));
				}

				fPathList.removeAll(elementsToRemove);
			}

			fDependencies = gprProject.getDependencies(false);
		}

		/**
		 * Return the list of root resource's path needed to be imported.
		 *
		 * @return
		 */
		public LinkedList <PathAssociation> getPathList () {
			return fPathList;
		}

		/**
		 * For a given os path, return the eclipse path that would correspond
		 * in the project given in parameter.
		 *
		 * @param path
		 * @param project
		 * @return
		 */
		public String getPathForOsPath (String strPath, IProject project) {
			IPath path = new Path (strPath);

			String fileName = path.lastSegment();
			String folderPath = path.removeLastSegments(1).toString();

			if (true) {

			}

			for (PathAssociation element : fPathList) {
				if (element.getOSPath().toOSString().equals(folderPath)) {
					IPath localPath = element.getLocalPath().setDevice("");

					return project.getLocation().toOSString()
					+ localPath.toString()
					+ fileName;
				}
			}

			return path.lastSegment ();
		}

		public LinkedList <IGNATProjectView> getDependencies () {
			return fDependencies;
		}
	}

	/**
	 * This class holds all the informations concerning a project that has to
	 * be imported in the workspace.
	 */
	private class ProjectToImport {

		private IProject fEclipseProject;
		private IProject fParentProject;
		private boolean fDoLinkFiles;
		private GPRProject fGPRProject;
		private GPRResourcesManager fFolderManager;

		/**
		 * When this constructor is called, we assumed that the eclipse project
		 * given in parameter is already created.
		 *
		 * @param eclipseProject
		 * @param gprProject
		 * @param folderManager
		 * @throws GPSError
		 */
		public ProjectToImport(IProject eclipseProject, GPRProject gprProject,
				GPRResourcesManager folderManager, IProject parentProject,
				boolean doLinkFiles) {

			fEclipseProject = eclipseProject;
			fParentProject = parentProject;
			fDoLinkFiles = doLinkFiles;
			fGPRProject = gprProject;
			fFolderManager = folderManager;
		}

		/**
		 * Return the eclipse project associated to this instance.
		 *
		 * @return
		 */
		public IProject getEclipseProject () {
			return fEclipseProject;
		}

		/**
		 * Return the GPR project associated to this instance.
		 *
		 * @return
		 */
		public GPRProject getGPRProject () {
			return fGPRProject;
		}

		/**
		 * Return the folder manager associated to this instance.
		 *
		 * @return
		 */
		public GPRResourcesManager getResourceManager () {
			return fFolderManager;
		}

		/**
		 * Will move the GPR to the current eclipse directory, and rename it
		 * according to the name given in parameter.
		 *
		 * @param newPath
		 */
		public void moveGPRAndRename(String newName)
				throws GNATbenchCoreException {
			IPath gprPath = fEclipseProject.getLocation();

			Editor_Package.Rename_And_Move(fRegistry.getRootProject()
					.getGPRProject(), fGPRProject.getGPRProject(),
					new Filesystem_String(newName),
					VFS_Package.Create(new Filesystem_String(gprPath
							.toOSString())), new Output_Proc() {

				public void Output_Proc_Body(AdaString S) {
					GNATbenchCorePlugin.getDefault().logError(
									"Error when Importing: " + S.toString(),
									null);

				}
			});
		}

		/**
		 * Opens and setup the eclipse project.
		 * @throws CoreException
		 */
		public void setup(IProgressMonitor monitor) throws CoreException,
				GNATbenchCoreException {
			if (!fEclipseProject.exists()) {
				fEclipseProject.create(new SubProgressMonitor (monitor, 0));
			}

			fEclipseProject.open(new SubProgressMonitor (monitor, 0));

			setupImportedProject(fEclipseProject, fDoLinkFiles, fGPRProject
					.getGPRProject().Project_Path(new AdaString("")).Full_Name(
							false).toString(), fFolderManager);

			if (fParentProject != null) {
				IProjectDescription description = fParentProject
						.getDescription();
				IProject[] references = description.getReferencedProjects();
				boolean found = false;

				for (int j = 0; j < references.length; ++j) {
					if (references[j].equals(fEclipseProject)) {
						found = true;
					}
				}

				if (!found) {
					IProject[] tmpReferences = new IProject[references.length + 1];
					System.arraycopy(references, 0, tmpReferences, 0,
							references.length);
					tmpReferences[tmpReferences.length - 1] = fEclipseProject;
					description.setReferencedProjects(tmpReferences);
					fParentProject.setDescription(description,
							new NullProgressMonitor());
				}
			}
		}
	}

	private class PathList {
		private LinkedList <PathAssociation> fPathList = new LinkedList <PathAssociation> ();
		private LinkedList <String> fNameList = new LinkedList <String> ();

		public IPath getEclipsePath (String osPath) {
			IPath path = new Path (osPath).removeTrailingSeparator();

			for (PathAssociation element : fPathList) {
				if (element.fOsPath.equals(path)) {
					return element.getLocalPath();
				}
			}

			String newName = getUniqueName(path.lastSegment());
			IPath newPath = new Path(newName);

			fPathList.add(new PathAssociation(path, newPath));

			return newPath;
		}

		/**
		 *  Return name if name is not already on the list - otherwise add
		 *  some discriminant suffix.
		 */
		public String getUniqueName (String name) {
			String newName = name;
			int duplicateIndex = 1;

			for (String oldName : fNameList) {
				if (oldName.equals(newName)) {
					newName = name + " (" + duplicateIndex + ")";
				}
			}

			fNameList.add(newName);

			return newName;
		}
	}

	// generate the makefile for the gnat project we are importing.  this is not
	// called in the WRS version of GNATbench.
	private void createGNATProMakefile(final IProject project,
			IProgressMonitor monitor) throws GNATbenchCoreException {

		final IFile file = project.getFile(new Path ("/" + IMakefile.MAKEFILE_NAME));

		final String makeFile;

		makeFile = GNATbenchIncrementalBuilder.makefileForGPR();

		try {
			if (!cancelBecauseExisting(file, monitor)) {
				file.create(new ByteArrayInputStream(makeFile.getBytes()),
						true, null);
			}
		} catch (CoreException e) {
			DisplayUtils.displayError("Import GNAT Project", "Makefile creation failed");
			GNATbenchCorePlugin.getDefault().logError(null, e);
		} // try

	} // createGNATProMakefile


	private void setupImportedProject(
		final IProject project,
		final boolean doLinkFile,
		final String gprProjectPath,
		final GPRResourcesManager resourceManager) throws CoreException
	{
		final GNATbenchProjectProperties props = GNATbenchProjectProperties
				.getPropertiesFor(project);

		if (doLinkFile) {
			props.setGPRPath(gprProjectPath);
		} else {
			props.setGPRPath
			(resourceManager.getPathForOsPath(gprProjectPath, project));
		}

		props.saveProjectFile();

		GNATbenchProjectNature.addNature(project);

	} // setupImportedProject

	/**
	 * This function returns true if the resource creation has to be cancelled
	 * because of of an already existing resource. If the resource exists and
	 * the user doesn't want to cancel the creation, the resource will be
	 * deleted.
	 */
	private boolean cancelBecauseExisting(final IResource resource,
			final IProgressMonitor monitor) {
		if (resource.exists()) {
			if (fOverwritePolicy == NOT_SET) {
				String message;

				if (resource instanceof IFile) {
					message = "The file "
							+ ((IFile) resource).getFullPath().toOSString();
				} else if (resource instanceof IFolder) {
					message = "The folder "
						+ ((IFolder) resource).getFullPath().toOSString();
				} else {
					message = "The resource " + resource.getName();
				}

				message += " already exists on the file system. Overwrite?";

				final String finalMessage = message;

				UIJob job = new UIJob ("Quering file overwrite") {

					public IStatus runInUIThread(IProgressMonitor monitor) {
						MessageDialog dialog = new MessageDialog(
								null,
								resource.getName() + " exists.",
								null,
								finalMessage,
								MessageDialog.QUESTION, new String[] { "Yes",
										"No", "Always", "Never" }, 1);

						fOverwritePolicy = dialog.open();

						return GNATbenchCorePlugin.OK_STATUS;
					}};

				job.schedule();
				try {
					job.join();
				} catch (InterruptedException e1) {
					GNATbenchCorePlugin.getDefault().logError(null, e1);
				}

				switch (fOverwritePolicy) {
				case YES:
					fOverwritePolicy = NOT_SET;
				case ALWAYS:
					try {
						resource.delete(true, monitor);
					} catch (CoreException e) {
						GNATbenchCorePlugin.getDefault().logError(null, e);
					}
					return false;
				case NO:
					fOverwritePolicy = NOT_SET;
				case NEVER:
					return true;
				}
			} else if (fOverwritePolicy == ALWAYS) {
				try {
					resource.delete(true, monitor);
				} catch (CoreException e) {
					GNATbenchCorePlugin.getDefault().logError(null, e);
				}
				return false;
			} else if (fOverwritePolicy == NEVER) {
				return true;
			}
		}

		return false;
	}

	public boolean done () {
		return fIsDone;
	}

	protected boolean isGNATLibraryProject(final GPRProject imported) {
		final Path importedPath;
		try {
			importedPath = new Path(imported.getFile().getOSPath());
		} catch (GNATbenchCoreException e) {
			return false;
		}
		final String predefinedPath = fRegistry.getProjectRegistry()
				.Get_Predefined_Project_Path().toString();
		final String [] predefinedPathParts = predefinedPath
				.split(System.getProperty("path.separator"));
		for (String entry : predefinedPathParts) {
			if (new Path(entry).isPrefixOf(importedPath)) {
				return true;
			}
		}
		return false;
	}

}

