/*******************************************************************************
 * Copyright (C) 2005-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *******************************************************************************/

package com.adacore.gnatbench.ui.internal.wizards;

import java.util.LinkedList;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.Path;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.MouseListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IImportWizard;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.actions.NewProjectAction;

import com.adacore.gnatbench.core.GNATbenchCoreException;
import com.adacore.gnatbench.core.GNATbenchSession;
import com.adacore.gnatbench.core.internal.GNATbenchCorePlugin;
import com.adacore.gnatbench.core.internal.utils.PluginUtils;
import com.adacore.gnatbench.core.projects.IGNATProjectRegistry;
import com.adacore.gnatbench.core.projects.IGNATProjectMessage;

/**
 * This class is used to import a existing GNAT project.
 */
public class GNATImport extends Wizard implements IImportWizard {

	private Combo fProjectChoice;
	private Button fNewProject;
	private Button fExistingProject;
	private Button fLinkMode;
	private Button fCopyMode;
	private Button useNewProjectWizard;

	// we only want to present the option to config via the NewProject
	// wizard in the WRS version of GNATbench because these are already
	// GNAT projects we are importing.
	final private boolean isWindRiverVersion = PluginUtils.isWRSPluginPresent();

	// this flag is set based on the wizard page and will only be used
	// in the WRS version of GNATbench
	private boolean requestedNewProjectWizard = false;

	// the project returned from the NewProject wizard
	private IProject projectFromWizard;

	private String fInputGPRPath;

	private boolean fGNATProjectValid = false;

	private class MainPage extends WizardPage {

		/**
		 * @param pageName
		 */
		protected MainPage(String pageName) {
			super(pageName);
			setTitle(pageName);
			setDescription("Specify the root project file for the existing " +
					       "GNAT project, where to put this project's " +
					       "content, and how to insert it.");
			setPageComplete(false);
		}

		/*
		 * (non-Javadoc)
		 *
		 * @see org.eclipse.jface.dialogs.IDialogPage#createControl(org.eclipse.swt.widgets.Composite)
		 */
		public void createControl(Composite parent) {
			Composite baseComp = new Composite(parent, SWT.NONE);
			setControl(baseComp);

			//****************************************************************
			// SELECTION OF THE GPR FILE
			//****************************************************************

			GridLayout topLayout = new GridLayout();
			baseComp.setLayout(topLayout);

			Composite locationComp = new Composite(baseComp, SWT.NONE);
			GridLayout locationLayout = new GridLayout();
			locationLayout.numColumns = 2;
			locationLayout.marginHeight = 1;
			locationLayout.marginWidth = 1;
			locationComp.setLayout(locationLayout);
			GridData gd = new GridData(GridData.FILL_HORIZONTAL);
			locationComp.setLayoutData(gd);

			Label label = new Label(locationComp, SWT.NONE);
			label.setText("Location of the root project:");
			gd = new GridData();
			gd.horizontalSpan = 2;
			label.setLayoutData(gd);

			final Text projectNameWidget = new Text(locationComp, SWT.SINGLE
					| SWT.BORDER);

			projectNameWidget.addModifyListener(new ModifyListener () {

				public void modifyText(ModifyEvent e) {
					fGNATProjectValid = false;
					fInputGPRPath = ((Text) e.getSource()).getText().trim();

					try {
						IGNATProjectRegistry dummy = GNATbenchSession
								.getDefault().getProjectRegistryFactory()
								.createAnonymousRegistry(
										GNATbenchCorePlugin.getDefault()
												.getEFSRegistry().getUniqueStore(
														new Path(fInputGPRPath)));

						if (!dummy.isValid ()) {
							fGNATProjectValid = false;
							MainPage.this.setPageComplete(false);

							String error = "";

							LinkedList <IGNATProjectMessage> messages
							 = dummy.getMessages ();

							for (IGNATProjectMessage message : messages) {
								if (message.getKind() == IGNATProjectMessage.Kind.ERROR) {
									error = error + message.getMessage() + "\n";
								}
							}

							MainPage.this.setErrorMessage(error);
						} else {
							fGNATProjectValid = true;
							MainPage.this.setPageComplete(true);
							MainPage.this.setErrorMessage(null);
						}

						dummy.unload();
					} catch (GNATbenchCoreException e1) {
						GNATbenchCorePlugin.getDefault().logError(null, e1);
						MainPage.this.setErrorMessage("Unexpected exception: "
								+ e1.getMessage());
						MainPage.this.setPageComplete(false);
					}

//			        if (isCurrentPage()) {
//			            getContainer().updateButtons();
//			        } // if
				}});

			gd = new GridData(GridData.FILL_HORIZONTAL);
			projectNameWidget.setLayoutData(gd);

			Button button = new Button (locationComp, SWT.PUSH);
			button.setText("Browse");
			button.addMouseListener(new MouseListener() {

				public void mouseDoubleClick(MouseEvent e) {
				}

				public void mouseDown(MouseEvent e) {
				}

				public void mouseUp(MouseEvent e) {
					FileDialog dialog = new FileDialog(getShell());
					dialog.setFilterExtensions(new String[] { "*.gpr" });

					String path = dialog.open();

					if (path != null) {
						projectNameWidget.setText(path);
					}
				}
			});

			//****************************************************************
			// SELECTION OF THE OUTPUT PROJECT
			//****************************************************************

			Composite projectComp = new Composite(baseComp, SWT.NONE);
			GridLayout projectLayout = new GridLayout();
			projectLayout.numColumns = 2;
			projectLayout.marginHeight = 0;
			projectLayout.marginWidth = 0;
			projectComp.setLayout(projectLayout);
			projectComp.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

			fNewProject = new Button (projectComp, SWT.RADIO);
			fNewProject.setText("Import into a new project");
			fNewProject.setSelection(true);
			gd = new GridData();
			gd.horizontalSpan = 2;
			fNewProject.setLayoutData(gd);

			fNewProject.addSelectionListener(new SelectionListener (){

				public void widgetSelected(SelectionEvent e) {
					fProjectChoice.setEnabled(false);
					if (useNewProjectWizard != null) {
						useNewProjectWizard.setEnabled(true);
					}
				}

				public void widgetDefaultSelected(SelectionEvent e) {
				}});


			// whether to configure using the NewProject wizard.  Only makes sense to use
			// in the WRS version of GNATbench because we are importing GNAT projects
			// so they are already configured in that sense.
			if (isWindRiverVersion) {
				useNewProjectWizard = new Button (projectComp, SWT.CHECK);
				useNewProjectWizard.setText("Configure with the New Project wizard?");
				useNewProjectWizard.setSelection(false);
				gd = new GridData();
				gd.horizontalSpan = 2;
				gd.horizontalIndent = 20;
				useNewProjectWizard.setLayoutData(gd);
			} // if

			fExistingProject = new Button(projectComp, SWT.RADIO);
			fExistingProject.setText("Import into an existing project:");
			gd = new GridData();
			fExistingProject.setLayoutData(gd);

			fExistingProject.addSelectionListener(new SelectionListener () {

				public void widgetSelected(SelectionEvent e) {
					fProjectChoice.setEnabled(true);

					if (useNewProjectWizard != null) {
						useNewProjectWizard.setEnabled(false);
					}
				}

				public void widgetDefaultSelected(SelectionEvent e) {
				}});

			fProjectChoice = new Combo(projectComp, SWT.SINGLE);
			fProjectChoice.setEnabled(false);
			IProject [] projects  =
				ResourcesPlugin.getWorkspace().getRoot().getProjects();

			for (int i = 0; i < projects.length; ++i) {
				fProjectChoice.add(projects [i].getName());
			}

			gd = new GridData(GridData.FILL_HORIZONTAL);
			fProjectChoice.setLayoutData(gd);

			//****************************************************************
			// SELECTION OF THE IMPORT MODE (COPY OR LINK)
			//****************************************************************

			Composite copyComp = new Composite(baseComp, SWT.NONE);
			GridLayout copyLayout = new GridLayout();
			copyLayout.numColumns = 1;
			copyLayout.marginHeight = 0;
			copyLayout.marginWidth = 0;
			copyComp.setLayout(copyLayout);
			copyComp.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

			fLinkMode = new Button (copyComp, SWT.RADIO);
			fLinkMode.setText("Link project files into workspace");
			fLinkMode.setSelection(true);
			gd = new GridData();
			fLinkMode.setLayoutData(gd);

			fCopyMode = new Button (copyComp, SWT.RADIO);
			fCopyMode.setText("Copy project files into workspace");
			fCopyMode.setSelection(false);
			gd = new GridData();
			fCopyMode.setLayoutData(gd);
		}

	}

	public GNATImport() {
		setNeedsProgressMonitor(true);
	}


	// the predicate used by the dialog manager to enable or disable the Finish button.
	// this is used in concert with updateButtons(), called in the event handler, so
	// that the Finish button is not enabled until we specify the path to the project
	// to be imported.
	public boolean canFinish() {
		return fGNATProjectValid;
	}


	/*
	 * (non-Javadoc)
	 *
	 * @see org.eclipse.jface.wizard.Wizard#performFinish()
	 */
	public boolean performFinish() {
		final String existingProjectName;

		if (fNewProject.getSelection()) {
			existingProjectName = null;
		} else {
			existingProjectName = fProjectChoice.getText();
		}

		// handle the "config via NewProject wizard" option, which we only
		// offer in the WRS version of GNATbench because the projects we
		// are importing are already GNAT projects.
		if (isWindRiverVersion) {
			// the user had the option to config with the NewProject wizard.
			// now we see if they took the option, and note it for further
			// reference elsewhere.
			requestedNewProjectWizard = useNewProjectWizard.getSelection();
			if (requestedNewProjectWizard) {
				// they have indeed requested to configure the new project
				projectFromWizard = getNewProjectFromWizard();
				// note they might hit the cancel button so the result can be null.
				if (projectFromWizard == null) {
					// the user changed his mind so we need to give them another
					// chance to decide how to do the import
					return false;
				} // if
			} // if
		} // if

		GNATImportOperation op = new GNATImportOperation();
		op.start(fLinkMode.getSelection(), existingProjectName, getContainer(),
				fInputGPRPath, projectFromWizard, isWindRiverVersion, true);

		return true;
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see org.eclipse.ui.IWorkbenchWizard#init(org.eclipse.ui.IWorkbench,
	 *      org.eclipse.jface.viewers.IStructuredSelection)
	 */
	public void init(IWorkbench workbench, IStructuredSelection selection) {
		// TODO Auto-generated method stub

	}

	public void addPages() {
		addPage(new MainPage("GNAT Project Selection"));
	}

	// Invoke the NewProject wizard defined by the platform, which will include
	// all those wizards that implement the INewWizard interface and are thus
	// registered automatically.  Invocation is done by creating an instance of the
	// platform-defined action for the wizard and running it, with a listener checking
	// for the change to the workspace.  A bit convoluted but that's the only option.
	private IProject getNewProjectFromWizard() {
		NewProjectListener listener = new NewProjectListener();
		ResourcesPlugin.getWorkspace().addResourceChangeListener(listener, IResourceChangeEvent.POST_CHANGE);
		// now our listener is awaiting the creation of a new project so we can run the NewProject wizard.

		// create the NewProject wizard's action and run it to invoke the wizard
		(new NewProjectAction(PlatformUI.getWorkbench().getActiveWorkbenchWindow())).run();

		// the listener will have acquired the new project if the user did not click on cancel
		// so we can deactivate the listener now, although it still exists (so we can query it).
		ResourcesPlugin.getWorkspace().removeResourceChangeListener(listener);

		// get the (possibly null) result from the wizard via listener
		IProject project = listener.getNewProject();
		return project;
	} // getNewProjectFromWizard


	// a listener that checks for the result of invoking the NewProject wizard.
	// used only by getConfiguredNewProject()
	class NewProjectListener implements IResourceChangeListener {

		private IProject newProject = null;

		public void resourceChanged(IResourceChangeEvent event) {
			IResourceDelta root = event.getDelta();
			IResourceDelta[] projectDeltas = root.getAffectedChildren();
			for (int i = 0; i < projectDeltas.length; i++) {
				IResourceDelta delta = projectDeltas[i];
				IResource resource = delta.getResource();
				if (delta.getKind() == IResourceDelta.ADDED) {
					newProject = (IProject)resource;
				} // if
			} // for
		} // resourceChanged

		public IProject getNewProject() {
			return newProject;
		} // getNewProject

	} // NewProjectListener
} // GNATImport

