/*******************************************************************************
 * Copyright (C) 2007-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *******************************************************************************/

package com.adacore.gnatbench.ui.internal.quickfix;

import java.util.HashSet;
import java.util.LinkedList;
import java.util.TreeMap;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.jface.text.BadLocationException;

import com.adacore.gnatbench.core.internal.GNATbenchCorePlugin;
import com.adacore.gnatbench.core.internal.adaeditor.AdaEditorToolPreferencesProvider;
import com.adacore.gnatbench.core.internal.analyzer.GeneralizedLocation;
import com.adacore.gnatbench.core.internal.projects.GNATProjectRegistry;
import com.adacore.gnatbench.core.projects.IGNATProjectRegistry;
import com.adacore.gnatbench.core.projects.IGNATProjectRegistryAssistant;
import com.adacore.gnatbench.library.LibraryMonitor;
import com.adacore.gnatbench.library.LibrarySemaphore;
import com.adacore.gnatbench.library.Codefix.Codefix_Package;
import com.adacore.gnatbench.library.Codefix.Fix_Options;
import com.adacore.gnatbench.library.Codefix.Error_Lists.Error_Message_Iterator;
import com.adacore.gnatbench.library.Codefix.Error_Lists.Error_Message_List;
import com.adacore.gnatbench.library.Codefix.Errors_Parser.Fix_Processor;
import com.adacore.gnatbench.library.Codefix.Formal_Errors.Solution_List;
import com.adacore.gnatbench.library.Codefix.GNAT_Parser.GNAT_Parser_Package;
import com.adacore.gnatbench.library.GNATCOLL.Filesystem.Filesystem_String;
import com.adacore.gnatbench.library.GNATCOLL.VFS.VFS_Package;
import com.adacore.gnatbench.library.Standard.AdaString;

public class QuickFixKernelAssistant implements IGNATProjectRegistryAssistant {
	public static final String ASSISTANT_ID = "com.adacore.gnatbench.ui.quickfix.kernelAssistant";

	private Error_Message_List fCurrentErrorList;
	private Fix_Processor fFixProcessor;
	private QuickFixNavigator fQuickFixNavigator;
	private TreeMap<Long, InitialPosition> fInitialPositions = new TreeMap<Long, InitialPosition>();
	private TreeMap <Long, Solution_List> fFixes = new TreeMap <Long, Solution_List> ();
	private HashSet <IQuickFixListener> fListeners = new HashSet <IQuickFixListener> ();

	private static class InitialPosition {
		int fLine;
		int fColumn;
		String fFile;

		public InitialPosition (String file, int line, int column) {
			fLine = line;
			fColumn = column;
			fFile = file;
		}

		public boolean equals (Object obj) {
			if (obj instanceof InitialPosition) {
				InitialPosition pos = (InitialPosition) obj;

				return pos.fLine == fLine
					&& pos.fColumn == fColumn
					&& pos.fFile.equals(fFile);
			} else {
				return false;
			}
		}
	}

	public void initialize(IGNATProjectRegistry registry) {


		fCurrentErrorList = new Error_Message_List ();
		fCurrentErrorList.Initialize();
		fFixProcessor = new Fix_Processor ();
		fQuickFixNavigator = new QuickFixNavigator();
		fQuickFixNavigator.Set_Registry(((GNATProjectRegistry) registry)
				.getProjectRegistry());
		fQuickFixNavigator
				.Set_Construct_Database(((GNATProjectRegistry) registry)
						.getConstructDatabase());

		GNAT_Parser_Package.Register_Parsers(fFixProcessor);

		fFixProcessor.Initialize_Parsers();
	}

	/**
	 * Ad the solution to the marker list if needed.
	 *
	 * @param marker
	 */
	private void addSolutionForMarker (IMarker marker) {
		//  In order to avoid infinite recursions.
		fFixes.put(new Long(marker.getId()), null);

		GeneralizedLocation loc = GeneralizedLocation.fromTextMarker(marker);

		//  If we've already got a fix for that marker, don't try to
		//  recompute the same thing a second time.
		if (fFixes.get(marker.getId()) != null) {
			return;
		}

		LibraryMonitor monitor = LibrarySemaphore.startGPSWork();

		try {
			Error_Message_Iterator it = fCurrentErrorList.First_At_Location(loc
					.getFile().toVirtualFile(), loc.getLine(), loc.getColumn()
					.getColumnIndex());

			Solution_List sols = new Solution_List();

			while (!it.At_End()) {
				AdaString.Ref catRef = new AdaString.Ref();
				Fix_Options options = new Fix_Options();

				options.Remove_Policy(Codefix_Package.Remove_Entity());

				fFixProcessor.Get_Solutions(fQuickFixNavigator, it, options,
						catRef, sols);

				it = it.Next();
			}

			if (sols.Length() > 0) {
				fFixes.put(new Long(marker.getId()), sols);
			} else {
				fFixes.put(new Long(marker.getId()), null);
			}
		} finally {
			LibrarySemaphore.stopGPSWork(monitor);
		}
	}

	/**
	 * Return true if there is at least one marker at this location which is
	 * associated with a fix.
	 *
	 * @param loc
	 * @return
	 */
	public boolean isFixable (GeneralizedLocation loc) {
		IFile file = loc.getFile().getFile();

		if (file == null || !file.exists()) {
			return false;
		}

		IMarker[] markers;

		try {
			file.refreshLocal(IFile.DEPTH_INFINITE, new NullProgressMonitor ());
			markers = file.findMarkers(
					IMarker.PROBLEM, true,
					IResource.DEPTH_INFINITE);
		} catch (CoreException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);

			return false;
		}

		LibraryMonitor monitor = LibrarySemaphore.startGPSWork();

		try {
			for (IMarker marker : markers) {
				GeneralizedLocation markerLoc = GeneralizedLocation
						.fromTextMarker(marker);

				try {
					if (markerLoc.getOffset() == loc.getOffset()) {
						if (!fFixes.containsKey(marker.getId())) {
							addSolutionForMarker(marker);
						}

						Solution_List sols = fFixes.get(marker.getId());

						if (sols != null && sols.Length() > 0) {
							return true;
						}
					}
				} catch (BadLocationException e) {
					GNATbenchCorePlugin.getDefault().logError(null, e);
				}
			}
		} finally {
			LibrarySemaphore.stopGPSWork(monitor);
		}

		return false;
	}

	/**
	 * Return true if this marker, or an other marker at the exact same
	 * location, can be fixed.
	 *
	 * @param marker
	 * @return
	 */
	public boolean isFixable (IMarker marker) {
		IMarker[] markers;

		IFile file = (IFile) marker.getResource();

		try {
			file.refreshLocal(IFile.DEPTH_INFINITE,
					new NullProgressMonitor());
			markers = file.findMarkers(
					IMarker.PROBLEM, true,
					IResource.DEPTH_INFINITE);
		} catch (CoreException e) {
			return false;
		}

		LibraryMonitor monitor = LibrarySemaphore.startGPSWork();

		try {
			for (IMarker storedMarker : markers) {
				if (marker.getAttribute(IMarker.CHAR_START, 0) == storedMarker
						.getAttribute(IMarker.CHAR_START, 0)) {

					if (!fFixes.containsKey(storedMarker.getId())) {
						addSolutionForMarker(storedMarker);
					}

					Solution_List sols = fFixes.get(storedMarker.getId());

					if (sols != null && sols.Length() > 0) {
						return true;
					}
				}
			}
		} finally {
			LibrarySemaphore.stopGPSWork(monitor);
		}

		return false;
	}

	/**
	 * Return the possible solutions for the errors at the given location
	 * @param loc
	 * @return
	 */
	public Solution_List getSolutions (IMarker marker) {
				LibraryMonitor monitor = LibrarySemaphore.startGPSWork();

		try {
			if (!fFixes.containsKey(marker.getId())) {
				addSolutionForMarker(marker);
			}

			Solution_List sols = fFixes.get(marker.getId());

			if (sols != null && sols.Length() > 0) {
				return sols;
			}
		} finally {
			LibrarySemaphore.stopGPSWork(monitor);
		}

		return null;
	}

	/**
	 * Return the possible solutions for the errors at the given location
	 * @param loc
	 * @return
	 */
	public Solution_List getSolutions (GeneralizedLocation loc) {
		IFile file = loc.getFile().getFile();

		if (file == null || !file.exists()) {
			return null;
		}

		IMarker[] markers;

		try {
			file.refreshLocal(IFile.DEPTH_INFINITE, new NullProgressMonitor ());
			markers = file.findMarkers(
					IMarker.PROBLEM, true,
					IResource.DEPTH_ZERO);
		} catch (CoreException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);

			return null;
		}

		LibraryMonitor monitor = LibrarySemaphore.startGPSWork();

		try {
			for (IMarker marker : markers) {
				GeneralizedLocation markerLoc = GeneralizedLocation
						.fromTextMarker(marker);

				try {
					if (markerLoc.getOffset() == loc.getOffset()) {
						if (!fFixes.containsKey(marker.getId())) {
							addSolutionForMarker(marker);
						}

						Solution_List sols = fFixes.get(marker.getId());

						if (sols != null && sols.Length() > 0) {
							return sols;
						}
					}
				} catch (BadLocationException e) {
					GNATbenchCorePlugin.getDefault().logError(null, e);
				}
			}
		} finally {
			LibrarySemaphore.stopGPSWork(monitor);
		}

		return null;
	}


	/**
	 * Add this marker to the list of errors to be fixed.
	 *
	 * @param marker
	 */
	public void addError (IMarker marker) {
		if (!GNATbenchCorePlugin.getDefault().getPluginPreferences()
				.getBoolean(AdaEditorToolPreferencesProvider.PREF_TOOLS_CODEFIX)) {
			return;
		}

		String message = marker.getAttribute(IMarker.MESSAGE, "");

		GeneralizedLocation loc = GeneralizedLocation.fromTextMarker(marker);

		fCurrentErrorList.Add_Error(loc.getFile().toVirtualFile(),
				loc.getLine(), loc.getColumn().getColumnIndex(), new AdaString(
						message));

		fInitialPositions.put(marker.getId(), new InitialPosition(loc
				.getFile().getOSPath(), loc.getLine(), loc.getColumn()
				.getColumnIndex()));

		for (IQuickFixListener listener : fListeners) {
			listener.markerAnalyzed(this, marker);
		}
	}

	/**
	 * Remove this marker to the list of errors to be fixed.
	 *
	 * @param marker
	 */
	public void removeError (IMarker marker) {
		if (marker == null) {
			return;
		}

		if (fFixes.containsKey(marker.getId())) {
			fFixes.remove(marker.getId());
		}

		if (fInitialPositions.containsKey(marker.getId ())) {
			InitialPosition pos = fInitialPositions.get(marker.getId ());

			LibraryMonitor monitor = LibrarySemaphore.startGPSWork();

			try {
				fCurrentErrorList.Clear_Messages_At_Location(VFS_Package
						.Create(new Filesystem_String(pos.fFile)), pos.fLine,
						pos.fColumn);
			} finally {
				LibrarySemaphore.stopGPSWork(monitor);
			}

			fInitialPositions.remove(marker.getId());
		}
	}

	/**
	 * Return the list of marker we know of at the location given in parameter.
	 *
	 * @param loc
	 * @return
	 */
	public LinkedList <IMarker> getErrorsAtLocation (GeneralizedLocation loc) {
		LinkedList <IMarker> markerList = new LinkedList <IMarker> ();

		IFile file = loc.getFile().getFile();

		if (file == null || !file.exists()) {
			return markerList;
		}

		IMarker[] markers;
		try {
			markers = file.findMarkers(
					"org.eclipse.core.resources.problemmarker", true,
					IResource.DEPTH_INFINITE);
		} catch (CoreException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);

			return markerList;
		}

		for (IMarker marker : markers) {
			GeneralizedLocation markerLoc = GeneralizedLocation
					.fromTextMarker(marker);

			try {
				if (markerLoc.getOffset() == loc.getOffset()) {
					markerList.add(marker);
				}
			} catch (BadLocationException e) {
				GNATbenchCorePlugin.getDefault().logError(null, e);
			}
		}

		return markerList;
	}

	/**
	 * Return the text navigator associated to this assistant.
	 *
	 * @return
	 */
	public QuickFixNavigator getQuickFixNav () {
		return fQuickFixNavigator;
	}

	public void release(IGNATProjectRegistry registry) {
		// TODO Auto-generated method stub

	}

	public void addListener (IQuickFixListener listener) {
		fListeners.add(listener);
	}

	public void removeListener (IQuickFixListener listener) {
		fListeners.remove(listener);
	}

	public void clearErrors () {
		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try {
			fCurrentErrorList.Clear_Messages();
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}
	}
}
