/*******************************************************************************
 * Copyright (C) 2007-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *******************************************************************************/

package com.adacore.gnatbench.ui.internal.quickfix;

import org.eclipse.core.resources.IMarker;
import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.TextSelection;
import org.eclipse.ui.PartInitException;

import com.adacore.ajis.NativeException;
import com.adacore.gnatbench.core.internal.GNATbenchCorePlugin;
import com.adacore.gnatbench.core.internal.analyzer.Column;
import com.adacore.gnatbench.core.internal.analyzer.GeneralizedFile;
import com.adacore.gnatbench.core.internal.analyzer.GeneralizedLocation;
import com.adacore.gnatbench.core.internal.codingstyle.AdaIndentBuffer;
import com.adacore.gnatbench.library.Codefix.Text_Manager.File_Cursor;
import com.adacore.gnatbench.library.Codefix.Text_Manager.Mark_Abstr;
import com.adacore.gnatbench.library.Codefix.Text_Manager.Text_Cursor;
import com.adacore.gnatbench.library.Codefix.Text_Manager.Text_Interface;
import com.adacore.gnatbench.library.GNATCOLL.VFS.Virtual_File;
import com.adacore.gnatbench.ui.internal.GNATbenchUIPlugin;
import com.adacore.gnatbench.library.Standard.AdaString;
import com.adacore.gnatbench.library.Utils.Utils_Package;

public class QuickFixInterface extends Text_Interface {

	private GeneralizedFile fFile = null;

	private static class QuickFixMark extends Mark_Abstr {
		public IMarker marker;

		public GeneralizedLocation loc;
	}

	@Override
	public void Constrain_Update() {
	}

	@Override
	public void Delete_Line(Text_Cursor Cursor) {
		GeneralizedLocation loc = GeneralizedLocation.fromLineColumn(fFile,
				Cursor.Get_Line(), new Column(1,
						Column.DEFAULT_GNAT_TAB_WIDTH));

		IDocument document = fFile.getDocument();

		try {
			GNATbenchUIPlugin.getDefault().openEditor(loc);
		} catch (PartInitException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		}

		try {
			document.replace(loc.getOffset(), document.getLineLength(loc
					.getLine() - 1), "");
		} catch (BadLocationException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		}

		//  TODO: this should be handled at a higher level
		Text_Has_Changed();
	}

	@Override
	public AdaString Get(Text_Cursor Cursor, int Len) {
		try {
			GeneralizedLocation loc = GeneralizedLocation.fromLineColumn(fFile,
					Cursor.Get_Line(), new Column(Cursor.Get_Column(),
							Column.DEFAULT_GNAT_TAB_WIDTH));
			IDocument doc = fFile.getDocument();

			return new AdaString (doc.get(loc.getOffset(), Len));
		} catch (BadLocationException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
			return new AdaString("");
		}
	}

	@Override
	public char Get(Text_Cursor Cursor) {
		try {
			int lineNumber = Cursor.Get_Line();

			if (lineNumber == 0) {
				lineNumber = 1;
			}

			GeneralizedLocation loc = GeneralizedLocation.fromLineColumn(fFile,
					lineNumber, new Column(Cursor.Get_Column(),
							Column.DEFAULT_GNAT_TAB_WIDTH));
			IDocument doc = fFile.getDocument();

			return doc.getChar(loc.getOffset());
		} catch (BadLocationException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
			return 0;
		}
	}

	@Override
	public File_Cursor Get_Current_Cursor(Mark_Abstr Mark) {
		File_Cursor newCursor = new File_Cursor ();

		QuickFixMark newMark = (QuickFixMark) Mark;
		GeneralizedLocation loc;

		if (newMark.marker != null) {
			loc = GeneralizedLocation
			.fromTextMarker(newMark.marker);
		} else {
			loc = newMark.loc;
		}

		newCursor.Set_File(fFile.toVirtualFile());
		newCursor.Set_Line(loc.getLine());
		newCursor.Set_Column(loc.getColumn().getColumnIndex());

		return newCursor;
	}

	@Override
	public AdaString Get_Line(Text_Cursor Cursor, int Start_Col) {
		int lineNumber = Cursor.Get_Line();

		if (lineNumber == 0) {
			lineNumber = 1;
		}

		String line = fFile.getLine(lineNumber);

		int offset;

		if (Start_Col == 0) {
			offset = Column.toIndex(new Column(Cursor.Get_Column(),
					Column.DEFAULT_GNAT_TAB_WIDTH), line);

		} else {
			offset = Column.toIndex(new Column(Start_Col,
					Column.DEFAULT_GNAT_TAB_WIDTH), line);
		}

		String subLine = line.substring(offset - 1);

		//  TODO: The binding generator should offer a way to create directly
		//  a string with the appropriate constraints.
		return Utils_Package.Slide(new AdaString (subLine), offset);
	}

	@Override
	public Mark_Abstr Get_New_Mark(File_Cursor Cursor) {
		QuickFixMark newMark = new QuickFixMark ();

		GeneralizedLocation loc = GeneralizedLocation.fromLineColumn(fFile,
				Cursor.Get_Line(), new Column(Cursor.Get_Column(),
						Column.DEFAULT_GNAT_TAB_WIDTH));
		if (loc.getFile().getFile() == null) {
			//  In this case, we're probably adding a marker outside of the
			//  Eclipse resources, e.g. the runtime. So we keep the location,
			//  knowning that if the file is modified, the location will not
			//  be valid anymore.
			//  TODO: This heuristic should be improved.
			newMark.loc = loc;
		} else {
			newMark.marker = loc.toMarker(GNATbenchCorePlugin.CODEFIX_MARKER);

			if (newMark.marker == null) {
				throw new NativeException("can't add a marker on "
						+ fFile.getOSPath() + ":" + Cursor.Get_Line() + ":"
						+ Cursor.Get_Column());
			}
		}

		return newMark;
	}

	@Override
	public void Initialize(Virtual_File File_Name) {
		fFile = GeneralizedFile.fromVirtualFile(null, File_Name);
	}

	@Override
	public int Line_Max() {
		IDocument document = fFile.getDocument();

		return document.getNumberOfLines();
	}

	@Override
	public AdaString Read_File() {
		IDocument document = fFile.getDocument();

		AdaString result = new AdaString (document.get());

		result.setOwner(Owner.NATIVE);
		//  The string is supposed to be freed by the Ada side.

		return result;
	}

	@Override
	public void Replace(Text_Cursor Cursor, int Len, AdaString New_Value) {
		try {
			GeneralizedLocation loc = GeneralizedLocation.fromLineColumn(fFile,
					Cursor.Get_Line(), new Column(Cursor.Get_Column(),
							Column.DEFAULT_GNAT_TAB_WIDTH));

			try {
				GNATbenchUIPlugin.getDefault().openEditor(loc);
			} catch (PartInitException e) {
				GNATbenchCorePlugin.getDefault().logError(null, e);
			}

			IDocument document = fFile.getDocument();

			document.replace(loc.getOffset(), Len, New_Value.toString());
		} catch (BadLocationException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		}

		//  TODO: this should be handled at a higher level
		Text_Has_Changed();
	}

	@Override
	public void Undo() {
		// TODO Auto-generated method stub
	}

	@Override
	public void Indent_Line(Text_Cursor Cursor) {
		try {
			GNATbenchUIPlugin.getDefault().openEditor(fFile);
		} catch (PartInitException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		}

		AdaIndentBuffer indentBuffer = new AdaIndentBuffer(GNATbenchCorePlugin
				.getDefault().getDocumentBuffer(fFile));

		GeneralizedLocation loc = GeneralizedLocation
				.fromLineColumn(fFile, Cursor.Get_Line(), new Column(1,
						Column.DEFAULT_GNAT_TAB_WIDTH));

		IDocument document = fFile.getDocument();

		try {
			indentBuffer.format(new TextSelection(document, loc.getOffset(),
					loc.getFile().getLine(loc.getLine()).length()), true);
		} catch (BadLocationException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		}

		//  TODO: this should be handled at a higher level
		Text_Has_Changed();
	}

	@Override
	public void Add_Line(Text_Cursor Cursor, AdaString New_Line, boolean Indent) {
		GeneralizedLocation loc = GeneralizedLocation.fromLineColumn(fFile,
				Cursor.Get_Line() + 1, new Column(1,
						Column.DEFAULT_GNAT_TAB_WIDTH));

		try {
			GNATbenchUIPlugin.getDefault().openEditor(loc);
		} catch (PartInitException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		}

		IDocument document = fFile.getDocument();

		try {
			String lineDelimiter = document.getLineDelimiter(0);

			if (lineDelimiter == null) {
				lineDelimiter = document.getLegalLineDelimiters() [0];
			}

			document.replace(loc.getOffset(), 0, New_Line.toString() + lineDelimiter);
		} catch (BadLocationException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		}

		if (Indent) {
			Text_Cursor tmp = new Text_Cursor ();

			tmp.Set_Location(Cursor.Get_Line() + 1, 1);
			Indent_Line(tmp);
		}

		//  TODO: this should be handled at a higher level
		Text_Has_Changed();
	}

	@Override
	public void Replace(Text_Cursor Start_Cursor, Text_Cursor End_Cursor,
			AdaString New_Value) {
		try {
			GeneralizedLocation locStart = GeneralizedLocation.fromLineColumn(fFile,
					Start_Cursor.Get_Line(), new Column(Start_Cursor
							.Get_Column(), Column.DEFAULT_GNAT_TAB_WIDTH));

			GeneralizedLocation locEnd = GeneralizedLocation.fromLineColumn(fFile,
					End_Cursor.Get_Line(), new Column(End_Cursor
							.Get_Column(), Column.DEFAULT_GNAT_TAB_WIDTH));


			try {
				GNATbenchUIPlugin.getDefault().openEditor(locStart);
			} catch (PartInitException e) {
				GNATbenchCorePlugin.getDefault().logError(null, e);
			}

			IDocument document = fFile.getDocument();

			document.replace(locStart.getOffset(), locEnd.getOffset()
					- locStart.getOffset() + 1, New_Value.toString());
		} catch (BadLocationException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		}

		//  TODO: this should be handled at a higher level
		Text_Has_Changed();
	}

}
