/*******************************************************************************
 * Copyright (C) 2007-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *******************************************************************************/

package com.adacore.gnatbench.ui.internal.quickfix;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IMarker;
import org.eclipse.core.resources.IMarkerDelta;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IResourceDeltaVisitor;
import org.eclipse.core.runtime.CoreException;

import com.adacore.gnatbench.core.GNATbenchSession;
import com.adacore.gnatbench.core.internal.GNATbenchCorePlugin;
import com.adacore.gnatbench.core.internal.projects.GNATProjectRegistry;
import com.adacore.gnatbench.library.LibraryMonitor;
import com.adacore.gnatbench.library.LibrarySemaphore;

/**
 * This class reacs to markers change, and update the corresponding resources,
 * e.g. the quick fix database.
 */
public class ErrorMarkerSpy implements IResourceChangeListener, IResourceDeltaVisitor {

	public void resourceChanged(IResourceChangeEvent event) {
		switch (event.getType()) {
		case IResourceChangeEvent.POST_CHANGE:
			// New GNATbench projects are added to the list of known projects for GPS,
			// and Ada source file addition/deletion causes a recompute(). Project
			// deletion is handled by the PRE_DELETE case above.  Changes to GPR files
			// are also recognized and cause a recompute().

			LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

			try {
				event.getDelta().accept(this);
			} catch (Exception e) {
				GNATbenchCorePlugin.getDefault().logError(null, e);
			} finally {
				LibrarySemaphore.stopGPSWork(libMonitor);
			}

			break;


		} // switch
	} // resourceChanged


	public boolean visit(IResourceDelta delta) throws CoreException {
		int flags = delta.getFlags();

		GNATProjectRegistry registry = null;
		QuickFixKernelAssistant fixKernel = null;

		if ((flags & IResourceDelta.MARKERS) != 0) {
			IMarkerDelta [] markerDeltas = delta.getMarkerDeltas();

			LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

			try {
				for (IMarkerDelta markerDelta : markerDeltas) {
					IMarker marker = markerDelta.getMarker();

					if (markerDelta.getKind() == IResourceDelta.ADDED) {
						if (!(marker.getResource() instanceof IFile)) {

							continue;
						}

						if (!marker
								.isSubtypeOf(GNATbenchCorePlugin.ADA_ERROR_MARKER)) {

							continue;
						}

						if (registry == null) {
							registry = (GNATProjectRegistry) GNATbenchSession
									.getDefault().getOrLoadRegistry(
											delta.getResource().getProject());

							if (registry == null) {
								return false;
							}

							fixKernel = (QuickFixKernelAssistant) registry
									.getAssistant(QuickFixKernelAssistant.ASSISTANT_ID);
						}

						fixKernel.addError(marker);
					} else if (markerDelta.getKind() == IResourceDelta.REMOVED) {
						if (registry == null) {
							registry = (GNATProjectRegistry) GNATbenchSession
							.getDefault().getOrLoadRegistry(
									delta.getResource().getProject());

							if (registry == null) {
								return false;
							}

							fixKernel = (QuickFixKernelAssistant) registry
							.getAssistant(QuickFixKernelAssistant.ASSISTANT_ID);
						}

						fixKernel.removeError(marker);
					}
				}
			} finally {
				LibrarySemaphore.stopGPSWork(libMonitor);
			}

			return true;
		}

		return true;
	} // visit
} // GPRResourceSpy
