/*****************************************************************************
 * Copyright (C) 2008-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *****************************************************************************/

package com.adacore.gnatbench.ui.internal.projectexplorer;

import org.eclipse.core.filesystem.URIUtil;
import org.eclipse.core.resources.IFolder;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.ui.model.WorkbenchLabelProvider;

import com.adacore.gnatbench.core.GNATbenchCoreException;
import com.adacore.gnatbench.core.GNATbenchSession;
import com.adacore.gnatbench.core.internal.GNATbenchCorePlugin;
import com.adacore.gnatbench.core.internal.filesystem.FileStoreReference;
import com.adacore.gnatbench.core.internal.gpswrappers.GPRProject;
import com.adacore.gnatbench.core.internal.projects.GNATProjectRegistry;
import com.adacore.gnatbench.core.internal.projects.GNATProjectRegistryFactory;
import com.adacore.gnatbench.ui.internal.GNATbenchUIPlugin;
import com.adacore.gnatbench.ui.internal.projectexplorer.GPRExplorerContentProvider.ResourceAnnotation;

/**
 * Provide label decorations for contents generated by the project explorer.
 *
 * @since 2.2
 */
public class GPRExplorerLabelProvider extends WorkbenchLabelProvider {

	/**
	 * In case of an GNAT source folder or object folder, replace with the
	 * appropriate icon.
	 */
	protected ImageDescriptor decorateImage(ImageDescriptor input,
            Object element) {

		if (element instanceof IFolder) {
			IFolder folder = (IFolder) element;

			try {
				if (GNATProjectRegistry.isManagedSourceFolder(folder)) {
					return GNATbenchUIPlugin.getDefault().getImageDescriptor(
							GNATbenchUIPlugin.ICON_SOURCE_FOLDER);
				} else if (GNATProjectRegistry.isManagedBinaryFolder(folder)) {
					return GNATbenchUIPlugin.getDefault().getImageDescriptor(
							GNATbenchUIPlugin.ICON_BINARY_FOLDER);
				}

			} catch (GNATbenchCoreException e) {
				GNATbenchCorePlugin.getDefault().logError(null, e);
			}
		} else if (element instanceof FileStoreReference) {
			FileStoreReference ref = (FileStoreReference) element;

			if (ref.isProject()) {
				GPRProject view = ((GNATProjectRegistryFactory) GNATbenchSession
						.getDefault().getProjectRegistryFactory())
						.getGPRForPath(URIUtil.toPath(ref.getFileStore()
								.toURI()));

				if (view == null || view.getEclipseProject() == null) {
					return GNATbenchUIPlugin.getDefault().getImageDescriptor(
							GNATbenchUIPlugin.ICON_FOREIGN_PRJ);
				} else {
					return GNATbenchUIPlugin.getDefault().getImageDescriptor(
							GNATbenchUIPlugin.ICON_PROJECT);
				}
			} else if (ref.getFileStore().fetchInfo().isDirectory()) {
				ResourceAnnotation annot = (ResourceAnnotation) ref
						.getData(GPRExplorerContentProvider.resourceAnnotationKey);

				if (GNATProjectRegistry.isManagedSourceFolder(ref
						.getFileStore())
						&& (annot == null || annot == ResourceAnnotation.SourceFolder)) {
					return GNATbenchUIPlugin.getDefault().getImageDescriptor(
							GNATbenchUIPlugin.ICON_SOURCE_FOLDER);
				} else if (GNATProjectRegistry.isManagedBinaryFolder(ref
						.getFileStore())) {
					return GNATbenchUIPlugin.getDefault().getImageDescriptor(
							GNATbenchUIPlugin.ICON_BINARY_FOLDER);
				} else {
					return GNATbenchUIPlugin.getDefault().getImageDescriptor(
							GNATbenchUIPlugin.ICON_FOLDER);
				}
			} else if (GPRExplorerContentProvider.isAdaSource (ref.getFileStore())) {
				return GNATbenchUIPlugin.getDefault().getImageDescriptor(
						GNATbenchUIPlugin.ICON_ADA_FILE);
			}
		}

        return input;
    }

	protected String decorateText(String input, Object element) {
		if (element instanceof FileStoreReference) {
			FileStoreReference ref = (FileStoreReference) element;

			if (ref.isProject()) {
				return (String) GNATbenchCorePlugin.getDefault()
						.getEFSRegistry().getSessionProperty(
								ref.getFileStore(),
								AdaDependenciesNode.fgProjectName);
			} else {
				return ref.getFileStore().getName();
			}
		}

		return input;
	}

}
