/*****************************************************************************
 * Copyright (C) 2007-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *****************************************************************************/

package com.adacore.gnatbench.ui.internal.projectexplorer;

import java.util.LinkedList;

import org.eclipse.core.filesystem.IFileStore;
import org.eclipse.core.filesystem.URIUtil;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IResourceDeltaVisitor;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.IWorkspaceRunnable;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.window.DefaultToolTip;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.TreeItem;
import org.eclipse.ui.IEditorDescriptor;


import com.adacore.gnatbench.core.GNATbenchCoreException;
import com.adacore.gnatbench.core.GNATbenchSession;
import com.adacore.gnatbench.core.internal.GNATbenchCorePlugin;
import com.adacore.gnatbench.core.internal.adaeditor.AdaDocumentBuffer;
import com.adacore.gnatbench.core.internal.analyzer.AdaConstruct;
import com.adacore.gnatbench.core.internal.analyzer.AdaConstructCutFilter;
import com.adacore.gnatbench.core.internal.analyzer.AdaConstructFilter;
import com.adacore.gnatbench.core.internal.analyzer.GeneralizedFile;
import com.adacore.gnatbench.core.internal.analyzer.IAdaConstructFilterProvider;
import com.adacore.gnatbench.core.internal.filesystem.FileStoreReference;
import com.adacore.gnatbench.core.internal.filesystem.FileStoreRegistry;
import com.adacore.gnatbench.core.internal.gpswrappers.GPRProject;
import com.adacore.gnatbench.core.internal.projects.GNATProjectRegistry;
import com.adacore.gnatbench.core.internal.projects.GNATProjectRegistryFactory;
import com.adacore.gnatbench.core.projects.IGNATProjectRegistry;
import com.adacore.gnatbench.core.projects.IGNATProjectRegistryListener;
import com.adacore.gnatbench.library.Language.Language_Category;
import com.adacore.gnatbench.ui.internal.GNATbenchUIPlugin;

/**
 * Provide extended project explorer for GPR-based projects.
 *
 * @since 2.2
 */
public final class GPRExplorerContentProvider extends WorkbenchContentProvider
		implements IGNATProjectRegistryListener {

	public static enum ResourceAnnotation {BinaryFolder, SourceFolder};

	public static String resourceAnnotationKey =
		"com.adacore.gnatbench.ui.internal.projectexplorer.resourceAnnoationKey";

	private TreeViewer fViewer = null;

    private abstract static class ChildrenRunnable implements Runnable {
		public Object [] fResult;
	}

	public GPRExplorerContentProvider () {
	}

	/**
	 * Add tooltip support to the TreeViewer
	 * @author ceol
	 *
	 */
	private class GNATExplorerToolTip extends DefaultToolTip {

		protected GNATExplorerToolTip(TreeViewer viewer) {
			// we recreate the tooltip each time and manage it manually
			super(viewer.getTree(),DefaultToolTip.RECREATE, false);
			setShift(new Point(10,15));
		}

		/**
		 * hide the Tooltip when there is no info to display
		 */
		@Override
		protected boolean shouldCreateToolTip(Event event){
			return (super.shouldCreateToolTip(event) && (getText(event) != null));
		}

		@Override
		protected String getText(Event event) {
			TreeItem it = fViewer.getTree().getItem(new Point (event.x, event.y));
			if (it != null && it.getData() != null){
				// get the text associated to the selected element
				String name = getAdaName (it.getData());
				if (name == null){
					return null;
				} else {
					return getAdaName (it.getData());
				}
			}
			return null;
		}
	}//class ToolTip


	public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
		super.inputChanged(viewer, oldInput, newInput);
		fViewer = (TreeViewer) viewer;
		new GNATExplorerToolTip(fViewer);
	}

	/**
	 *
	 * @param file
	 * @return the Ada name
	 */
	public String getAdaNameFromFile(GeneralizedFile file){
		String name = new String();
		AdaConstruct[] construct = (AdaConstruct[]) getChildrenOfAdaSource(file);
		if (construct.length > 0){
			name += construct[0].getLabel(null);
		} else {
			return null;
		}

		for (int i = 1; i < construct.length; i++)
			name += (", " + construct[i].getLabel(null));
		return name;
	}

	/**
	 * Retrieve the Ada name of the file from its content
	 * @param an object
	 * @return the Ada name if it is a file, or the name of
	 * the resource
	 */
	public String getAdaName (Object node) {
		if (node instanceof IFile){
			try {
				if (GNATProjectRegistry.isManagedSourceFile((IFile) node)) {
					return getAdaNameFromFile(GeneralizedFile.fromWorkspace((IFile) node));
				}
			} catch (GNATbenchCoreException e) {
					GNATbenchCorePlugin.getDefault().logError(null, e);
			}
		} else if (node instanceof GeneralizedFile) {
			return getAdaNameFromFile((GeneralizedFile) node);
		} else if (node instanceof FileStoreReference) {
			final FileStoreReference ref = (FileStoreReference) node;

			if (!ref.isProject() && !ref.getFileStore().fetchInfo().isDirectory()) {
				//  We are on a file
				IFileStore file = ref.getFileStore();
				IProject project = null;

				if (ref.getResource() != null) {
					project = ref.getResource().getProject();
				}

				if (isAdaSource (file)) {
					return getAdaNameFromFile(GeneralizedFile.fromOSPath(
							project, URIUtil.toPath(file.toURI()).toOSString()));

				}
			} else {
				return ref.getFileStore().getName();
			}
		}

		//other cases
		return null;
	}

	public Object [] getChildren (Object parent) {

		if (parent instanceof IProject) {
			return getChildrenOfProject((IProject) parent);
		} else if (parent instanceof IFolder) {
			return getChildrenOfFolder((IFolder) parent);
		} else if (parent instanceof IFile) {
			try {
				if (GNATProjectRegistry.isManagedSourceFile((IFile) parent)) {
					return getChildrenOfAdaSource(GeneralizedFile
							.fromWorkspace((IFile) parent));
				}
			} catch (GNATbenchCoreException e) {
				GNATbenchCorePlugin.getDefault().logError(null, e);
			}
		} else if (parent instanceof GeneralizedFile) {
			return getChildrenOfAdaSource((GeneralizedFile) parent);
		} else if (parent instanceof FileStoreReference) {
			final FileStoreReference ref = (FileStoreReference) parent;

			if (ref.isProject()) {
				final GNATProjectRegistry reg = ((GNATProjectRegistryFactory) GNATbenchSession
						.getDefault().getProjectRegistryFactory())
						.getRegistryForPath(URIUtil.toPath(ref.getFileStore()
								.toURI()));

				if (reg != null) {
					ChildrenRunnable runnable = new ChildrenRunnable() {

						public void run() {
							GPRProject gpr = reg.getProjectFromLocation(URIUtil
									.toPath(ref.getFileStore().toURI()));

							if (gpr != null) {
								try {
									fResult = getChildrenOfGNATProject(gpr,
											ref.getResource()).toArray();
								} catch (GNATbenchCoreException e) {
									GNATbenchCorePlugin.getDefault().logError(
											null, e);
								}
							}
						}
					};

					reg.run(runnable);

					return runnable.fResult;
				}
			} else if (ref.getFileStore().fetchInfo().isDirectory()) {
				return getChildrenOfFolder(ref);
			} else {
				//  We are on a file

				IFileStore file = ref.getFileStore();
				IProject project = null;

				if (ref.getResource() != null) {
					project = ref.getResource().getProject();
				}

				if (isAdaSource (file)) {
					return getChildrenOfAdaSource(GeneralizedFile.fromOSPath(
							project, URIUtil.toPath(file.toURI()).toOSString()));
				}
			}
		} else if (parent instanceof AdaConstruct) {
			AdaConstruct construct = (AdaConstruct) parent;

			return construct.getChildren(fgFilterDeclarations);
		}

		return super.getChildren(parent);
	}

	public boolean hasChildren (Object element) {
		// This is done in order to avoid computing the children for all sub
		//  nodes when possible.

		try {
			if (element instanceof IFile
					&& GNATProjectRegistry.isManagedSourceFile((IFile) element)) {
				return true;
			} else if (element instanceof FileStoreReference) {
				FileStoreReference ref = (FileStoreReference) element;

				if (isAdaSource (ref.getFileStore())) {
					//  If we are on a source file, do not do any additional
					//  computation and assume that we have children.

					return true;
				}
			} else if (element instanceof AdaRuntimeNode) {
				return true;
			} else if (element instanceof IProject) {
				return ((IProject) element).isOpen();
			}
		} catch (GNATbenchCoreException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		}

		return super.hasChildren(element);
	}

	public Object getParent (Object element) {
		Object parent = null;

		if (element instanceof IFile) {
			IFile file = (IFile) element;

			try {
				parent = localizeParentOfFile(new FileStoreReference(
						GNATbenchCorePlugin.getDefault().getEFSRegistry()
								.getUniqueStore(
										URIUtil.toURI(file.getLocation())),
						file), null, ResourcesPlugin.getWorkspace().getRoot());
			} catch (GNATbenchCoreException e) {
				GNATbenchCorePlugin.getDefault().logError(null, e);
			}
		} else if (element instanceof FileStoreReference) {
			FileStoreReference ref = (FileStoreReference) element;

			parent = localizeParentOfFile(ref, null, ResourcesPlugin
					.getWorkspace().getRoot());
		} else if (element instanceof AdaConstruct){
			AdaConstruct construct = (AdaConstruct) element;
			AdaConstruct parentConstr = construct
					.getParent(fgFilterDeclarations);

			if (parentConstr == null) {
				GeneralizedFile genFile = construct.getLocation().getFile();

				if (genFile.getFile () != null) {
					parent = genFile.getFile();
				} else {
					parent = genFile;
				}
			} else {
				parent = parentConstr;
			}
		} else {
			parent = super.getParent(element);
		}

		return parent;
	}

	private Object localizeParentOfFile(FileStoreReference refToLocate,
			Object from, Object current) {

		if (current instanceof IWorkspaceRoot) {
			//  Fallback to the end of the procedure.
		} else if (current instanceof IProject) {
			if (refToLocate.getResource() == null
					|| !refToLocate.getResource().getProject().equals(
							(IProject) current)) {
				return null;
			}
		} else if (current instanceof IFolder) {
			IFolder folder = (IFolder) current;
			IResource[] resources = FileStoreRegistry.getResources(refToLocate
					.getFileStore());

			boolean found = false;

			for (int j = 0; j < resources.length; ++j) {
				if (folder.getLocation()
						.isPrefixOf(resources[j].getLocation())) {
					found = true;
					break;
				}
			}

			if (!found) {
				return null;
			}
		} else if (current instanceof IFile) {
			IFile fileFound = (IFile) current;

			if (fileFound.getLocation().equals(
					URIUtil.toPath(refToLocate.getFileStore().toURI()))) {

				return from;
			} else {
				return null;
			}
		} else if (current instanceof AdaRuntimeNode) {

		} else if (current instanceof AdaDependenciesNode) {

		} else if (current instanceof FileStoreReference) {
			FileStoreReference ref = (FileStoreReference) current;

			if (ref.getFileStore().equals(refToLocate.getFileStore())) {
				return from;
			}

			if (ref.getFileStore().fetchInfo().isDirectory()) {
				//  If we're on a directory, check that this directory is
				//  a prefix of the element we're looking for

				IResource[] resources = FileStoreRegistry.getResources(refToLocate
						.getFileStore());

				IPath refPath = URIUtil.toPath(ref.getFileStore().toURI());

				boolean found = false;

				for (int j = 0; j < resources.length; ++j) {
					if (refPath.isPrefixOf(resources[j].getLocation())) {
						found = true;
						break;
					}
				}

				if (!found) {
					return null;
				}
			} else {
				//  If we're not in a directory, then there's nothing more
				//  to do

				return null;
			}
		}

		Object [] children = getChildren (current);

		for (int j = 0; j < children.length; ++j) {
			Object result = localizeParentOfFile(refToLocate, current,
					children[j]);

			if (result != null) {
				return result;
			}
		}

		return null;
	}

	static private IAdaConstructFilterProvider fgFileFilter = new IAdaConstructFilterProvider() {
		public AdaConstructFilter getFilter() {
			return new AdaConstructCutFilter () {
				public boolean simpleFilter(AdaConstruct construct) {
					int category = construct.getCategory();
					return (category != Language_Category.Cat_With
							&& category != Language_Category.Cat_Use);
				}};
		}};

	/**
	 * Filter out all constructs that are not declarations
	 *
	 * TODO: Move this outside of the public interface.
	 */
	public static final IAdaConstructFilterProvider fgFilterDeclarations = new IAdaConstructFilterProvider() {
		public AdaConstructFilter getFilter() {
			return new AdaConstructCutFilter() {
				public boolean simpleFilter(AdaConstruct construct) {
					int category = construct.getCategory();

					return (category >= Language_Category.Cat_Package && category <= Language_Category.Cat_Local_Variable)
							|| category == Language_Category.Cat_Field;
				}
			};
		}
	};

	private Object [] getChildrenOfAdaSource (GeneralizedFile file) {
		AdaDocumentBuffer buffer = GNATbenchCorePlugin.getDefault()
				.getDocumentBuffer(file);

		AdaConstruct[] constructs = buffer.getAnalyzer().getRoots(fgFileFilter);

		return constructs;
	}

	private class GetChildrenOfProjectRunnable implements IWorkspaceRunnable {

		public Object [] fResult = new Object [0];

		private IProject fPrj;

		public GetChildrenOfProjectRunnable(IProject prj) {
			fPrj = prj;
		}

		public void run(IProgressMonitor monitor) throws CoreException {
			if (!fPrj.isOpen()) {
				return;
			}

			final GNATProjectRegistry registry = (GNATProjectRegistry) GNATbenchSession
					.getDefault().getOrLoadRegistry(fPrj);

			if (registry == null || !registry.isValid()) {
				try {
					fResult = fPrj.members();

					return;
				} catch (CoreException e) {
					GNATbenchCorePlugin.getDefault().logError(null, e);
				}
			}

			//  FIXME: potential leak - where is this listener unregistered ?
			registry.addListener(GPRExplorerContentProvider.this);

			try {
				final LinkedList<Object> result = new LinkedList<Object>();

				IResource[] resources = fPrj.members();

				for (IResource res : resources) {
					if (!GNATProjectRegistry.isManagedResource(res)) {
						result.add(res);
					}
				}

				ChildrenRunnable runnable = new ChildrenRunnable() {
					public void run() {
						GPRProject gpr =  registry.getGPRFor(fPrj);

						if (gpr != null) {
							try {
								result.addAll(getChildrenOfGNATProject(gpr, fPrj));

								result.add(new AdaRuntimeNode(this, fPrj));
							} catch (GNATbenchCoreException e) {
								GNATbenchCorePlugin.getDefault().logError(
										null, e);
							}
						}
					}
				};

				registry.run(runnable);

				fResult = result.toArray();
				return;
			} catch (CoreException e) {
				GNATbenchCorePlugin.getDefault().logError(null, e);
			} catch (GNATbenchCoreException e) {
				GNATbenchCorePlugin.getDefault().logError(null, e);
			}

			return;
		}
	}

	private Object [] getChildrenOfProject (final IProject project) {
		try {
			GetChildrenOfProjectRunnable runnable = new GetChildrenOfProjectRunnable (project);

			if (ResourcesPlugin.getWorkspace().isTreeLocked()) {
				//  If the tree is already locked for modification, we can't
				//  run a workspace runnable job - so just run the runnable

				runnable.run(new NullProgressMonitor ());
			} else {
				ResourcesPlugin.getWorkspace().run(
						runnable, project,
						IWorkspace.AVOID_UPDATE, new NullProgressMonitor());
			}
			return runnable.fResult;

		} catch (CoreException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);

			return new Object [0];
		}
	}

	/**
	 *
	 * @param gpr
	 * @param refResource Resource to be used when doing system resource
	 * references.
	 * @return
	 * @throws GNATbenchCoreException
	 */
	private LinkedList<Object> getChildrenOfGNATProject(GPRProject gpr,
			IResource refResource)
			throws GNATbenchCoreException {
		GNATProjectRegistry registry = gpr.getRegistry();

		LinkedList <Object> result = new LinkedList<Object> ();

		LinkedList<IFileStore> sourceDirs = gpr.getSourceDirectories(false);
		LinkedList<IFileStore> objectDirs = gpr.getObjectDirectories(false);
		IFileStore execDir = gpr.getExecDirectory();

		if (execDir != null) {
			boolean found = false;

			for (IFileStore objDir : objectDirs) {
				if (objDir.equals(execDir)) {
					found = true;
				}
			}

			if (!found) {
				objectDirs.add(execDir);
			}
		}

		//  Add the source directories at the root of the project

		for (IFileStore dir : sourceDirs) {
			IResource wsRes = FileStoreRegistry.getDefaultResource(dir);

			if (wsRes != null && wsRes instanceof IFolder) {
				result.add(wsRes);
			} else {
				FileStoreReference ref = new FileStoreReference(dir,
						refResource);
				ref.setData(resourceAnnotationKey, ResourceAnnotation.SourceFolder);
				result.add (ref);
			}
		}

		//  Add the object directories at the root of the project

		for (IFileStore dir : objectDirs) {
			IResource wsRes = FileStoreRegistry.getDefaultResource (dir);

			if (wsRes != null && wsRes instanceof IFolder) {
				result.add(wsRes);
			} else {
				FileStoreReference ref = new FileStoreReference(dir,
						refResource);
				ref.setData(resourceAnnotationKey, ResourceAnnotation.BinaryFolder);

				result.add (ref);
			}
		}

		if (gpr.getDependencies(false).size() > 0) {
			result.add(new AdaDependenciesNode(registry, URIUtil.toPath(gpr
					.getProjectFile().toURI())));
		}

		return result;
	}

	private Object [] getChildrenOfFolder (IFolder folder) {
		try {
			IResource [] members = folder.members();

			if (GNATProjectRegistry.isManagedSourceFolder(folder)) {
				LinkedList <Object> result = new LinkedList <Object> ();

				for (int j = 0; j < members.length; ++j) {
					if (GNATProjectRegistry.isManagedSourceFile(members[j])) {
						result.add(members[j]);
					}
				}

				return result.toArray();
			} else {
				return members;
			}
		} catch (GNATbenchCoreException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		} catch (CoreException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		}

		return new Object [0];
	}

	private Object [] getChildrenOfFolder (FileStoreReference folder) {
		ResourceAnnotation annot = (ResourceAnnotation) folder
				.getData(GPRExplorerContentProvider.resourceAnnotationKey);

		try {
			if (GNATProjectRegistry
					.isManagedSourceFolder(folder.getFileStore())
					&& (annot == null || annot == ResourceAnnotation.SourceFolder)) {
				IFileStore res[] = GNATbenchCorePlugin.getDefault()
				.getEFSRegistry().getUniqueChildren(
						folder.getFileStore());

				LinkedList <Object> result = new LinkedList <Object> ();

				for (IFileStore file : res) {
					if (GNATProjectRegistry.isManagedSourceFile(file)) {
						result.add(new FileStoreReference(file, folder
								.getResource()));
					}
				}

				return result.toArray();
			} else {
				IFileStore res[] = GNATbenchCorePlugin.getDefault()
				.getEFSRegistry().getUniqueChildren(
						folder.getFileStore());

				Object children[] = new Object[res.length];

				for (int i = 0; i < res.length; ++i) {
					children[i] = new FileStoreReference(res[i], folder
							.getResource());
				}

				return children;
			}
		} catch (GNATbenchCoreException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		} catch (CoreException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		}

		return new Object [0];
	}

	public void recomputed(IGNATProjectRegistry registry) {
		if (fViewer != null) {
			Display.getDefault().asyncExec(new Runnable () {
				public void run() {
					fViewer.refresh();
				}});
		}
	}

	/**
	 * Return true if this editor is an Ada file according to the editor
	 * preferences.
	 * @return
	 */
	public static boolean isAdaSource (IFileStore file) {
		IEditorDescriptor editorDesc = GNATbenchUIPlugin.getDefault().getWorkbench().getEditorRegistry()
		.getDefaultEditor(file.getName());

		if (editorDesc != null
				&& editorDesc.getId().matches("com\\.adacore\\.gnatbench\\..*")) {
			return true;
		} else {
			return false;
		}
	}

	@Override
	protected Object [] addedObjectsFilter (Object [] objs) {
		LinkedList <Object> result = new LinkedList <Object> ();

		for (Object o : objs) {
			if (getParent (o) instanceof IResource) {
				result.add(o);
			}
		}

		return result.toArray();
	}

	protected void processDelta(IResourceDelta delta) {
		super.processDelta(delta);

		try {
			new IResourceDeltaVisitor () {

				public boolean visit(final IResourceDelta delta) throws CoreException {
					if ((delta.getFlags() & IResourceDelta.OPEN) != 0) {
						Display.getDefault().asyncExec(new Runnable () {

							public void run() {
								//  On project opening, we don't have any
								//  registry listener anymore - so we need
								//  to do an explicit refresh in order to have
								//  corresponding nodes added to the view.
								fViewer.refresh (delta.getResource());
							}});
					} else {
						for (IResourceDelta sub : delta.getAffectedChildren()) {
							visit (sub);
						}
					}

					return false;
				}

			}.visit(delta);
		} catch (CoreException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		}
	}
}
