/*******************************************************************************
 * Copyright (C) 2006-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *******************************************************************************/

package com.adacore.gnatbench.ui.internal.jobs;

import java.io.BufferedReader;
import java.io.File;
import java.io.InputStream;
import java.io.InputStreamReader;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.WorkspaceJob;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;

import com.adacore.gnatbench.core.internal.GNATbenchCorePlugin;
import com.adacore.gnatbench.core.internal.GNATbenchProjectProperties;
import com.adacore.gnatbench.core.internal.toolchain.ToolChain;
import com.adacore.gnatbench.core.internal.toolchain.ToolChain.NoExtensionFound;
import com.adacore.gnatbench.ui.internal.GNATbenchUIPlugin;
import com.adacore.gnatbench.ui.internal.utils.DisplayUtils;

public class GNATstubJob extends WorkspaceJob {

	String specFileName;
	String bodyFileName;
	String workingDirName;
	IProject project;

	public GNATstubJob(String workingDir, String specFileName,
			String bodyFileName, IProject project) {
		super("Ada Body Generation");

		this.specFileName = specFileName;
		this.bodyFileName = bodyFileName;
		this.workingDirName = workingDir;
		this.project = project;
	} // ctor

	public IStatus runInWorkspace(IProgressMonitor monitor) {
		final GNATbenchProjectProperties props = GNATbenchProjectProperties
				.getPropertiesFor(project);
		String projectFileName = props.getGPROSPath();

		if (System.getProperty("os.name").startsWith("Windows")) {
			projectFileName = "\"" + projectFileName + "\"";
			specFileName = "\"" + specFileName + "\"";
			bodyFileName = "\"" + bodyFileName + "\"";
		} // if

		// Switch -f forces an overwrite if the output file already exists.
		// The initiator of this job is responsible for determining if that
		// is appropriate.

		final String[] commandTail = { "stub", "-f", "-q", "-P",
				projectFileName, specFileName, "-o", bodyFileName };

		// ultimately the command must start with the builder name prefix and
		// "gnat", such as "gnat stub" or "powerpc-elf-gnat stub" and so on.

		final String builderName;
		try {
			builderName = ToolChain.active().name(project);
			// eg: i586-wrs-vxworks-gnatmake, or just gnatmake, or gnaampmake,
			// or ...
		} catch (NoExtensionFound e) {
			return new Status(
					IStatus.ERROR,
					GNATbenchUIPlugin.getId(),
					IStatus.OK,
					"GNATstubJob failed because no toolchain definition was found",
					null);
		} // try

		final int makePos = builderName.lastIndexOf("make");
		if (makePos == -1) {
			return new Status(IStatus.ERROR, GNATbenchUIPlugin.getId(),
					IStatus.OK, "GNATstubJob could not recognize builder '"
							+ builderName + "'", null);
		} // if

		String[] actualCommand = new String[1 + commandTail.length];

		actualCommand[0] = builderName.subSequence(0, makePos).toString();

		for (int j = 0; j < commandTail.length; j++) {
			actualCommand[1 + j] = commandTail[j];
		} // for

		final File workingDir = new File(workingDirName);

		try {
			Process proc = Runtime.getRuntime().exec(actualCommand, null,
					workingDir);

			InputStream stderr = proc.getErrorStream();
			InputStreamReader isr = new InputStreamReader(stderr);
			BufferedReader br = new BufferedReader(isr);

			String line = null;
			String message = null;
			while ((line = br.readLine()) != null) {
				if (message == null) {
					message = line;
				} else {
					message = message + "\n" + line;
				} // if
			} // while
			if (message != null) {
				DisplayUtils.displayError("Package Body Stub Generation",
						message);
			} // if

			final int exitVal = proc.waitFor();
			if ((exitVal != 0) && (message == null)) {
				return new Status(IStatus.ERROR, GNATbenchUIPlugin.getId(),
						IStatus.OK, "GNATstubJob failed with error code "
								+ exitVal, null);
			} // if
		} catch (Throwable e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
			return new Status(IStatus.ERROR, GNATbenchUIPlugin.getId(),
					IStatus.OK, "GNATstubJob failed due to an exception", null);
		} // try

		return new Status(IStatus.OK, GNATbenchUIPlugin.getId(), IStatus.OK,
				"", null);
	} // runInWorkspace

} // GNATstubJob
