/*******************************************************************************
 * Copyright (C) 2007-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *******************************************************************************/

package com.adacore.gnatbench.ui.internal.codeassist;

import java.util.LinkedList;

import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.ITextViewer;
import org.eclipse.jface.text.TextSelection;
import org.eclipse.jface.text.contentassist.ContentAssistEvent;
import org.eclipse.jface.text.contentassist.ICompletionListener;
import org.eclipse.jface.text.contentassist.ICompletionProposal;
import org.eclipse.jface.text.contentassist.IContentAssistProcessor;
import org.eclipse.jface.text.contentassist.IContentAssistant;
import org.eclipse.jface.text.contentassist.IContentAssistantExtension2;
import org.eclipse.jface.text.contentassist.IContextInformation;
import org.eclipse.jface.text.contentassist.IContextInformationValidator;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;

import com.adacore.gnatbench.core.GNATbenchSession;
import com.adacore.gnatbench.core.internal.GNATbenchCorePlugin;
import com.adacore.gnatbench.core.internal.adaeditor.AdaDocumentBuffer;
import com.adacore.gnatbench.core.internal.analyzer.AdaConstruct;
import com.adacore.gnatbench.core.internal.codingstyle.AdaIndentBuffer;
import com.adacore.gnatbench.core.internal.projects.GNATProjectRegistry;
import com.adacore.gnatbench.library.LibraryMonitor;
import com.adacore.gnatbench.library.LibrarySemaphore;
import com.adacore.gnatbench.library.Completion.Completion_Iterator;
import com.adacore.gnatbench.library.Completion.Completion_List;
import com.adacore.gnatbench.library.Completion.Completion_Manager;
import com.adacore.gnatbench.library.Completion.Completion_Package;
import com.adacore.gnatbench.library.Completion.Completion_Proposal;
import com.adacore.gnatbench.library.Completion.Completion_Resolver;
import com.adacore.gnatbench.library.Completion.Ada.Ada_Completion_Manager;
import com.adacore.gnatbench.library.Completion.Ada.Constructs_Extractor.Constructs_Extractor_Package;
import com.adacore.gnatbench.library.Completion.History.Storable_Proposal;
import com.adacore.gnatbench.library.Utils.Utils_Package;
import com.adacore.gnatbench.library.GNATCOLL.Filesystem.Filesystem_String;
import com.adacore.gnatbench.library.GNATCOLL.VFS.VFS_Package;
import com.adacore.gnatbench.library.GNATCOLL.VFS.Virtual_File;
import com.adacore.gnatbench.library.Language.Ada.Ada_Package;
import com.adacore.gnatbench.ui.internal.adaeditor.AdaEditor;
import com.adacore.ajis.IProxy;
import com.adacore.ajis.IProxy.Owner;
import com.adacore.gnatbench.library.Standard.AdaString;

/**
 * This class process completions using the GPS library.
 */
public class AdaCompletionProcessor implements IContentAssistProcessor,
		ICompletionListener {

	private AdaEditor fEditor;
	private Completion_Manager fManager;
	private Completion_Resolver fResolver;
	private Completion_List fList;
	private LinkedList<AdaCompletionProposal> fLastResults = new LinkedList<AdaCompletionProposal>();

	private AdaString fAdaBuffer;
	//  It's expected to have this string never read locally, since it's here
	//  only to prevent premature garbage collection

	public class AdaCompletionProposal implements ICompletionProposal {

		Completion_Proposal fProposal;
		int fCompletedLength;
		int fFrom;
		CompletionKernelAssistant fCompletionAssistant;

		public AdaCompletionProposal(Completion_Proposal proposal, int from,
				int completedLength,
				CompletionKernelAssistant completionAssistant) {
			fProposal = proposal;
			fFrom = from;
			fCompletedLength = completedLength;
			fCompletionAssistant = completionAssistant;
		}

		public void apply(IDocument document) {
			LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

			try {
				String completionStr = fProposal.Get_Completion().toString();

				document.replace
				   (fFrom,
					fCompletedLength,
					completionStr);

				//  The following code assumes that the caret will always be
				//  put on the first line (in the multiple lines completion
				//  case) and the indentation is thus done only on the next
				//  ones.

				int nextLineOffset = completionStr.indexOf("\n");

				if (nextLineOffset != -1) {
					TextSelection selection = new TextSelection(document, fFrom
							+ nextLineOffset + 1, completionStr.length()
							- nextLineOffset - 1);

				   new AdaIndentBuffer(fEditor.getDocumentBuffer())
							.format(selection, true);
				}

				fEditor.getSourceViewerPublic().getTextWidget().setCaretOffset(
						fFrom + fProposal.Get_Caret_Offset());


				if (fProposal instanceof Storable_Proposal) {
					fCompletionAssistant.getHistory().Prepend_Proposal(fProposal);
				}
			} catch (BadLocationException e) {
				GNATbenchCorePlugin.getDefault().logError(null, e);
			} finally {
				LibrarySemaphore.stopGPSWork(libMonitor);
			}
		}
		public String getAdditionalProposalInfo() {
			String res;

			LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

			try {
				res = fProposal.Get_Documentation().toString();
			} finally {
				LibrarySemaphore.stopGPSWork(libMonitor);
			}

			res = res.replaceAll("\n", "\n<br/>");

			return res;
		}

		public IContextInformation getContextInformation() {
			// TODO Auto-generated method stub
			return null;
		}

		public String getDisplayString() {
			LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

			try {
				return fProposal.Get_Label().toString();
			} finally {
				LibrarySemaphore.stopGPSWork(libMonitor);
			}
		}

		public Image getImage() {
			LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

			try {
				return AdaConstruct.getImage(fProposal.Get_Category(),
						fProposal.Get_Visibility());
			} finally {
				LibrarySemaphore.stopGPSWork(libMonitor);
			}
		}

		public Point getSelection(IDocument document) {
			// TODO Auto-generated method stub
			return null;
		}

		public void dispose () {
			//  FIXME: This statement causes segmentation faults - investigate
			//  why and how this should be freed (happen not to be freed in
			//  GPS either)
			// fProposal.Free();
		}
	}

	public AdaCompletionProcessor (AdaEditor editor, IContentAssistant contentAssistant) {
		fEditor = editor;

		if (contentAssistant instanceof IContentAssistantExtension2) {
			//  Without this, the container will not be able to retreive the
			//  memory used during the completion

			((IContentAssistantExtension2) contentAssistant)
					.addCompletionListener(this);
		}
	}

	public ICompletionProposal[] computeCompletionProposals(
			ITextViewer viewer, int offset) {
		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try {
			synchronized (this) {
				freePreviousSession();

				GNATProjectRegistry registry = (GNATProjectRegistry) GNATbenchSession
						.getDefault().getOrLoadRegistry(
								fEditor.getFile().getProject());

				if (registry == null) {
					return null;
				}

				CompletionKernelAssistant completionAssistant = (CompletionKernelAssistant) registry
						.getAssistant(CompletionKernelAssistant.ASSISTANT_ID);

				// Force update of the current file in order to use it last
				// contents.
				registry.fConstructDatabase.Update_Contents(fEditor.getFile()
						.toVirtualFile());

				AdaDocumentBuffer buffer = fEditor.getDocumentBuffer();
				Virtual_File currentFile = VFS_Package
						.Create(new Filesystem_String(buffer.getFile()
								.getOSPath()));

				fManager = new Ada_Completion_Manager();
				((IProxy) fManager).setOwner(Owner.NATIVE);

				fAdaBuffer = buffer.getAdaString();
				//  It is absolutely mandatory to hold a reference on the
				//  string here, in order to avoid unexpected garbage
				//  collection while the resolver / manager / context is
				//  still existing on the native side.

				fResolver = Constructs_Extractor_Package
						.New_Construct_Completion_Resolver(
								registry.fConstructDatabase, currentFile, fAdaBuffer);

				fManager.Register_Resolver(completionAssistant.getHistory());
				fManager.Register_Resolver(fResolver);

				int utf8Offset = Utils_Package.UTF16_Offset_To_UTF8_Offset(
						fAdaBuffer, offset);

				fList = fManager.Get_Initial_Completion_List(fManager
						.Create_Context(currentFile, fAdaBuffer,
								Ada_Package.Ada_Lang(), utf8Offset));

				Completion_Iterator it = fList.First();

				int completedLength = fList.Get_Completed_String().toString()
						.length();

				int total = 0;

				// TODO: Here, we simply limit at 100 elements - we should find
				// a way to do something smarter in the future.
				while (!it.At_End() && total < 100) {
					fLastResults.add(new AdaCompletionProposal(it.Get_Proposal(),
							offset - completedLength, completedLength,
							completionAssistant));

					it.Next();

					total = total + 1;
				}

				it.Free();

				return fLastResults.toArray(new ICompletionProposal[fLastResults
						.size()]);
			}
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}
	}

	public IContextInformation[] computeContextInformation(ITextViewer viewer,
			int offset) {
		// TODO Auto-generated method stub
		return null;
	}

	public char[] getCompletionProposalAutoActivationCharacters() {
		return new char [] {'.', '('};
	}

	public char[] getContextInformationAutoActivationCharacters() {
		// TODO Auto-generated method stub
		return null;
	}

	public IContextInformationValidator getContextInformationValidator() {
		// TODO Auto-generated method stub
		return null;
	}

	public String getErrorMessage() {
		// TODO Auto-generated method stub
		return null;
	}

	public void assistSessionEnded(ContentAssistEvent event) {
		freePreviousSession();
	}

	public void assistSessionStarted(ContentAssistEvent event) {
		// TODO Auto-generated method stub

	}

	public void selectionChanged(ICompletionProposal proposal,
			boolean smartToggle) {
		// TODO Auto-generated method stub

	}

	public void freePreviousSession () {
		if (fManager != null) {
			LibraryMonitor monitor = LibrarySemaphore.startGPSWork();

			try {
				synchronized (this) {
					fList.Free();
					fResolver.Free();
					Completion_Package
							.Free((Completion_Manager.Ref) fManager
									.NewProxyRef());

					fManager = null;
					fList = null;
					fResolver = null;

					fAdaBuffer = null;
					//  Now that we deleted all native object using the string,
					//  we don't need to avoid garbage collection anymore and
					//  can safely remove the reference.

					for (AdaCompletionProposal prop : fLastResults) {
						prop.dispose();
					}

					fLastResults.clear();
				}
			} finally {
				LibrarySemaphore.stopGPSWork(monitor);
			}
		}
	}
}
