/*******************************************************************************
 * Copyright (c) 2005, 2007 AdaCore.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *******************************************************************************/

package com.adacore.gnatbench.ui.internal.browsing;

import java.util.LinkedList;

/**
 * This is the base class for every node displayed in the AdaSearchWindow.
 * 
 * @author Quentin Ochem
 */
public class SearchNode {
	private SearchNode fParent;
	private LinkedList <SearchNode> fChildren = new LinkedList<SearchNode>();
	protected LinkedList <MatchTag> fMatches = new LinkedList<MatchTag>();
	private boolean fInDispose = false;
	
	/**
	 * Create a new node for the parent given in parameter. The child list of 
	 * the parent is automatically updated.
	 * 
	 * @param parent
	 */
	public SearchNode(SearchNode parent) {
		fParent = parent;
		
		if (fParent != null) {
			fParent.fChildren.add(this);
		}
	}
	
	/**
	 * Return the parent of the node, null if the node is a root.
	 * 
	 * @return
	 */
	public SearchNode getParent() {
		return fParent;
	}
	
	/**
	 * Remove all the childs of the node, and the current node from the parent.
	 * If the parent becomes an empty node, then dispose will be called on it
	 * as well (and recursively).
	 *
	 */
	public void dispose() {
		if (fInDispose) return;
		fInDispose = true;
		
		for (SearchNode element : fChildren) {
			element.dispose();
		}
		
		if (fParent != null) {
			fParent.removeChild(this);
			
			if (fParent.getChildren().size() == 0) {
				fParent.dispose();
			}
		}
		
		fInDispose = false;
		
		hide();
	}
	
	/**
	 * Removes a child from a node. This call is secured, if the node is 
	 * currently in a dispose call, then nothing will be done here.
	 * 
	 * @param node
	 */
	private void removeChild(SearchNode node) {
		if (fInDispose) {
			return;
		}

		fChildren.remove(node);
	}
	
	/**
	 * Returns the children list of the node. This list should no be changed 
	 * outside the class.
	 * 
	 * @return
	 */
	public LinkedList <SearchNode> getChildren() {
		return fChildren;
	}
	
	/**
	 * Adds a match to the match list of this node.
	 * 
	 * @param match
	 */
	public void addMatch (MatchTag match) {
		fMatches.add(match);
	}

	/**
	 * Hides the corresponding matches, e.g. display the annotations in the 
	 * text viewer.
	 */
	public void hide() {
		for (MatchTag element : fMatches) {
			element.hide();
		}
	}
	
	/**
	 * Shows the corresponding matches, e.g. display the annotations in the 
	 * text viewer.
	 */
	public void show() {
		for (MatchTag element : fMatches) {
			element.show();
		}
	}

	/**
	 * Return the match that is set at the corresponding offset, inside the
	 * node, null if none.
	 * 
	 * @param offset
	 * @return
	 */
	public MatchTag getMatchAt(int offset) {
		for (MatchTag element : fMatches) {
			if (element.getOffset() == offset) {
				return element;
			}
		}
		
		return null;
	}
	
	/**
	 * Returns the list of the matches contained in this node.
	 * 
	 * @return
	 */
	public LinkedList <MatchTag> getMatches() {
		return fMatches;
	}
}
