/*******************************************************************************
 * Copyright (C) 2005-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *******************************************************************************/

package com.adacore.gnatbench.ui.internal.browsing;

import org.eclipse.core.resources.IMarker;
import org.eclipse.core.runtime.CoreException;

import com.adacore.gnatbench.core.internal.GNATbenchCorePlugin;
import com.adacore.gnatbench.core.internal.analyzer.GeneralizedLocation;

/**
 * This class stores matches, and manage their graphical behavior.
 */
public class MatchTag {
	private GeneralizedLocation fLocation;
	private IMarker fMarker;
	private String fName;
	private int fOffset;
	SearchNode fNode;

	/**
	 * Creates an entity tag, and insert the corresponding marker into the
	 * text.
	 * @param location
	 * @param name
	 */
	public MatchTag(GeneralizedLocation location, String name, int offset,
			SearchNode node) {
		fLocation = location;
		fName = name;
		fOffset = offset;
		fNode = node;
	}

	public int getLine() {
		return fLocation.getLine() - 1;
	}

	public int getColumn() {
		return fLocation.getColumn().getColumnIndex();
	}

	public String getName() {
		return fName;
	}

	public IMarker getMarker() {
		return fMarker;
	}

	public SearchNode getNode() {
		return fNode;
	}

	/**
	 * Creates the corresponding marker, if the resource is in the workspace.
	 */
	private void createMarker() {
		if (fLocation.getFile().getFile () != null) {
			fMarker = fLocation.toMarker(
					GNATbenchCorePlugin.ADA_LOCATION_MARKER);

			try {
				fMarker.setAttribute(GNATbenchCorePlugin.LOCATION_MARKER_TEXT,
						fName);
			} catch (CoreException e) {
				GNATbenchCorePlugin.getDefault().logError(null, e);
			}
		}
	}

	/**
	 * Deletes the corresponding marker.
	 */
	private void deleteMarker() {
		if (fMarker != null) {
			try {
				fMarker.delete();
				fMarker = null;
			} catch (CoreException e) {
				GNATbenchCorePlugin.getDefault().logError(null, e);
			}
		}
	}

	/**
	 * Shows the corresponding marker and annotation.
	 */
	public void show() {
		fLocation = GNATbenchCorePlugin.getDefault().getDocumentBuffer(
				fLocation.getFile()).getClosestLocation(fLocation, fName);

		createMarker();
	}

	/**
	 * Hides the corresponding marker and annotation.
	 */
	public void hide() {
		deleteMarker();
	}

	/**
	 * @return the location of the match.
	 */
	public GeneralizedLocation getLocation() {
		if (fMarker != null) {
			return GeneralizedLocation.fromTextMarker(fMarker);
		} else {
			return fLocation;
		}
	}

	/**
	 * @return the offset where the match has initially been created.
	 */
	public int getOffset() {
		return fOffset;
	}
}

