/*******************************************************************************
 * Copyright (C) 2005-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *******************************************************************************/

package com.adacore.gnatbench.ui.internal.browsing;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ILabelProviderListener;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.OpenEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.StructuredViewer;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.search.ui.ISearchResultListener;
import org.eclipse.search.ui.SearchResultEvent;
import org.eclipse.search.ui.text.AbstractTextSearchViewPage;
import org.eclipse.search.ui.text.Match;
import org.eclipse.search.ui.text.MatchEvent;
import org.eclipse.search.ui.text.RemoveAllEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.PartInitException;

import com.adacore.gnatbench.core.internal.GNATbenchCorePlugin;
import com.adacore.gnatbench.core.internal.analyzer.GeneralizedFile;
import com.adacore.gnatbench.core.internal.analyzer.GeneralizedLocation;
import com.adacore.gnatbench.ui.internal.GNATbenchUIPlugin;
import com.adacore.gnatbench.ui.internal.adaeditor.AdaEditor;

/**
 * This is the view for AdaSearchResult.
 */
public class AdaSearchView extends AbstractTextSearchViewPage {
	public static final int DISPLAY_TABLE = 1;
	public static final int DISPLAY_TREE = 2;

	private Action fCurrentGroupMode = null;

	private int fDisplayMode;

	/**
	 * This action groups the matches by project.
	 *
	 * @author Quentin Ochem
	 */
	private class ProjectGroup extends Action {
		public ProjectGroup() {
			super ("Group By Project", AS_CHECK_BOX);

			setToolTipText("Group By Project");
		}

		public ImageDescriptor getImageDescriptor() {
			return GNATbenchUIPlugin.getDefault().getImageDescriptor("/img/",
					"prj_mode.gif");
		}

		public void run() {
			fCurrentGroupMode.setChecked(false);
			fCurrentGroupMode = this;
			fCurrentGroupMode.setChecked(true);

			AdaSearchView.this.getViewer().refresh();
		}
	}

	/**
	 * This action groups the matches by file.
	 *
	 * @author Quentin Ochem
	 */
	private class FileGroup extends Action {
		public FileGroup() {
			super ("Group By File", AS_CHECK_BOX);

			setToolTipText("Group By File");
		}


		public ImageDescriptor getImageDescriptor() {
			return GNATbenchUIPlugin.getDefault().getImageDescriptor("/img/",
					"file_mode.gif");
		}

		public void run() {
			fCurrentGroupMode.setChecked(false);
			fCurrentGroupMode = this;
			fCurrentGroupMode.setChecked(true);

			AdaSearchView.this.getViewer().refresh();
		}
	}

	/**
	 * This action groups the matches by package.
	 */
	private class PackageGroup extends Action {
		public PackageGroup() {
			super ("Group By Package", AS_CHECK_BOX);

			setToolTipText("Group By Package");
		}

		public ImageDescriptor getImageDescriptor() {
			return GNATbenchUIPlugin.getDefault().getImageDescriptor("/img/",
					"package_mode.gif");
		}

		public void run() {
			fCurrentGroupMode.setChecked(false);
			fCurrentGroupMode = this;
			fCurrentGroupMode.setChecked(true);

			AdaSearchView.this.getViewer().refresh();
		}
	}

	private class SearchContentProvider	implements ITreeContentProvider, ISearchResultListener {

		private AdaSearchResult fResults;

		public Object[] getChildren(Object parentElement) {
			if (parentElement instanceof SearchNode) {
				Object[] result = ((SearchNode) parentElement).getChildren()
						.toArray();
				return result;
			}

			return new Object[]{};
		}

		public Object getParent(Object element) {
			if (element instanceof SearchNode) {
				return ((SearchNode) element).getParent();
			}

			return null;
		}

		public boolean hasChildren(Object element) {
			if (element instanceof FileNode
					|| element instanceof FolderNode
					|| element instanceof ProjectNode) {
				return true;
			} else {
				return ((SearchNode) element).getChildren().size() > 0;
			}
		}

		public Object[] getElements(Object inputElement) {
			fResults = (AdaSearchResult) inputElement;

			if (fDisplayMode == DISPLAY_TABLE) {
				Object [] r = fResults.getMatchingNodes().toArray();

				return r;
			} else if (fDisplayMode == DISPLAY_TREE) {
				if (fCurrentGroupMode instanceof FileGroup) {
					return fResults.getFileNodes().toArray();
				} else if (fCurrentGroupMode instanceof ProjectGroup) {
					return fResults.getProjectNodes().toArray();
				} else if (fCurrentGroupMode instanceof PackageGroup) {
					return fResults.getPackageNodes().toArray();
				}
			}

			return new Object[]{};
		}

		public void dispose() {
			if (fResults != null) {
				fResults.clear();
			}
		}

		public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
			if (fResults != null) {
				fResults.removeListener(this);
				fResults.hide();
			}

			if (newInput instanceof AdaSearchResult) {
				fResults = (AdaSearchResult) newInput;
				fResults.show();
				fResults.addListener(this);
			} else {
				fResults = null;
			}
		}

		/* (non-Javadoc)
		 * @see org.eclipse.search.ui.ISearchResultListener#searchResultChanged(org.eclipse.search.ui.SearchResultEvent)
		 */
		public void searchResultChanged(SearchResultEvent e) {
			if (e instanceof MatchEvent) {
				MatchEvent evt = (MatchEvent) e;

				if (evt.getKind() == MatchEvent.REMOVED) {
					fViewer.refresh();
				}
			}
			else if (e instanceof RemoveAllEvent) {

			}
		}
	} // SearchContentProvider

	private class SearchLabelProvider implements ILabelProvider {

		public Image getImage(Object element) {
			if (element instanceof FolderNode) {
				return GNATbenchUIPlugin.getDefault().getImage(
						GNATbenchUIPlugin.ICON_FOLDER);
			} else if (element instanceof FileNode) {
				return GNATbenchUIPlugin.getDefault().getImage(
						GNATbenchUIPlugin.ICON_ADA_FILE);
			} else if (element instanceof ProjectNode) {
				return GNATbenchUIPlugin.getDefault().getImage(
						GNATbenchUIPlugin.ICON_PROJECT);
			} else if (element instanceof ConstructNode) {
				return ((ConstructNode) element).getConstruct().getImage();
			}

			return null;
		}

		/**
		 * Returns the text corresponding to the given element, adding the file
		 * or path name when relevant.
		 */
		public String getText(Object element) {
			if (element instanceof SearchNode) {
				if (isRootNode(((SearchNode)element))) {
					if (element instanceof ConstructNode) {
						ConstructNode construct = (ConstructNode) element;

						return element.toString()
								+ " - "
								+ construct.getConstruct().getLocation()
										.getFile().getOSPath();
					} else if (element instanceof FileNode) {
						FileNode file = (FileNode) element;

						return file.getName()
								+ " - "
								+ file.getAdaFile().getOSPath()
										.replaceAll("[^\\\\/]*$", "");
					}
				}

				if (element instanceof FileNode) {
					return ((FileNode) element).getName();
				}
			}

			return element.toString();
		}

		public void addListener(ILabelProviderListener listener) {
			// TODO Auto-generated method stub

		}

		public void dispose() {
			// TODO Auto-generated method stub
		}

		public boolean isLabelProperty(Object element, String property) {
			// TODO Auto-generated method stub
			return false;
		}

		public void removeListener(ILabelProviderListener listener) {
			// TODO Auto-generated method stub

		}
	}

	/**
	 * Implements the behavior of a double click on an item in the tree.
	 *
	 * @author Quentin Ochem
	 */
	private class DoubleClick implements IDoubleClickListener {

		public void doubleClick(DoubleClickEvent event) {
			jumpToSelection();
		}

	}

	private SearchContentProvider fContentProvider = null;
	private StructuredViewer fViewer = null;

	public AdaSearchView() {
		fContentProvider = new SearchContentProvider();
	}

	protected void elementsChanged(Object[] objects) {
		if (getViewer() instanceof StructuredViewer) {
			StructuredViewer viewer = (StructuredViewer) getViewer();
			viewer.refresh();
		}
	}

	protected void clear() {
		fContentProvider.dispose();

		if (fViewer != null) {
			fViewer.refresh();
			fViewer.setInput(null);
		}
	}

	public void dispose() {
		fContentProvider.dispose();
	}

	protected void configureTreeViewer(TreeViewer viewer) {
		fDisplayMode = DISPLAY_TREE;

		viewer.setLabelProvider(new SearchLabelProvider ());
		viewer.setContentProvider(fContentProvider);
		viewer.addDoubleClickListener(new DoubleClick());

		fViewer = viewer;
	}

	protected void configureTableViewer(TableViewer viewer) {
		fDisplayMode = DISPLAY_TABLE;

		viewer.setLabelProvider(new SearchLabelProvider ());
		viewer.setContentProvider(fContentProvider);
		viewer.addDoubleClickListener(new DoubleClick());

		fViewer = viewer;
	}

	public void gotoNextMatch() {
		super.gotoNextMatch();

		jumpToMatch();
	}

	public void gotoPreviousMatch() {
		super.gotoPreviousMatch();

		jumpToMatch();
	}

	/**
	 * Focus the editor to the current match. Fix the position if necessary
	 * possible.
	 */
	public void jumpToMatch () {
		Match match = getCurrentMatch();

		if (match == null) return;

		SearchNode node = ((SearchNode) match.getElement());
		MatchTag matchTag = node.getMatchAt(match.getOffset());

		GeneralizedLocation location = matchTag.getLocation();

		try {
			IEditorPart editor = GNATbenchUIPlugin.getDefault().openEditor(
					location);

			if (editor instanceof AdaEditor) {

				AdaEditor adaEditor = (AdaEditor) editor;

				adaEditor.focus(location);
				setFocus();
			}
		} catch (PartInitException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		}
	}

	/**
	 * Jump to the current object selected in the search view, if relevant.
	 */
	public void jumpToSelection () {
		Object node = ((StructuredSelection) fViewer.getSelection())
				.getFirstElement();

		if (node instanceof FileNode) {
			FileNode file = (FileNode) node;

			GeneralizedFile genFile = file.getAdaFile();

			try {
				GNATbenchUIPlugin.getDefault().openEditor(genFile);
			} catch (PartInitException e) {
				GNATbenchCorePlugin.getDefault().logError(null, e);
			}
		} else if (node instanceof ConstructNode) {
			ConstructNode construct = (ConstructNode) node;

			GeneralizedLocation location = construct.getConstruct()
					.getLocation();

			try {
				GNATbenchUIPlugin.getDefault().openEditor(location);
			} catch (PartInitException e) {
				GNATbenchCorePlugin.getDefault().logError(null, e);
			}

		}
	}

	/**
	 * This function is overloaded here in order to remove the default behavior
	 * when double-clicking on an entry of the view.
	 */
	protected void handleOpen(OpenEvent event) {
		StructuredSelection selection =
			(StructuredSelection) event.getSelection();

		if (selection.getFirstElement() instanceof FolderNode
				|| selection.getFirstElement() instanceof ProjectNode) {
			super.handleOpen(event);
		}
	}

	protected void fillToolbar(IToolBarManager tbm) {
		super.fillToolbar(tbm);

		if (fDisplayMode == DISPLAY_TREE) {
			fCurrentGroupMode = new PackageGroup();
			fCurrentGroupMode.setChecked(true);

			tbm.appendToGroup("group.viewerSetup", new Separator());
			tbm.appendToGroup("group.viewerSetup", new ProjectGroup());
			tbm.appendToGroup("group.viewerSetup", new FileGroup());
			tbm.appendToGroup("group.viewerSetup", fCurrentGroupMode);
		}
	}

	/**
	 * Return true if the node given in parameter is supposed to be one of the
	 * root of the view, false otherwise.
	 *
	 * @param node
	 * @return
	 */
	private boolean isRootNode (SearchNode node) {
		if (node.getParent() == null) {
			return true;
		}

		if (fDisplayMode == DISPLAY_TABLE) {
			return true;
		} else if (fDisplayMode == DISPLAY_TREE) {
			if (fCurrentGroupMode instanceof FileGroup) {
				return node instanceof FileNode;
			} else if (fCurrentGroupMode instanceof ProjectGroup) {
				return node instanceof ProjectNode;
			} else if (fCurrentGroupMode instanceof PackageGroup) {
				return node instanceof ConstructNode &&
				!(node.getParent() instanceof ConstructNode) ;
			}
		}

		return false;
	}
}