/*******************************************************************************
 * Copyright (C) 2005-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *******************************************************************************/

package com.adacore.gnatbench.ui.internal.browsing;

import java.util.LinkedList;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.text.BadLocationException;
import org.eclipse.search.ui.ISearchQuery;
import org.eclipse.search.ui.text.AbstractTextSearchResult;
import org.eclipse.search.ui.text.IEditorMatchAdapter;
import org.eclipse.search.ui.text.IFileMatchAdapter;
import org.eclipse.search.ui.text.Match;

import com.adacore.gnatbench.core.GNATbenchSession;
import com.adacore.gnatbench.core.internal.GNATbenchCorePlugin;
import com.adacore.gnatbench.core.internal.analyzer.AdaConstruct;
import com.adacore.gnatbench.core.internal.analyzer.AdaConstructFilter;
import com.adacore.gnatbench.core.internal.analyzer.AdaConstructPassTroughFilter;
import com.adacore.gnatbench.core.internal.analyzer.AdaSimpleConstruct;
import com.adacore.gnatbench.core.internal.analyzer.GeneralizedLocation;
import com.adacore.gnatbench.core.internal.analyzer.IAdaConstructFilterProvider;
import com.adacore.gnatbench.core.internal.browsing.ReferencesIterator;
import com.adacore.gnatbench.core.internal.projects.GNATProjectRegistry;
import com.adacore.gnatbench.library.LibraryMonitor;
import com.adacore.gnatbench.library.LibrarySemaphore;
import com.adacore.gnatbench.library.Language.Language_Category;
import com.adacore.gnatbench.library.Language.Tree.Category_Array;
import com.adacore.gnatbench.library.Language.Tree.Construct_Tree;
import com.adacore.gnatbench.library.Language.Tree.Construct_Tree_Iterator;
import com.adacore.gnatbench.library.Language.Tree.Position_Type;
import com.adacore.gnatbench.library.Language.Tree.Relative_Position;
import com.adacore.gnatbench.library.Language.Tree.Tree_Language;
import com.adacore.gnatbench.library.Language.Tree.Tree_Package;
import com.adacore.gnatbench.library.Language.Tree.Ada.Ada_Package;
import com.adacore.gnatbench.library.Language.Tree.Database.Structured_File;
import com.adacore.gnatbench.library.GNATCOLL.Filesystem.Filesystem_String;
import com.adacore.gnatbench.library.GNATCOLL.VFS.VFS_Package;

public class AdaSearchResult extends AbstractTextSearchResult implements
		IAdaConstructFilterProvider {

	private ISearchQuery fQuery;

	private String fLabel;

	private String fSearchString;

	private SearchNode fRoot = new SearchNode(null);

	private LinkedList <MatchTag> fMatches = new LinkedList <MatchTag> ();

	private Job fChangeStateJob;

	/**
	 * @param query
	 * @param label
	 *            The label that would be displayed on the view.
	 * @param searchString
	 *            The string on wich the search has been made.
	 */
	public AdaSearchResult(AdaQueryAllReferences query, String label,
			String searchString) {
		super();

		fQuery = query;
		fLabel = label;
		fSearchString = searchString;
	}

	public IEditorMatchAdapter getEditorMatchAdapter() {
		return null;
	}

	public IFileMatchAdapter getFileMatchAdapter() {
		return null;
	}

	public String getLabel() {
		return fLabel + " (" + fMatches.size() + " matches)";
	}

	public String getTooltip() {
		return null;
	}

	public ImageDescriptor getImageDescriptor() {
		return null;
	}

	public ISearchQuery getQuery() {
		return fQuery;
	}

	/**
	 * Set the internal structure based on a list of locations, and removed the
	 * previous one.
	 *
	 * @param results a list of AdaLocation.
	 */
	public void setResults(ReferencesIterator results,
			IProgressMonitor monitor) {
		monitor.beginTask("analyzing results", results.totalProgress());

		int currentProgress = 0;

		Tree_Language adaTreeLang;
		Category_Array nullCategoryArray;

		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try {
			adaTreeLang = Ada_Package.Ada_Tree_Lang();
			nullCategoryArray = Tree_Package.Null_Category_Array();
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}


		for (; !results.atEnd(); results.next()) {
			int newProgress = results.currentProgress();

			monitor.worked(newProgress - currentProgress);
			currentProgress = newProgress;

			if (monitor.isCanceled()) {
				break;
			}

			GeneralizedLocation location = results.current();

			if (location == null) {
				continue;
			}

			IProject project = location.getFile().getProject();
			GNATProjectRegistry registry = (GNATProjectRegistry) GNATbenchSession
					.getDefault().getOrLoadRegistry(project);

			libMonitor = LibrarySemaphore.startGPSWork();

			try {
				Structured_File file = registry.fConstructDatabase
						.Get_Or_Create(VFS_Package
								.Create(new Filesystem_String(location
										.getFile().getOSPath())), adaTreeLang);

				Construct_Tree tree = file.Get_Tree();

				Construct_Tree_Iterator iter = tree.Get_Iterator_At(
						Tree_Package.To_Location(location.getLine(),
								location.getColumn().getColumnIndex()),
						Position_Type.Start_Construct,
						Relative_Position.Enclosing, nullCategoryArray);

				int offset = 0;

				try {
					offset = location.getOffset();
				} catch (BadLocationException e) {
					//  In this case, the location from the ALI file is not
					//  up to date - just forget the error silently and
					//  continue to others

					continue;
				}

				ConstructNode node = getOrCreateConstructNode(iter, tree,
						location);

				MatchTag match = new MatchTag(location, getSearchString(), offset,
						node);

				node.addMatch(match);

				addMatch(new Match(node, offset, location.getLength()));

				match.show();

				synchronized (this) {
					fMatches.add(match);
				}
			} finally {
				LibrarySemaphore.stopGPSWork(libMonitor);
			}
		}

		monitor.done();

		results.destroy();
	}

	/**
	 * @param construct
	 * @return
	 */
	private ConstructNode getOrCreateConstructNode(Construct_Tree_Iterator it,
			Construct_Tree tree, GeneralizedLocation loc) {

		while (!it.equals(Tree_Package.Null_Construct_Tree_Iterator())
				&& it.Get_Construct().Category() >= Language_Category.Cat_Loop_Statement) {

			it = tree.Get_Parent_Scope(it);
		}

		Construct_Tree_Iterator parent = tree.Get_Parent_Scope(it);

		ConstructNode node;

		AdaSimpleConstruct newConstruct = new AdaSimpleConstruct(it
				.Get_Construct(), loc.getFile());

		if (!parent.equals(Tree_Package.Null_Construct_Tree_Iterator())) {
			ConstructNode parentNode;
			parentNode = getOrCreateConstructNode(parent, tree, loc);
			node = parentNode.getConstructChild(newConstruct, this);

			if (node == null) {
				node = new ConstructNode(parentNode, newConstruct);
			}
		} else {
			FileNode fileNode = getOrCreateFileNode(loc);

			node = fileNode.getConstructChild(newConstruct, this);
			if (node == null) {
				node = new ConstructNode(fileNode, newConstruct);
			}
		}

		return node;
	}

	/**
	 * clear all the results contained in this instance, and remove every
	 * annotation from the editor.
	 */
	public void clear() {
		fRoot.dispose();
		fMatches.clear();
	}

	/**
	 * Returns the string on wich the search has been launched.
	 *
	 * @return
	 */
	public String getSearchString() {
		return fSearchString;
	}

	/**
	 * Returns the file pointed by the path, assuming that the last item is a
	 * file name. Missing folder classes are created.
	 *
	 * @param path
	 * @return
	 */
	private FileNode getOrCreateFileNode(GeneralizedLocation location) {
		String filePath = location.getFile().getEclipsePath();
		IProject baseProject = location.getFile().getProject();

		String[] path = filePath.split("\\\\|/");
		LinkedList <SearchNode> currentFolders = fRoot.getChildren();
		SearchNode lastNode = fRoot;
		boolean found;

		// This loop analyzes all the elements of path.
		for (int currentFolderId = 0; currentFolderId < path.length - 1; currentFolderId++) {

			if (path[currentFolderId].equals(""))
				continue;

			found = false;

			// Search in the folders the current folder
			for (SearchNode next : currentFolders) {

				if (next instanceof FolderNode) {
					FolderNode folder = (FolderNode) next;

					if (folder.getName().equals(path[currentFolderId])) {
						lastNode = folder;
						found = true;
						break;
					}
				} else if (next instanceof ProjectNode) {
					if (baseProject != null) {
						ProjectNode project = (ProjectNode) next;
						if (project.getName().equals(baseProject.getName())) {
							lastNode = project;
							found = true;
							baseProject = null;
							break;
						}
					}
				}
			}

			if (!found) {
				if (baseProject != null) {
					lastNode = new ProjectNode(baseProject.getName(), lastNode);
					baseProject = null;
				} else {
					lastNode = new FolderNode(path[currentFolderId], lastNode);
				}
			}

			currentFolders = lastNode.getChildren();
		}

		for (SearchNode element : lastNode.getChildren()) {
			if (element instanceof FileNode) {
				if (((FileNode) element).getName()
						.equals(path[path.length - 1])) {
					return (FileNode) element;
				}
			}
		}

		FileNode file = new FileNode(location.getFile(), lastNode);

		return file;
	}

	public void removeMatch(Match match) {
		if (match.getElement() instanceof SearchNode) {
			SearchNode node = ((SearchNode) match.getElement());

			for (MatchTag element : node.getMatches()) {
				fMatches.remove(element);
			}

			node.dispose();
		}

		super.removeMatch(match);
	}

	public void removeMatches(Match[] matches) {
		for (int i = 0; i < matches.length; i++) {
			if (matches[i].getElement() instanceof SearchNode) {
				SearchNode node = ((SearchNode) matches[i].getElement());

				for (MatchTag element : node.getMatches()) {
					fMatches.remove(element);
				}

				node.dispose();
			}
		}

		super.removeMatches(matches);
	}

	/**
	 * Shows makers and annotations for all the entities contained in the
	 * result.
	 *
	 */
	public synchronized void show() {
		final Job previousJob = fChangeStateJob;

		fChangeStateJob = new Job ("Show matches") {

			protected IStatus run(IProgressMonitor monitor) {
				if (previousJob != null) {
					try {
						previousJob.join();
					} catch (InterruptedException e) {
						GNATbenchCorePlugin.getDefault().logError(null, e);
					}
				}

				for (MatchTag element : fMatches) {
					element.show();
				}

				return new Status(IStatus.OK, GNATbenchCorePlugin.getId(),
						IStatus.OK, "", null);
			}};

		fChangeStateJob.schedule();
	}

	/**
	 * Hides markers and annotations for all entities contained in the ressult.
	 *
	 */
	public synchronized void hide() {
		final Job previousJob = fChangeStateJob;

		fChangeStateJob = new Job ("Hide matches") {

			protected IStatus run(IProgressMonitor monitor) {
				if (previousJob != null) {
					try {
						previousJob.join();
					} catch (InterruptedException e) {
						GNATbenchCorePlugin.getDefault().logError(null, e);
					}
				}

				for (MatchTag element : fMatches) {
					element.hide();
				}

				return new Status(IStatus.OK, GNATbenchCorePlugin.getId(),
						IStatus.OK, "", null);
			}};

			fChangeStateJob.schedule();
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see com.adacore.gnatbench.core.analyzer.IAdaConstructFilterProvider#getFilter()
	 */
	public AdaConstructFilter getFilter() {
		return new AdaConstructPassTroughFilter() {

			public boolean simpleFilter(AdaConstruct construct) {
				int category = construct.getCategory();

				return (category >= Language_Category.Cat_Package && category <= Language_Category.Cat_Local_Variable)
						|| category == Language_Category.Cat_With
						|| category == Language_Category.Cat_Use
						|| category == Language_Category.Cat_Field;
			}
		};
	}

	/**
	 * Returns all matches in trees, using projects node for roots whenever
	 * it's possible, otherwise the deepest node.
	 *
	 * @return
	 */
	public synchronized LinkedList <SearchNode>getProjectNodes() {
		LinkedList <SearchNode>result = new LinkedList<SearchNode>();

		for (MatchTag element : fMatches) {
			SearchNode current = element.getNode();

			while (current.getParent() != null
					&& current.getParent() != fRoot) {
				current = current.getParent();
			}

			if (!result.contains(current)) {
				result.add(current);
			}
		}

		return result;
	}

	/**
	 * Returns all matches in trees, using files node for roots whenever
	 * it's possible, otherwise the deepest node.
	 *
	 * @return
	 */
	public synchronized LinkedList <SearchNode>getFileNodes() {
		LinkedList <SearchNode>result = new LinkedList<SearchNode>();

		for (MatchTag element : fMatches) {
			SearchNode current = element.getNode();

			while (current.getParent() != null
					&& !(current instanceof FileNode)) {

				current = current.getParent();
			}

			if (!result.contains(current)) {
				result.add(current);
			}
		}

		return result;
	}

	/**
	 * Returns all matches in trees, using package node for roots whenever
	 * it's possible, otherwise the deepest node.
	 *
	 * @return
	 */
	public synchronized LinkedList <SearchNode> getPackageNodes() {
		LinkedList <SearchNode> result = new LinkedList<SearchNode>();

		for (MatchTag element : fMatches) {
			SearchNode current = element.getNode();

			while (current.getParent() != null
					&& current.getParent() instanceof ConstructNode) {

				current = current.getParent();
			}

			if (!result.contains(current)) {
				result.add(current);
			}
		}

		return result;
	}

	/**
	 * Returns all nodes containing a match.
	 *
	 * @return
	 */
	public synchronized LinkedList <SearchNode> getMatchingNodes() {
		LinkedList <SearchNode> result = new LinkedList<SearchNode>();

		for (MatchTag element : fMatches) {
			SearchNode current = element.getNode();

			if (!result.contains(current)) {
				result.add(current);
			}
		}

		return result;

	}

}