/*******************************************************************************
 * Copyright (C) 2005-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *******************************************************************************/

package com.adacore.gnatbench.ui.internal.browsing;

import java.io.IOException;
import java.util.LinkedList;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IMenuCreator;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.viewers.ColumnLayoutData;
import org.eclipse.jface.viewers.ColumnPixelData;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.DecoratingLabelProvider;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ILabelProviderListener;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.part.ViewPart;

import com.adacore.gnatbench.core.internal.GNATbenchCorePlugin;
import com.adacore.gnatbench.core.internal.adaeditor.AdaDocumentBuffer;
import com.adacore.gnatbench.core.internal.analyzer.AdaConstruct;
import com.adacore.gnatbench.core.internal.analyzer.AdaConstructFilter;
import com.adacore.gnatbench.core.internal.analyzer.AdaConstructPassTroughFilter;
import com.adacore.gnatbench.core.internal.analyzer.AdaEntityReference;
import com.adacore.gnatbench.core.internal.analyzer.GeneralizedLocation;
import com.adacore.gnatbench.core.internal.analyzer.IAdaConstructFilterProvider;
import com.adacore.gnatbench.core.internal.browsing.AdaCodeBrowser;
import com.adacore.gnatbench.core.internal.gpswrappers.Entity;
import com.adacore.gnatbench.ui.internal.GNATbenchUIPlugin;
import com.adacore.gnatbench.ui.internal.adaeditor.AdaPairMatcher;
import com.adacore.gnatbench.library.Language.Language_Category;

import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.window.Window;

/**
 * This class represents the call tree view.
 */
public class AdaCallTree extends ViewPart implements IAdaConstructFilterProvider {

	private TreeViewer fTreeViewer = null;
	private TableViewer fTableViewer = null;

	static final private int CALLING_MODE = 0;
	static final private int CALLED_BY_MODE = 1;

	private int fMode = CALLED_BY_MODE;

	private static String MATCH_CALL_IMAGE = "match_call_image";

	private ContentProvider fContentProvider;

	private AdaCodeBrowser fCodeBrowser;

	private AdaConstruct fCurrentConstruct;

	/**
	 * LinkedList of AdaConstruct at the root of seraches.
	 */
	private LinkedList<AdaConstruct> fHistory = new LinkedList<AdaConstruct>();

	private class InvalidEntity extends Exception {

		/**
		 *
		 */
		private static final long serialVersionUID = 4239266384712924998L;

	}

	/**
	 * Each item in the call tree will be stored by an instance of this class.
	 * Multiples matches are also stored here>
	 */
	private class CallNode {
		private LinkedList <CallNode> fChildren = null;
		private int fMode;

		private Entity fBrowsedEntity;

		private AdaConstruct fBrowsedConstruct;

		public LinkedList<AdaEntityReference> fReferences = new LinkedList<AdaEntityReference>();

		public CallNode(int mode, AdaEntityReference reference)
				throws InvalidEntity {
			fMode = mode;

			if (mode == CALLING_MODE) {
				fBrowsedEntity = new Entity (reference);
			} else if (mode == CALLED_BY_MODE) {
				fBrowsedEntity = getEnclosingEntity(reference);
			}

			if (fBrowsedEntity.isValid()) {
				fBrowsedConstruct = GNATbenchCorePlugin.getDefault()
				.getDocumentBuffer(
						fBrowsedEntity.getSpecOrElseBody().getFile())
						.getAnalyzer()
						.getConstructAt(
								fBrowsedEntity.getSpecOrElseBody(),
								AdaCallTree.this);

				fReferences.add (reference);
			} else {
				throw new InvalidEntity ();
			}
		}

		/**
		 * Returns a reference to the entity that encloses the reference given
		 * in parameter.
		 *
		 * @param reference
		 * @return
		 * @throws GPSError
		 * @throws PythonError
		 */
		private Entity getEnclosingEntity(AdaEntityReference reference) {
			AdaConstruct construct = GNATbenchCorePlugin.getDefault()
					.getDocumentBuffer(
							reference.getLocation().getFile()).getAnalyzer()
					.getConstructAt(reference.getLocation(), AdaCallTree.this);

			return new Entity(
					reference.getLocation().getFile().getProject(),
					construct);
		}

		/**
		 * Returns a list of CallNode containing the children of the node,
		 * according to the mode given earlier to the constructor. Computation
		 * are needed at each call, so this function is slow.
		 *
		 * @return
		 * @throws GPSError
		 */
		public Object[] getChildren() {
			if (fChildren != null) {
				return fChildren.toArray();
			} else {
				if (fBrowsedConstruct.getCategory()
						== Language_Category.Cat_Package) {
					return new Object [] {};
				}

				new Job("Analyze children"){

					protected IStatus run(IProgressMonitor monitor) {
						fChildren = internalGetChildren(monitor);

						Display.getDefault ().syncExec(new Runnable () {

							public void run() {
								fTreeViewer.refresh(CallNode.this);
							}});

						return new Status(IStatus.OK,
							GNATbenchUIPlugin.getId(), IStatus.OK, "", null);
					}}.schedule();

				return new Object [] {new String ("...")};
			}
		}

		/**
		 * Process the children of a given node. This primitive is potentially
		 * long running, so it should be called within a separate thread.
		 *
		 * @param monitor
		 * @return
		 */
		private LinkedList <CallNode> internalGetChildren(IProgressMonitor monitor) {
			LinkedList <CallNode> result = new LinkedList<CallNode>() ;

			LinkedList <LinkedList<AdaEntityReference>> references = null;

			AdaEntityReference baseReference = null;

			GeneralizedLocation entityLocation = fBrowsedEntity
					.getBodyOrElseSpec();

			try {
				baseReference = GNATbenchCorePlugin.getDefault()
						.getDocumentBuffer(entityLocation.getFile()).getEntity(
								entityLocation);
			} catch (BadLocationException e1) {
				GNATbenchCorePlugin.getDefault().logError(null, e1);
			}

			switch (fMode) {
			case CALLING_MODE:
				references = AdaCodeBrowser.getCalls(baseReference, false, monitor);
				break;
			case CALLED_BY_MODE:
				references = AdaCodeBrowser.getCalls(baseReference, true, monitor);
				break;
			}

			for (LinkedList <AdaEntityReference> list : references) {
				CallNode node = null;

				//  Add each element in the proper node, create one if
				//  needed.

				for (AdaEntityReference ref : list) {
					try {
						if (node == null) {

							node = new CallNode(fMode, ref);

							//  If there is a problem when the node is created,
							//  we want to be able to give other result, with
							//  an error notice. It's better than having just
							//  the first results work

							//  TODO: Does this still make sense with the new
							//  library ?
							//  result.add (new ErrorNode (ref));
						} else {
							node.fReferences.add(ref);
						}
					} catch (InvalidEntity e) {
						//  We don't do anything in that case - it most
						//  probably mean that the reference is out of date. We
						//  may want to trace something though...
					}
				}

				if (node != null) {
					result.add (node);
				}
			}

			return result;
		}

		/**
		 * Returns True if the entity contained in the call node is the same as
		 * the one given in parameter.
		 *
		 * @param entityReference
		 * @return
		 * @throws GPSError
		 */
		public boolean correspondTo(Entity entityReference) {

			if (entityReference.getBody () != null) {
				if (fBrowsedEntity.getBody() == null) return false;

				return entityReference.getBody().equals(
						fBrowsedEntity.getBody());
			} else if (entityReference.getSpec() != null) {
				if (fBrowsedEntity.getSpec() == null) return false;

				return entityReference.getSpec().equals(
						fBrowsedEntity.getSpec());
			}

			return false;
		}

		/**
		 * Returns the entity represented by this node.
		 *
		 * @return
		 */
		public Entity getBrowsedEntity () {
			return fBrowsedEntity;
		}

		/**
		 * Returns the construct represented by this node.
		 *
		 * @return
		 */
		public AdaConstruct getBrowsedConstruct() {
			return fBrowsedConstruct;
		}

		/**
		 * Frees the entity related to this node, and its children.
		 *
		 * @throws GPSError
		 *
		 */
		public void freeEntities() {
			//  fBrowsedEntity.free();
			freeChildrenEntities();
		}

		/**
		 * Frees the entities related to the children of this node.
		 *
		 * @throws GPSError
		 */
		private void freeChildrenEntities() {
			if (fChildren != null) {
				for (CallNode element : fChildren) {
					((CallNode) element).freeEntities();
				}

				fChildren.clear();
			}
		}

		public String getText() {
			if (fBrowsedConstruct != null) {
				return fBrowsedConstruct.toString();
			} else {
				return fBrowsedEntity.getName();
			}
		}

		public Image getImage() {
			if (fBrowsedConstruct != null) {
				return fBrowsedConstruct.getImage();
			} else {
				return null;
			}
		}

		public void jumpInto() {
			try {
				if (fBrowsedConstruct != null) {
					GNATbenchUIPlugin.getDefault().openEditor(
							fBrowsedConstruct.getLocation());
				} else {
					GNATbenchUIPlugin.getDefault().openEditor(
							fBrowsedEntity.getBodyOrElseSpec());
				}
			} catch (PartInitException e) {
				GNATbenchCorePlugin.getDefault().logError(null, e);
			}
		}
	}

	/**
	 * This class is used when it's impossible to get the proper entity for
	 * whatever reason. It should be used as less as possible
	 *
	 * @author ochem
	 *
	 */
	private class ErrorNode {

		private AdaEntityReference fReference;

		public ErrorNode (AdaEntityReference ref) {
			fReference = ref;
		}

		public String getLabel () {
			return fReference.getLocation().toString();
		}

		public void jumpInto () {
			try {
				GNATbenchUIPlugin.getDefault().openEditor(
						fReference.getLocation());
			} catch (PartInitException e) {
				GNATbenchCorePlugin.getDefault().logError(null, e);
			}
		}
	}

	private class ContentProvider implements ITreeContentProvider {

		private CallNode fRootNode;

		private int fLastMode = 0;
		private Object fLastInput = null;

		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.ITreeContentProvider#getChildren(java.lang.Object)
		 */
		public Object[] getChildren(Object parentElement) {
			if (parentElement instanceof CallNode) {
				return ((CallNode) parentElement).getChildren();
			}

			return new Object[]{};
		}

		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.ITreeContentProvider#getParent(java.lang.Object)
		 */
		public Object getParent(Object element) {

			return null;
		}

		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.ITreeContentProvider#hasChildren(java.lang.Object)
		 */
		public boolean hasChildren(Object element) {
			if (element instanceof CallNode) {
				return true;
			} else {
				return false;
			}
		}

		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.IStructuredContentProvider#getElements(java.lang.Object)
		 */
		public Object[] getElements(Object inputElement) {

			if (fLastInput == inputElement && fLastMode == fMode) {
				return new Object[] {fRootNode};
			}

			fLastInput = inputElement;
			fLastMode = fMode;

			if (fRootNode != null) {
				fRootNode.freeEntities();
			}

			if (inputElement != null) {
				try {
					fRootNode = new CallNode(fMode,
							(AdaEntityReference) inputElement);

					return new Object[] {fRootNode};
				} catch (InvalidEntity e) {
					// The entity is invalid, there's nothing we can do here...
					return new Object [] {};
				}

			}


			fRootNode = null;
			return new Object [] {};
		}

		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.IContentProvider#dispose()
		 */
		public void dispose() {

		}

		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.IContentProvider#inputChanged(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
		 */
		public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {

		}

		/**
		 * Frees the entities created during the browsing.
		 *
		 * @throws GPSError
		 */
		public void freeEntities() {
			if (fRootNode != null) {
				fRootNode.freeEntities();
			}
		}

	}

	private class LabelProvider implements ILabelProvider {

		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.ILabelProvider#getImage(java.lang.Object)
		 */
		public Image getImage(Object element) {
			if (element instanceof CallNode) {
				CallNode callNode = (CallNode) element;

				return callNode.getImage();
			} else if (element instanceof ErrorNode) {
				return GNATbenchUIPlugin.getDefault().getImage(
						GNATbenchUIPlugin.GENERIC_ERROR_ICON);
			}

			return null;
		}

		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.ILabelProvider#getText(java.lang.Object)
		 */
		public String getText(Object element) {
			if (element instanceof CallNode) {
				CallNode callNode = (CallNode) element;

				return callNode.getText();
			} else if (element instanceof ErrorNode) {
				return ((ErrorNode) element).getLabel();
			}

			return element.toString();
		}

		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.IBaseLabelProvider#addListener(org.eclipse.jface.viewers.ILabelProviderListener)
		 */
		public void addListener(ILabelProviderListener listener) {

		}

		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.IBaseLabelProvider#dispose()
		 */
		public void dispose() {

		}

		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.IBaseLabelProvider#isLabelProperty(java.lang.Object, java.lang.String)
		 */
		public boolean isLabelProperty(Object element, String property) {
			return false;
		}

		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.IBaseLabelProvider#removeListener(org.eclipse.jface.viewers.ILabelProviderListener)
		 */
		public void removeListener(ILabelProviderListener listener) {

		}

	}

	/**
	 * Switches the mode of the three to callee mode, and resets the view.
	 *
	 * @author Quentin Ochem
	 */
	private class ShowCallerHierarchy extends Action {
		public ShowCallerHierarchy() {
			super ("Show Caller Hierarchy", AS_RADIO_BUTTON);
			setChecked(true);
		}

		public void run () {
			fMode = CALLED_BY_MODE;
			fTreeViewer.refresh();
			fTreeViewer.expandToLevel(2);
		}

		public ImageDescriptor getImageDescriptor() {
			return GNATbenchUIPlugin.getDefault().getImageDescriptor("/img/",
					"ch_callers.gif");
		}

		public String getToolTipText () {
			return "Show Caller Hierarchy";
		}
	}

	/**
	 * Switches the mode of the three to caller mode, and resets the view.
	 *
	 * @author Quentin Ochem
	 */
	private class ShowCalleeHierarchy extends Action {
		public ShowCalleeHierarchy() {
			super ("Show Callee Hierarchy", AS_RADIO_BUTTON);
		}

		public void run () {
			fMode = CALLING_MODE;
			fTreeViewer.refresh();
			fTreeViewer.expandToLevel(2);
		}

		public ImageDescriptor getImageDescriptor() {
			return GNATbenchUIPlugin.getDefault().getImageDescriptor("/img/",
					"ch_callees.gif");
		}

		public String getToolTipText () {
			return "Show Callee Hierarchy";
		}
	}

	private class CallHistory implements IMenuCreator {

		public void dispose() {

		}

		public Menu getMenu(Control parent) {
			Menu m = new Menu (parent);

			for (AdaConstruct element : fHistory) {
				final AdaConstruct theElement = element;

				MenuItem mi = new MenuItem (m, SWT.CHECK);

				mi.setText(element.getName() + element.getProfile());
				mi.setImage(element.getImage());
				mi.addSelectionListener(new SelectionListener () {

					public void widgetSelected(SelectionEvent e) {
						setRootConstruct(theElement);

					}

					public void widgetDefaultSelected(SelectionEvent e) {

					}});

				if (fCurrentConstruct == element) {
					mi.setSelection(true);
				}
				else {
					mi.setSelection(false);
				}
			}


			return m;
		}

		public Menu getMenu(Menu parent) {

			return null;
		}

	}

	private class ShowHistory extends Action {
		public ShowHistory (Control parent) {
			super ("Show History List", AS_DROP_DOWN_MENU);

			setMenuCreator (new CallHistory ());
		}

		public ImageDescriptor getImageDescriptor() {
			return GNATbenchUIPlugin.getDefault().getImageDescriptor("/img/",
					"history_list.gif");
		}

		public String getToolTipText () {
			return "Show History List";
		}

	    public void run() {
	    	AdaCallTreeHistoryDialog dlg = new AdaCallTreeHistoryDialog(
					AdaCallTree.this.getSite().getShell(), AdaCallTree.this);

			if (dlg.open() == Window.OK) {
				LinkedList <AdaConstruct>removed = dlg.getRemoved();

				for (AdaConstruct element : removed) {
					fHistory.remove(element);
				}
			}
	    }
	}
	public AdaCallTree () throws IOException {
		if (GNATbenchUIPlugin.getDefault().getImage(MATCH_CALL_IMAGE) == null) {
			GNATbenchUIPlugin.getDefault().addImage(MATCH_CALL_IMAGE, "/img/",
					"occ_match.gif");
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.WorkbenchPart#createPartControl(org.eclipse.swt.widgets.Composite)
	 */
	public void createPartControl(Composite parent) {
		FillLayout layout = new FillLayout();
		layout.type = SWT.VERTICAL;

		parent.setLayout(layout);

		fTreeViewer = new TreeViewer(parent, SWT.MULTI | SWT.H_SCROLL
				| SWT.V_SCROLL);

		fContentProvider = new ContentProvider();

		fTreeViewer.setContentProvider(fContentProvider);
		fTreeViewer.setLabelProvider(new DecoratingLabelProvider(
				new LabelProvider(), GNATbenchUIPlugin.getDefault()
						.getWorkbench().getDecoratorManager()
						.getLabelDecorator()));

		IActionBars actionBar = this.getViewSite().getActionBars();

		actionBar.getToolBarManager().add(new ShowCallerHierarchy());
		actionBar.getToolBarManager().add(new ShowCalleeHierarchy());
		actionBar.getToolBarManager().add(new ShowHistory(parent));

		fTreeViewer.addDoubleClickListener(new IDoubleClickListener() {

			public void doubleClick(DoubleClickEvent event) {
				StructuredSelection selection = (StructuredSelection) event
						.getSelection();

				if (selection.getFirstElement() instanceof CallNode) {
					CallNode node = (CallNode) selection.getFirstElement();

					node.jumpInto();
				} else if (selection.getFirstElement() instanceof ErrorNode) {
					ErrorNode node = (ErrorNode) selection.getFirstElement();

					node.jumpInto();
				}
			}});

		fTreeViewer.addSelectionChangedListener(new ISelectionChangedListener() {

			public void selectionChanged(SelectionChangedEvent event) {
				fTableViewer.getTable().removeAll();

				StructuredSelection selection = (StructuredSelection) event
								.getSelection();

				if (!(selection.getFirstElement() instanceof CallNode)) return;

				CallNode node = (CallNode) ((StructuredSelection) event
						.getSelection()).getFirstElement();

				if (node == null) return;

				for (AdaEntityReference ref : node.fReferences) {
					TableItem item = new TableItem(fTableViewer.getTable(), SWT.NONE);
					item.setImage(GNATbenchUIPlugin.getDefault()
									.getImage(MATCH_CALL_IMAGE));
							item
									.setText(1, ref.getLocation().getLine()
											+ "");

					AdaDocumentBuffer buffer = GNATbenchCorePlugin
									.getDefault().getDocumentBuffer(
											ref.getLocation().getFile());

					int lastOffset = ref.getOffset() + ref.getName().length();

					lastOffset = buffer.getFirstNonWhite(lastOffset);

					if (buffer.getContent().charAt(lastOffset) == '(') {
						try {
							lastOffset = AdaPairMatcher.getCharOffset(
											buffer.getDocument(), ')', '(',
											lastOffset + 1, 1);
						} catch (BadLocationException e) {
							GNATbenchCorePlugin.getDefault().logError(
											null, e);
						} catch (Exception e) {
							GNATbenchCorePlugin.getDefault().logError(
											null, e);
						}
					}

					if (buffer.getContent().charAt(lastOffset) == ')') lastOffset++;

					item.setText(2, buffer.getContent().substring(
									ref.getOffset(), lastOffset)
									.replaceAll("[\n\r ]+", " "));
					item.setData(ref);
				}
			}});

		fTableViewer = new TableViewer (parent, SWT.MULTI | SWT.H_SCROLL
				| SWT.V_SCROLL | SWT.FILL | SWT.FULL_SELECTION);
		createColumns();

		fTableViewer.addSelectionChangedListener(new ISelectionChangedListener() {

			public void selectionChanged(SelectionChangedEvent event) {
				AdaEntityReference ref = (AdaEntityReference) ((StructuredSelection) event
								.getSelection()).getFirstElement();

				try {
					GNATbenchUIPlugin.getDefault().openEditor(
									ref.getLocation());
				} catch (PartInitException e) {
					GNATbenchCorePlugin.getDefault().logError(null, e);
				}

				fTableViewer.getControl().forceFocus();

			}});
	}

	/**
	 * Creates the columns on the table viewer.
	 */
    private void createColumns() {
        TableLayout layout = new TableLayout();
        fTableViewer.getTable().setLayout(layout);
        fTableViewer.getTable().setHeaderVisible(true);

        String [] columnHeaders = new String [] {"", "Line", "Call"};

        ColumnLayoutData columnLayouts[] = {
                new ColumnPixelData(19, false),
                new ColumnWeightData(60),
                new ColumnWeightData(300)};

        for (int i = 0; i < columnHeaders.length; i++) {
            layout.addColumnData(columnLayouts[i]);
            TableColumn tc = new TableColumn(fTableViewer.getTable(), SWT.NONE,
					i);
            tc.setResizable(columnLayouts[i].resizable);
            tc.setText(columnHeaders[i]);
        }
    }

	/* (non-Javadoc)
	 * @see org.eclipse.ui.part.WorkbenchPart#setFocus()
	 */
	public void setFocus() {
		// TODO Auto-generated method stub
	}

	/**
	 * Sets the root entity of the view, reset the view and expands the first
	 * level.
	 *
	 * @param rootEntity
	 */
	public void setRootEntity (AdaEntityReference rootEntity) {
		Entity realEntity = null;

		GeneralizedLocation entityLocation = null;

		realEntity = new Entity(rootEntity);
		entityLocation = realEntity.getBodyOrElseSpec();

		if (entityLocation == null) {
			return;
		}

		AdaConstruct construct = GNATbenchCorePlugin.getDefault()
				.getDocumentBuffer(
						entityLocation.getFile())
				.getAnalyzer().getConstructAt(entityLocation, this);

		fHistory.add(construct);

		setRootConstruct (construct);
	}

	/**
	 * Sets the root entity of the view, reset the view and expands the first
	 * level.
	 *
	 * @param rootEntity
	 */
	public void setRootConstruct (AdaConstruct construct) {
		if (fCodeBrowser == null) {
			fCodeBrowser = new AdaCodeBrowser();
		}

		GeneralizedLocation entityLocation = construct.getLocation();

		if (construct != null
				&& (construct.getCategory() < Language_Category.Cat_Task
						|| construct.getCategory() > Language_Category.Cat_Entry
						|| !construct.getLocation().equals(entityLocation))) {

			return;

		}

		try {
			fTreeViewer.setInput(GNATbenchCorePlugin.getDefault()
					.getDocumentBuffer(
							entityLocation.getFile())
					.getEntity(entityLocation));
		} catch (BadLocationException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		}

		fTreeViewer.refresh();
		fTreeViewer.expandToLevel(2);

		fCurrentConstruct = construct;
	}

	/* (non-Javadoc)
	 * @see com.adacore.gnatbench.core.analyzer.IAdaConstructFilterProvider#getFilter()
	 */
	public AdaConstructFilter getFilter() {
		return new AdaConstructPassTroughFilter () {

			public boolean simpleFilter(AdaConstruct construct) {
				// TODO: This filter has a limitation: it does not take into
				// account parameters that are subprograms.

				int category = construct.getCategory();

				return category >= Language_Category.Cat_Package
						&& category <= Language_Category.Cat_Entry;
			}};
	}

	public void dispose() {

	}

	public LinkedList <AdaConstruct> getHistory () {
		return fHistory;
	}

}
