/*****************************************************************************
 * Copyright (C) 2006-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *****************************************************************************/

package com.adacore.gnatbench.ui.internal.adaeditor;

import java.util.LinkedList;

import org.eclipse.core.filesystem.IFileStore;
import org.eclipse.core.filesystem.URIUtil;
import org.eclipse.jface.dialogs.InputDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.MouseListener;
import org.eclipse.swt.events.TraverseEvent;
import org.eclipse.swt.events.TraverseListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.List;
import org.eclipse.swt.widgets.Shell;

import com.adacore.gnatbench.core.GNATbenchCoreException;
import com.adacore.gnatbench.core.internal.GNATbenchCorePlugin;
import com.adacore.gnatbench.core.internal.projects.GNATProjectRegistry;
import com.adacore.gnatbench.library.LibraryMonitor;
import com.adacore.gnatbench.library.LibrarySemaphore;
import com.adacore.gnatbench.library.GNATCOLL.VFS.File_Array;
import com.adacore.gnatbench.library.GNATCOLL.VFS.VFS_Package;

/**
 * This view assist the user to open any file referenced by the GPR files of
 * the project, even external ones.
 */
public class OpenFileFromProject extends InputDialog {

	String [] fSources;
	List fFilesList;
	GNATProjectRegistry fRegistry;

	/**
	 * Reacts to the tab button, ask for completion instead of traversing.
	 */
	private class CompleteSourcesFile implements TraverseListener {

		public void keyTraversed(TraverseEvent e) {
			if (e.detail == SWT.TRAVERSE_TAB_NEXT
					|| e.detail == SWT.TRAVERSE_TAB_PREVIOUS) {
				e.doit = false;

				fFilesList.removeAll();

				String uncompleteName = getText().getText();

				String completion = null;

				for (int i = 0; i < fSources.length; ++i) {
					if (fSources [i].indexOf(uncompleteName) == 0) {
						fFilesList.add(fSources [i]);

						if (completion == null) {
							completion = fSources [i];
						} else {
							for (int j = 0; j < completion.length(); ++j) {
								if (j >= fSources [i].length()) {
									completion = completion.substring(0, j);

									break;
								}

								if (!(completion.charAt(j) == fSources[i]
										.charAt(j))) {
									completion = completion.substring(0, j);

									break;
								}
							}
						}
					}
				}

				getText().setText(completion);
				getText().setSelection(completion.length());
			}
		}

	}

	public OpenFileFromProject(GNATProjectRegistry registry, Shell parentShell) {
		super(parentShell, "Open File From GPR Projects",
				"Open a source file from a GPR", "", null);

		fRegistry = registry;
	}

	protected Control createDialogArea(Composite parent) {
		Composite composite = (Composite) super
				.createDialogArea(parent);

		fFilesList = new List(composite, SWT.SINGLE | SWT.FILL | SWT.V_SCROLL
				| SWT.BORDER);
		fFilesList
				.setLayoutData(new GridData(GridData.GRAB_HORIZONTAL
						| GridData.HORIZONTAL_ALIGN_FILL
						| GridData.VERTICAL_ALIGN_FILL));

		fFilesList.addMouseListener(new MouseListener (){

			public void mouseDoubleClick(MouseEvent e) {
				List sourceList = (List) e.getSource();

				if (sourceList.getSelection().length > 0) {
					getText().setText(sourceList.getSelection() [0]);
				}
			}

			public void mouseDown(MouseEvent e) {
				// TODO Auto-generated method stub

			}

			public void mouseUp(MouseEvent e) {
				// TODO Auto-generated method stub

			}});

		getText().addTraverseListener(new CompleteSourcesFile());

		for (int i = 0; i < 10; ++i) {
			fFilesList.add("");
		}

		LinkedList <IFileStore> prjSources;

		try {
			prjSources = fRegistry.getRootProject().getSourceFiles(true);
		} catch (GNATbenchCoreException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
			return composite;
		}

		LibraryMonitor libMonitor = LibrarySemaphore.startGPSWork();

		try {
			File_Array stdFiles = fRegistry.getProjectRegistry().Get_Predefined_Source_Files();

			fSources = new String [prjSources.size() + stdFiles.Length()];

			int srcInd = 0;

			for (IFileStore file : prjSources) {
				fSources[srcInd] = URIUtil.toPath(file.toURI()).toOSString();
				srcInd = srcInd + 1;
			}

			for (int i = stdFiles.First(); i <= stdFiles.Last(); ++i) {
				fSources[srcInd] = stdFiles.Get_Element_At(i)
				.Full_Name(false).toString();

				srcInd = srcInd + 1;
			}

			VFS_Package
					.Unchecked_Free((File_Array.Ref) stdFiles.NewProxyRef());
		} finally {
			LibrarySemaphore.stopGPSWork(libMonitor);
		}

		for (int i = 0; i < fSources.length; ++i) {
			int index = fSources[i].lastIndexOf('\\');

			if (index == -1) {
				index = fSources[i].lastIndexOf('/');
			}

			fSources [i] = fSources [i].substring(index + 1);
		}

		return composite;
	}

}
