/*******************************************************************************
 * Copyright (C) 2007-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *******************************************************************************/

package com.adacore.gnatbench.ui.internal.adaeditor;

import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.BadPartitioningException;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.IDocumentExtension3;
import org.eclipse.jface.text.IRegion;
import org.eclipse.jface.text.Region;
import org.eclipse.jface.text.source.ICharacterPairMatcher;

import com.adacore.gnatbench.core.internal.GNATbenchCorePlugin;

public class AdaPairMatcher implements ICharacterPairMatcher {

	public static class CharNotFound extends Exception {

		/**
		 *
		 */
		private static final long serialVersionUID = 1869190072604422939L;
	}

	int fAnchor;

	public void clear() {

	}

	public void dispose() {
	}

	public int getAnchor() {
		return fAnchor;
	}

	/**
	 * This function returns a the offset of a character.
	 *
	 * @param charSought the character sought
	 * @param charIncrement if the character is in a recursive definition
	 * (like {}, or () ), then charIncrement represents the character that
	 * opens a nested search
	 * @param offset the offset where to start the search
	 * @param step usually +1 or -1, meaning forward or backwards
	 * @return
	 * @throws BadLocationException
	 * @throws CharNotFound
	 */
	public static int getCharOffset(IDocument document, char charSought,
			char charIncrement, int offset, int step)
			throws BadLocationException, CharNotFound {
		IDocumentExtension3 extension = (IDocumentExtension3) document;
		String contentType = "";

		for (int k = offset; k >= 0 && k < document.getLength(); k += step) {
			try {
				contentType = extension.getContentType(
						GNATbenchCorePlugin.ADA_PARTITIONING, k, false);
			} catch (BadLocationException e1) {
				GNATbenchCorePlugin.getDefault().logError(null, e1);
				throw new CharNotFound();
			} catch (BadPartitioningException e1) {
				GNATbenchCorePlugin.getDefault().logError(null, e1);
				throw new CharNotFound();
			}

			if (contentType != IDocument.DEFAULT_CONTENT_TYPE) {
				continue;
			}

			char charAtOffset = document.getChar(k);

			if (charAtOffset == charSought) {
				return k;
			}

			if (charAtOffset == charIncrement) {
				k = getCharOffset(document, charSought,
						charIncrement, k + step, step);
			}
		}

		throw new CharNotFound();
	}

	public IRegion match(IDocument iDocument, int initialOffset) {
		int offset = initialOffset;

		int newOffsetOpen = -1, newOffsetClose = -1;

		if (offset >= iDocument.getLength()) {
			offset = iDocument.getLength() - 1;
		}

		char charAtOffset;

		try {
			charAtOffset = iDocument.getChar(offset);

			if (offset > 0) {
				offset--;
				charAtOffset = iDocument.getChar(offset);
			} else {
				return null;
			}
		} catch (BadLocationException e2) {
			return null;
		}

		String contentType = "";

		try {
			IDocumentExtension3 extension = (IDocumentExtension3) iDocument;
			contentType = extension.getContentType(
					GNATbenchCorePlugin.ADA_PARTITIONING, offset, false);
		} catch (BadLocationException e1) {
			return null;
		} catch (BadPartitioningException e1) {
			return null;
		}

		if (contentType != IDocument.DEFAULT_CONTENT_TYPE) {
			return null;
		}

		try {
			if (charAtOffset == '(') {
				newOffsetOpen = offset;
				newOffsetClose = getCharOffset(iDocument, ')', '(', offset + 1, 1);
				fAnchor = LEFT;
			} else if (charAtOffset == ')') {
				newOffsetClose = offset;
				newOffsetOpen = getCharOffset(iDocument, '(', ')', offset - 1, -1);
				fAnchor = RIGHT;
			} else {
				return null;
			}
		} catch (CharNotFound e) {
			return null;
		} catch (BadLocationException e) {
			return null;
		}

		return new Region (newOffsetOpen, newOffsetClose - newOffsetOpen + 1);
	}

}
