/*******************************************************************************
 * Copyright (c) 2007 AdaCore.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *******************************************************************************/

package com.adacore.gnatbench.ui.internal.adaeditor;

import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.IDocument;

import com.adacore.gnatbench.core.internal.analyzer.GeneralizedLocation;

public class AdaEditorTextUtils {

	private AdaEditor editor;
	private IDocument document;

	public int currentLine;
	public int currentColumn;


	public AdaEditorTextUtils(final AdaEditor editor) {
		this.editor = editor;
		document = editor.getDocumentBuffer().getDocument();
		
		GeneralizedLocation loc = editor.getCurrentLocation();
		
		currentColumn = loc.getColumn().getColumnIndex() - 1;
		currentLine = loc.getLine() - 1;
	} // ctor


	public String getLine (final int lineNumber) throws BadLocationException {
		final String text = editor.getDocumentBuffer().getContent();
		final int first = document.getLineOffset(lineNumber);
		int last = first;
		while ((last < text.length()) && (text.charAt(last) != '\n')) {
			++last;
		} // while
		if (last != text.length()) { // this is not the last line
			++last;
		} // if
		return text.substring(first, last);
	} // getLine


	public String getCurrentLine () throws BadLocationException {
		final String text = editor.getDocumentBuffer().getContent();
		final int first = document.getLineOffset(currentLine);
		int last = first;
		while ((last < text.length()) && (text.charAt(last) != '\n')) {
			++last;
		} // while
		if (last != first) { // this is not the last line
			++last;
		} // if
		return text.substring(first, last);
	} // getCurrentLine


	public void insertText (final String text) throws BadLocationException {
		final int offsetLineBegin = document.getLineOffset(currentLine);		
		document.replace(offsetLineBegin + currentColumn, 0, text);		
		// show the cursor at the new position
		currentColumn += text.length();
		editor.selectAndReveal(offsetLineBegin + currentColumn, 0);
	} // insertText

	
	public void insertTab () throws BadLocationException {	
		final int tabWidth = AdaEditorPrefUtils.tabWidth();		
		final int offsetLineBegin = document.getLineOffset(currentLine);	
		
		int k = offsetLineBegin;
		int logicalColumn = 0;
		while (logicalColumn < currentColumn) {
			if (document.getChar(k) == '\t') {
				int nextTabStop = ((logicalColumn + tabWidth) / tabWidth) * tabWidth;		
				logicalColumn = nextTabStop;
			} else {
				++logicalColumn;
			} // if
			++k;
		} // for	
		
		final int insertionColumn = k; 
		document.replace(insertionColumn, 0, "\t");

		// show the cursor at the new position
		editor.selectAndReveal(insertionColumn + 1, 0);

		final int finalTabStop = ((currentColumn + tabWidth) / tabWidth) * tabWidth;		
		currentColumn = finalTabStop;
	} // insertTab


	public void insertLine (final String line) throws BadLocationException {
		insertText ("\n" + line);
		++currentLine;
		currentColumn = line.length();
	} // insertLine


	public void replaceLine (String newLine) throws BadLocationException {
		final String text = editor.getDocumentBuffer().getContent();
		final int offsetLineBegin = document.getLineOffset(currentLine);
		int lineEnd = text.indexOf('\n', offsetLineBegin);
		if (lineEnd == -1) {
			// reached the end of the file
			lineEnd = text.length();
		} // if	
		document.replace(offsetLineBegin, lineEnd - offsetLineBegin, newLine);		
		currentColumn = newLine.length();
	} // replaceLine


	public void setCursorPosition (final int line, final int column) throws BadLocationException {
		final int offsetLineBegin = document.getLineOffset(line);
		editor.selectAndReveal(offsetLineBegin + column, 0);		
		currentColumn = column;
		currentLine = line;
	} // setCursorPosition 


	public void gotoEOL (final int line) throws BadLocationException {
		final String text = editor.getDocumentBuffer().getContent();
		final int offsetLineBegin = document.getLineOffset(line);
		int column = 0;
		while (offsetLineBegin + column < text.length() 
			   && text.charAt(offsetLineBegin + column) != '\n') 
		{
			++column;
		} // while
		editor.selectAndReveal(offsetLineBegin + column, 0);		
		currentColumn = column;
		currentLine = line;
	} // gotoEOL

	
	public void gotoCurrentEOL () throws BadLocationException {
		final String text = editor.getDocumentBuffer().getContent();
		final int offsetLineBegin = document.getLineOffset(currentLine);
		int column = 0;
		while (offsetLineBegin + column < text.length() 
			   && text.charAt(offsetLineBegin + column) != '\n') 
		{
			++column;
		} // while
		editor.selectAndReveal(offsetLineBegin + column, 0);		
		currentColumn = column;
	} // gotoCurrentEOL

	
} // AdaEditorTextUtils
