/*******************************************************************************
 * Copyright (C) 2005-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *******************************************************************************/

package com.adacore.gnatbench.ui.internal.adaeditor;

import org.eclipse.core.filesystem.URIUtil;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.DocumentEvent;
import org.eclipse.jface.text.FindReplaceDocumentAdapter;
import org.eclipse.jface.text.IDocumentListener;
import org.eclipse.jface.text.IRegion;
import org.eclipse.jface.text.ITextSelection;
import org.eclipse.jface.text.TextSelection;
import org.eclipse.jface.text.source.ISourceViewer;
import org.eclipse.jface.text.source.SourceViewer;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IPageLayout;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.part.IShowInSource;
import org.eclipse.ui.part.IShowInTargetList;
import org.eclipse.ui.part.ShowInContext;
import org.eclipse.ui.texteditor.ChainedPreferenceStore;
import org.eclipse.ui.texteditor.ContentAssistAction;
import org.eclipse.ui.texteditor.ITextEditorActionConstants;
import org.eclipse.ui.texteditor.ITextEditorActionDefinitionIds;
import org.eclipse.ui.texteditor.SourceViewerDecorationSupport;
import org.eclipse.ui.views.contentoutline.IContentOutlinePage;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.jface.text.source.IVerticalRuler;

import org.eclipse.jface.text.source.projection.ProjectionViewer;
import org.eclipse.jface.text.source.projection.ProjectionSupport;
import org.eclipse.jface.util.IPropertyChangeListener;
import org.eclipse.jface.util.PropertyChangeEvent;
import org.eclipse.jface.viewers.IPostSelectionProvider;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;

import com.adacore.gnatbench.library.Language.Language_Category;
import com.adacore.gnatbench.core.GNATbenchCoreException;
import com.adacore.gnatbench.core.internal.GNATbenchCorePlugin;
import com.adacore.gnatbench.core.internal.adaeditor.AbstractAdaEditor;
import com.adacore.gnatbench.core.internal.adaeditor.AdaEditorToolPreferencesProvider;
import com.adacore.gnatbench.core.internal.analyzer.AdaAnalyzer;
import com.adacore.gnatbench.core.internal.analyzer.AdaAnalyzerJob;
import com.adacore.gnatbench.core.internal.analyzer.AdaConstruct;
import com.adacore.gnatbench.core.internal.analyzer.AdaConstructFilter;
import com.adacore.gnatbench.core.internal.analyzer.AdaConstructPassTroughFilter;
import com.adacore.gnatbench.core.internal.analyzer.AdaEntityReference;
import com.adacore.gnatbench.core.internal.analyzer.GeneralizedLocation;
import com.adacore.gnatbench.core.internal.analyzer.IAdaConstructFilterProvider;
import com.adacore.gnatbench.core.internal.browsing.IAdaEntitySelectionProvider;
import com.adacore.gnatbench.core.internal.filesystem.FileStoreReference;
import com.adacore.gnatbench.core.analyzer.IAdaEntityReference.SearchScopes;
import com.adacore.gnatbench.ui.internal.GNATbenchUIPlugin;
import com.adacore.gnatbench.ui.internal.browsing.AdaActionOpenBody;
import com.adacore.gnatbench.ui.internal.browsing.AdaActionOpenCallHierarchy;
import com.adacore.gnatbench.ui.internal.browsing.AdaActionOpenDeclaration;
import com.adacore.gnatbench.ui.internal.browsing.AdaActionReferences;
import com.adacore.gnatbench.ui.internal.codingstyle.AdaActionFormat;
import com.adacore.gnatbench.ui.internal.codingstyle.AdaActionIndent;
import com.adacore.gnatbench.ui.internal.projectexplorer.GPRExplorerContentProvider;

/**
 * This class holds the main functionalities of the ada-aware editor.
 */
public class AdaEditor
	extends AbstractAdaEditor
	implements IPropertyChangeListener, IDocumentListener,
			IAdaEntitySelectionProvider, IShowInTargetList, IShowInSource
{

	private AdaContentOutlinePage fOutlinePage;
	private AdaBlocFolding fBlocFolding;
	private SourceViewer fViewer = null;
	private AdaAnalyzerJob fAnalyzerJob;
	private boolean fBlocFoldingEnabled = GNATbenchCorePlugin.getDefault()
			.getPreferenceStore().getBoolean(
					AdaEditorToolPreferencesProvider.PREF_TOOLS_BLOC_FOLDING);

	/**
	 * Creates a new AdaEditor.
	 */
	public AdaEditor() {
		super();
	}

	/**
	 * Parses the whole document in order to retrieve the semantic information.
	 */
	public void updateSemantics() {
		getDocumentBuffer().getAnalyzer().analyze();
	}

	/**
	 *
	 * @return The current semantic analyzer for the document.
	 */
	public AdaAnalyzer getAnalyzer() {
		return getDocumentBuffer().getAnalyzer();
	}

	/**
	 * This function is the concrete constructor of the ada editor.
	 */
	protected void initializeEditor() {
		super.initializeEditor();

		setDocumentProvider(GNATbenchCorePlugin.getDefault()
				.getDocumentProvider());

		PlatformUI.getWorkbench().getThemeManager().getCurrentTheme()
				.addPropertyChangeListener(this);
		fBlocFolding = new AdaBlocFolding();
		fOutlinePage = new AdaContentOutlinePage(this);

		setSourceViewerConfiguration(new AdaSourceViewerConfiguration(this));
		setRulerContextMenuId ("#AdaRulerContext");
		setPreferenceStore(new ChainedPreferenceStore(new IPreferenceStore[] {
				GNATbenchCorePlugin.getDefault().getPreferenceStore(),
				getPreferenceStore() }));
	}

	/**
	 * Creates the source viewer, and initialize some event-aware object on it.
	 */
	protected ISourceViewer createSourceViewer(Composite parent,
			IVerticalRuler ruler, int styles) {

		SourceViewer viewer;

		if (fBlocFoldingEnabled) {
			viewer = new ProjectionViewer(parent, ruler,
				getOverviewRuler(), isOverviewRulerVisible(), styles);
		} else {
			viewer = new SourceViewer(parent, ruler,
					getOverviewRuler(), isOverviewRulerVisible(), styles);
		}

		getOverviewRuler().addAnnotationType(
				GNATbenchUIPlugin.getId() + ".AdaLocationAnnotation");

		// ensure decoration support has been created and configured.
		getSourceViewerDecorationSupport(viewer);

		fViewer = viewer;

		return viewer;
	}

	/**
	 * Called when the highlighed range, e.g the range indicator, has to be
	 * updated.
	 */
	protected void adjustHighlightRange(int offset, int length) {
		super.adjustHighlightRange(offset, length);

		ISourceViewer viewer = getSourceViewer();
		viewer.setRangeIndication(offset, length, false);
	}

	/**
	 * This function wrapps the existing getSourceViewer, wich is a private
	 * function.
	 *
	 * @return
	 */
	public ISourceViewer getSourceViewerPublic() {
		return getSourceViewer();
	}


	@SuppressWarnings("unchecked")
	public Object getAdapter(Class required) {
		if (IContentOutlinePage.class.equals(required)) {
			if (fOutlinePage == null) {
				fOutlinePage = new AdaContentOutlinePage(this);
			}
			return fOutlinePage;
		}

		/*
		 * if (fProjectionSupport != null) { Object adapter=
		 * fProjectionSupport.getAdapter(getSourceViewer(), required); if
		 * (adapter != null) return adapter; }
		 */

		return super.getAdapter(required);
	}

	/**
	 * To be understood and documented.
	 */
	public void doSetInput(IEditorInput input) throws CoreException {
		super.doSetInput(input);

		fBlocFolding.setDocument(getDocumentProvider().getDocument(input));
		getDocument().addDocumentListener(this);
		fAnalyzerJob = new AdaAnalyzerJob("Compute semantics", getDocumentBuffer()
				.getAnalyzer());

		if (fOutlinePage != null && fOutlinePage.getViewer () != null) {
			fOutlinePage.getViewer().setInput(getAnalyzer());
		}
	}

	/**
	 * This function is used to refresh the text viewer whenever the preferences
	 * are changed.
	 */
	public void propertyChange(PropertyChangeEvent event) {
		if (event.getProperty().indexOf("com.adacore") != -1) {
			getSourceViewer().invalidateTextPresentation();
			getSourceViewer().resetVisibleRegion();
		}
	}

	public void refreshWidget() {
		getSourceViewer().getTextWidget().redraw();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.text.IDocumentListener#documentAboutToBeChanged(org.eclipse.jface.text.DocumentEvent)
	 */
	public void documentAboutToBeChanged(DocumentEvent event) {

	}

	/*
	 * @see org.eclipse.ui.IWorkbenchPart#createPartControl(org.eclipse.swt.widgets.Composite)
	 */
	public void createPartControl(Composite parent) {
		super.createPartControl(parent);

		if (fBlocFoldingEnabled) {
			ProjectionViewer viewer = (ProjectionViewer) getSourceViewer();

			ProjectionSupport projectionSupport = new ProjectionSupport(viewer,
					getAnnotationAccess(), getSharedColors());
			projectionSupport.install();

			// turn projection mode on
			viewer.doOperation(ProjectionViewer.TOGGLE);

			fBlocFolding.setAnnotationModel(viewer
					.getProjectionAnnotationModel());
			getDocumentBuffer().getAnalyzer().addAdaAnalyzerListener(
					fBlocFolding);
			fBlocFolding.setRoots(getDocumentBuffer().getAnalyzer().getRoots(
					fBlocFolding));
		}

		if (getSelectionProvider() instanceof IPostSelectionProvider) {
			((IPostSelectionProvider) getSelectionProvider())
					.addPostSelectionChangedListener(new RangeIndicator ());
		} else {
			getSelectionProvider().addSelectionChangedListener(new RangeIndicator ());
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.text.IDocumentListener#documentChanged(org.eclipse.jface.text.DocumentEvent)
	 */
	public void documentChanged(DocumentEvent event) {
		fAnalyzerJob.notifyChange();
	}

	/**
	 * Puts the caret and the text of the editor at the specified position.
	 *
	 * @param location
	 */
	public void focus(GeneralizedLocation location) {
		try {
			selectAndReveal(location.getOffset(), location.getLength());

			if (fOutlinePage != null) {
				fOutlinePage.focusEditor();
			}
		} catch (BadLocationException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		}
	}

	/**
	 * Returns the current location of the caret in the editor.
	 *
	 * @return
	 */
	public GeneralizedLocation getCurrentLocation() {
		TextSelection selection = (TextSelection) fViewer.getSelection();
		int offsetBegin = selection.getOffset();

		GeneralizedLocation location = GeneralizedLocation
		.fromAbsoluteLocation(getFile(),
				selection.getStartLine() + 1, offsetBegin);

		location.setLength(selection.getLength());

		return location;
	}

	/**
	 * @return The current entity pointed by the caret, null if none.
	 */
	public AdaEntityReference getEntitySelected() {
		TextSelection selection = (TextSelection) fViewer.getSelection();

		return getDocumentBuffer().getEntity(selection.getOffset());
	}

	public void doSave(IProgressMonitor monitor) {
		if (GNATbenchCorePlugin
				.getDefault()
				.getPreferenceStore()
				.getBoolean(
						AdaEditorPreferences.PREF_CODING_STYLE_REMOVE_SPACES)) {
			removeTrailingSpaces();
		}

		super.doSave(monitor);
	}

	public void doSaveAs(IProgressMonitor monitor) {
		if (GNATbenchCorePlugin
				.getDefault()
				.getPreferenceStore()
				.getBoolean(
						AdaEditorPreferences.PREF_CODING_STYLE_REMOVE_SPACES)) {
			removeTrailingSpaces();
		}

		super.doSaveAs();
	}

	private void removeTrailingSpaces() {
		FindReplaceDocumentAdapter findReplace = new FindReplaceDocumentAdapter(
				getDocument());

		try {
			IRegion region = null;
			int index = 0;

			while (index < getDocument().getLength()
					&& (region = findReplace.find(index, "[ \t]+$", true, false,
							false, true)) != null) {
				findReplace.replace("", false);
				index = region.getOffset();
			}
		} catch (BadLocationException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		}
	}

	protected void editorContextMenuAboutToShow(IMenuManager menu) {
		super.editorContextMenuAboutToShow(menu);

		MenuManager referencesManager = new MenuManager ("References");
		addAction(referencesManager, "ReferencesInProject");
		addAction(referencesManager, "ReferencesInHierarchy");
		addAction(referencesManager, "ReferencesInWorkspace");

		menu.appendToGroup(ITextEditorActionConstants.GROUP_SAVE, new Separator());
		addAction(menu, ITextEditorActionConstants.GROUP_SAVE, "OpenDeclaration");
		addAction(menu, ITextEditorActionConstants.GROUP_SAVE, "OpenBody");
		addAction(menu, ITextEditorActionConstants.GROUP_SAVE, "OpenCallHierarchy");
		menu.appendToGroup(ITextEditorActionConstants.GROUP_SAVE, new Separator());
		addAction(menu, ITextEditorActionConstants.GROUP_SAVE, "NextAdaUnit");
		addAction(menu, ITextEditorActionConstants.GROUP_SAVE, "PrevAdaUnit");

		addAction(menu, ITextEditorActionConstants.GROUP_EDIT, "ToggleComment");
		addAction(menu, ITextEditorActionConstants.GROUP_EDIT, "Refill");
		addAction(menu, ITextEditorActionConstants.GROUP_EDIT, "CorrectIndent");
		addAction(menu, ITextEditorActionConstants.GROUP_EDIT, "Format");
		menu.appendToGroup(ITextEditorActionConstants.GROUP_EDIT, new Separator());
		addAction(menu, ITextEditorActionConstants.GROUP_EDIT, "InsertEnd");
		addAction(menu, ITextEditorActionConstants.GROUP_EDIT, "GenerateBody");
		menu.appendToGroup(ITextEditorActionConstants.GROUP_EDIT, new Separator());
		menu.appendToGroup(ITextEditorActionConstants.GROUP_EDIT, referencesManager);

		menu.appendToGroup("additions", new Separator());
		menu.appendToGroup("additions", getAction("CompileFile"));
		menu.appendToGroup("additions", getAction("AnalyzeFile"));

	}

	protected void createActions() {
		super.createActions();

		Action action;

		action = new AdaActionOpenCallHierarchy(AdaEditorMessages
				.getResourceBundle(), "AdaOpenCallHierarchy.", this);
		action.setActionDefinitionId(GNATbenchUIPlugin.getId()
				+ ".AdaOpenCallHierarchy");
		setAction("OpenCallHierarchy", action);

		action = new AdaActionIndent(AdaEditorMessages.getResourceBundle(),
				"AdaIndent.", this);
		action.setActionDefinitionId(GNATbenchUIPlugin.getId()
				+ ".AdaCorrectIndent");
		setAction("CorrectIndent", action);

		action = new AdaActionFormat(AdaEditorMessages.getResourceBundle(),
				"AdaFormat.", this);
		action.setActionDefinitionId(GNATbenchUIPlugin.getId()
				+ ".AdaFormat");
		setAction("Format", action);

		action = new AdaActionToggleComment(AdaEditorMessages
				.getResourceBundle(), "AdaToggleComment.", this);
		action.setActionDefinitionId(GNATbenchUIPlugin.getId()
				+ ".AdaToggleComment");
		setAction("ToggleComment", action);

		action = new AdaActionTabIndent(AdaEditorMessages
				.getResourceBundle(), "AdaTabIndent.", this);
		action.setActionDefinitionId(GNATbenchUIPlugin.getId()
				+ ".AdaTabIndent");
		setAction("TabIndent", action);

		action = new AdaActionRefill(AdaEditorMessages
				.getResourceBundle(), "AdaRefill.", this);
		action.setActionDefinitionId(GNATbenchUIPlugin.getId()
				+ ".AdaRefill");
		setAction("Refill", action);

		action = new AdaActionInsertEnd(AdaEditorMessages
				.getResourceBundle(), "AdaEndCompletion.", this);
		action.setActionDefinitionId(GNATbenchUIPlugin.getId()
				+ ".AdaEndCompletion");
		setAction("InsertEnd", action);

		action = new AdaActionSmartSpace(AdaEditorMessages
				.getResourceBundle(), "AdaSmartSpace.", this);
		action.setActionDefinitionId(GNATbenchUIPlugin.getId()
				+ ".AdaSmartSpace");
		setAction("Expand Ada Syntax", action);

		action = new AdaActionNextSubprogram(AdaEditorMessages
				.getResourceBundle(), "AdaNextUnit.", this);
		action.setActionDefinitionId(GNATbenchUIPlugin.getId()
				+ ".AdaNextUnit");
		setAction("NextAdaUnit", action);

		action = new AdaActionPrevSubprogram(AdaEditorMessages
				.getResourceBundle(), "AdaPrevUnit.", this);
		action.setActionDefinitionId(GNATbenchUIPlugin.getId()
				+ ".AdaPrevUnit");
		setAction("PrevAdaUnit", action);

		action = new AdaActionOpenDeclaration(AdaEditorMessages
				.getResourceBundle(), "AdaOpenDeclaration.", this);
		action.setActionDefinitionId(GNATbenchUIPlugin.getId()
				+ ".AdaOpenDeclaration");
		setAction("OpenDeclaration", action);

		action = new AdaActionOpenBody(AdaEditorMessages.getResourceBundle(),
				"AdaOpenBody.", this);
		action
				.setActionDefinitionId(GNATbenchUIPlugin.getId()
						+ ".AdaOpenBody");
		setAction("OpenBody", action);

		action = new AdaActionReferences(AdaEditorMessages.getResourceBundle(),
				"AdaReferencesInProject.", this, this,
				SearchScopes.SCOPE_PROJECT);
		setAction("ReferencesInProject", action);

		action = new AdaActionReferences(AdaEditorMessages.getResourceBundle(),
				"AdaReferencesInHierarchy.", this, this,
				SearchScopes.SCOPE_PROJECT_HIERARCHY);
		setAction("ReferencesInHierarchy", action);

		action = new AdaActionReferences(AdaEditorMessages.getResourceBundle(),
				"AdaReferencesInWorkspace.", this, this,
				SearchScopes.SCOPE_WORKSPACE);
		setAction("ReferencesInWorkspace", action);

		action = new AdaActionCompileFile (AdaEditorMessages
				.getResourceBundle(), "AdaCompileFile.", this);
		action.setActionDefinitionId(GNATbenchUIPlugin.getId()
				+ ".AdaCompileFile");
		setAction("CompileFile", action);

		action = new AdaActionAnalyzeFile (AdaEditorMessages
				.getResourceBundle(), "AdaAnalyzeFile.", this);
		action.setActionDefinitionId(GNATbenchUIPlugin.getId()
				+ ".AdaAnalyzeFile");
		setAction("AnalyzeFile", action);

		action = new AdaActionGenerateBody(AdaEditorMessages
				.getResourceBundle(), "AdaGenerateBody.", this);
		setAction("GenerateBody", action);

		action = new ContentAssistAction (AdaEditorMessages
				.getResourceBundle(), "ContentAssistProposal.", this);
		String id = ITextEditorActionDefinitionIds.CONTENT_ASSIST_PROPOSALS;
		action.setActionDefinitionId(id);
		setAction ("ContentAssistProposal", action);
		markAsStateDependentAction ("ContentAssistProposal", true);
	}

	/*
	 * @see org.eclipse.ui.texteditor.AbstractDecoratedTextEditor#initializeKeyBindingScopes()
	 */
	protected void initializeKeyBindingScopes() {
		setKeyBindingScopes
		(new String[]
					{ GNATbenchCorePlugin.getId() + ".adaEditorContext" });
	}

	protected void configureSourceViewerDecorationSupport(SourceViewerDecorationSupport support) {
		support.setMatchingCharacterPainterPreferenceKeys(
				AdaEditorPreferences.ADA_MATCHING_BRACKETS,
				AdaEditorPreferences.ADA_MATCHING_BRACKETS_COLOR);
		support.setCharacterPairMatcher(new AdaPairMatcher());

		super.configureSourceViewerDecorationSupport(support);
	}

	/**
	 * This class reacts to the change in the selection, and update the
	 * highlighted range according to the encosing scope.
	 */
	private class RangeIndicator implements ISelectionChangedListener,
			IAdaConstructFilterProvider {

		public void selectionChanged(SelectionChangedEvent event) {

			ITextSelection selection = (ITextSelection) event.getSelection();

			GeneralizedLocation loc = GeneralizedLocation.fromAbsoluteLocation(
					getFile(), selection.getStartLine() + 1, selection
							.getOffset());

			AdaConstruct enclosingConstruct = getDocumentBuffer().getAnalyzer()
					.getConstructAt(loc, this);

			if (enclosingConstruct != null) {
				AdaEditor.this.adjustHighlightRange(enclosingConstruct
						.getOffsetBegin(), enclosingConstruct.getOffsetEnd()
						- enclosingConstruct.getOffsetBegin());
			}
		}

		public AdaConstructFilter getFilter() {
			return new AdaConstructPassTroughFilter () {

				public boolean simpleFilter(AdaConstruct construct) {
					int category = construct.getCategory();
					return (category >= Language_Category.Cat_Package
					&& category <= Language_Category.Cat_Union)
					|| (category >= Language_Category.Cat_Loop_Statement);
				}};
		}
	}

	IPreferenceStore getPreferenceStorePublic () {
		return super.getPreferenceStore();
	}

	public String[] getShowInTargetIds() {
		return new String[] {IPageLayout.ID_RES_NAV,
				"com.adacore.gnatbench.ui.projectexplorer.ProjectExplorer"};
		// TODO: use a constant instead of the above literal
	}

	public ShowInContext getShowInContext() {
		StructuredSelection ss = null;

		AdaEntityReference entity = getEntitySelected();
		if (entity != null && entity.isRegularIdentifier()) {
			AdaConstruct construct = getAnalyzer().getConstructAt(
					entity.getLocation(),
					GPRExplorerContentProvider.fgFilterDeclarations);

			if (construct != null) {
				ss = new StructuredSelection(construct);
			}
		}

		if (getFile ().getFile () != null) {
			return new ShowInContext(getFile ().getFile(), ss);
		} else {
			try {
				return new ShowInContext(new FileStoreReference(
						GNATbenchCorePlugin.getDefault().getEFSRegistry()
								.getUniqueStore(
										URIUtil.toURI(getFile().getOSPath())),
						getProject()), ss);
			} catch (GNATbenchCoreException e) {
				GNATbenchCorePlugin.getDefault().logError(null, e);
			}
		}

		return null;
	}

	@Override
	public void dispose () {
		getDocumentBuffer().getAnalyzer().removeAdaAnalyzerListener(
				fBlocFolding);
		getDocument().removeDocumentListener(this);

		super.dispose();
	}
}
