/*******************************************************************************
 * Copyright (c) 2005, 2007 AdaCore.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *******************************************************************************/

package com.adacore.gnatbench.ui.internal.adaeditor;

import java.util.Arrays;
import java.util.Comparator;
import java.util.HashSet;
import java.util.LinkedList;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.DecoratingLabelProvider;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ILabelProviderListener;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.events.KeyEvent;
import org.eclipse.swt.events.KeyListener;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.MouseListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.progress.UIJob;
import org.eclipse.ui.views.contentoutline.ContentOutlinePage;

import com.adacore.gnatbench.core.internal.GNATbenchCorePlugin;
import com.adacore.gnatbench.core.internal.analyzer.AdaAnalyzer;
import com.adacore.gnatbench.core.internal.analyzer.AdaConstruct;
import com.adacore.gnatbench.core.internal.analyzer.AdaConstructCutFilter;
import com.adacore.gnatbench.core.internal.analyzer.AdaConstructFilter;
import com.adacore.gnatbench.core.internal.analyzer.AdaConstructPassTroughFilter;
import com.adacore.gnatbench.core.internal.analyzer.AdaEntityReference;
import com.adacore.gnatbench.core.internal.analyzer.IAdaAnalyzerListener;
import com.adacore.gnatbench.core.internal.analyzer.IAdaConstructFilterProvider;
import com.adacore.gnatbench.core.internal.browsing.IAdaEntitySelectionProvider;
import com.adacore.gnatbench.core.analyzer.IAdaEntityReference.SearchScopes;
import com.adacore.gnatbench.ui.internal.GNATbenchUIPlugin;
import com.adacore.gnatbench.ui.internal.browsing.AdaActionReferences;
import com.adacore.gnatbench.library.Language.Construct_Visibility;
import com.adacore.gnatbench.library.Language.Language_Category;

/**
 * This class manages the outline and the corresponding actions.
 * 
 * @author ochem
 */
public class AdaContentOutlinePage extends ContentOutlinePage implements
		KeyListener, MouseListener, IAdaEntitySelectionProvider {

	private AdaEditor fEditor;

	final static private String ICON_WITH_LIST = "icon_with_list";
	
	private boolean fHidePrivate = false;
	private boolean fDoSort = false;
	private boolean fDoGroup = false;
	private boolean fHideFields = false;
	private boolean fLinkedWithEditor = false;

	private ContentProvider fContentProvider;
	
	private TreeViewer fViewer;

	/**
	 * When this flag is true, selection changes in the outline will focus the
	 * editor on the corresponding entity.
	 */
	private boolean fFocusOutline = true;

	/**
	 * This classes is able to order constructs, based on their name / category
	 * and visiblity. 
	 *  
	 * @author Quentin Ochem
	 */
	private class ConstructComparator implements Comparator <AdaConstruct> {
		
		/**
		 * Associate each category with an ordered number
		 * 
		 * @param cat
		 * @return
		 */
		public int getCategoryOrder(int cat) {
			if (cat == Language_Category.Cat_Package) {
				return 1;
			} else if (cat >= Language_Category.Cat_Task && cat <= Language_Category.Cat_Entry) {
				return 4;
			} else if (cat >= Language_Category.Cat_Class && cat <= Language_Category.Cat_Subtype) {
				return 2;
			} else if (cat == Language_Category.Cat_Variable) {
				return 3;
			}
			 
			return 0;
		}
		
		public int compare(AdaConstruct c0, AdaConstruct c1) {
			
			int cat0 = getCategoryOrder(c0.getCategory());
			int cat1 = getCategoryOrder(c1.getCategory());

			if (cat0 == cat1 && c0.getVisibility() == c1.getVisibility()) {
				return c0.getName().compareTo(c1.getName());	
			} else if (cat0 == cat1) {
				if (c0.getVisibility() > c1.getVisibility()) {
					return -1;
				} else {
					return 1;
				}
			} else {
				if (cat0 > cat1) {
					return 1;
				} else {
					return -1;
				}
			}
			
		}
	}
	
	/**
	 * This is a special category displayed at the root of the outline, holding
	 * all the package with declarations.
	 * 
	 * @author Quentin Ochem	
	 */
	private class WithDeclarations implements IAdaAnalyzerListener {
		private LinkedList<AdaConstruct> fWith = new LinkedList<AdaConstruct>();
		
		public void add (AdaConstruct construct) {
			fWith.add(construct);
		}
		
		public String toString() {
			return "with clauses";
		}
		
		public AdaConstruct[] getChildren() {
			return fWith.toArray(new AdaConstruct [fWith.size()]);
		}

		/* (non-Javadoc)
		 * @see com.adacore.gnatbench.core.analyzer.IAdaAnalyzerListener#elementAdded(com.adacore.gnatbench.core.analyzer.AdaConstruct)
		 */
		public void elementAdded(AdaConstruct element) {
			fWith.add(element);
			
			if (fWith.size() == 1) {
				fContentProvider.reload();
			} else {			
				fViewer.refresh(this, false);
			}
		}

		/* (non-Javadoc)
		 * @see com.adacore.gnatbench.core.analyzer.IAdaAnalyzerListener#elementRemoved(com.adacore.gnatbench.core.analyzer.AdaConstruct)
		 */
		public void elementRemoved(AdaConstruct element) {
			fWith.remove(element);
			
			if (fWith.size() == 0) {
				fContentProvider.reload();
			} else {			
				fViewer.refresh(this, false);
			}
		}

		/* (non-Javadoc)
		 * @see com.adacore.gnatbench.core.analyzer.IAdaAnalyzerListener#elementModified(com.adacore.gnatbench.core.analyzer.AdaConstruct)
		 */
		public void elementModified(AdaConstruct element) {
			
		}

		public void setRoots(AdaConstruct[] roots) {
			fWith.clear();
			
			for (int i = 0; i < roots.length; i++) {
				fWith.add(roots[i]);
			}
		}
		
		public boolean hasChildren() {
			return fWith.size() > 0;
		}

		/* (non-Javadoc)
		 * @see com.adacore.gnatbench.core.analyzer.IAdaConstructFilterProvider#getFilter()
		 */
		public AdaConstructFilter getFilter() {
			return new AdaConstructCutFilter () {
				public boolean simpleFilter(AdaConstruct construct) {
					return construct.getCategory() == Language_Category.Cat_With;
				}
			};
		}

		public void finishedChangesBatch() {
			// TODO Auto-generated method stub
			
		}

		public void startChangesBatch() {
			// TODO Auto-generated method stub
			
		}
	}
	
	private class LabelProvider implements ILabelProvider {

		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.ILabelProvider#getImage(java.lang.Object)
		 */
		public Image getImage(Object element) {
			Image image = null;
			
			if (element instanceof AdaConstruct) {
				AdaConstruct adaElement = (AdaConstruct) element;
				return adaElement.getImage();			
			}
			else if (element instanceof WithDeclarations) {
				image = GNATbenchUIPlugin.getDefault().getImage(ICON_WITH_LIST);
			}
			
			return image;
		}

		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.ILabelProvider#getText(java.lang.Object)
		 */
		public String getText(Object element) {
			if (fDoGroup && element instanceof AdaConstruct) {
				if (((AdaConstruct) element).getNbParts() > 0) {
					return "* " + element.toString();
				}
				
				return element.toString();
			}
			
			return element.toString();
		}

		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.IBaseLabelProvider#addListener(org.eclipse.jface.viewers.ILabelProviderListener)
		 */
		public void addListener(ILabelProviderListener listener) {

		}

		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.IBaseLabelProvider#dispose()
		 */
		public void dispose() {

		}

		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.IBaseLabelProvider#isLabelProperty(java.lang.Object, java.lang.String)
		 */
		public boolean isLabelProperty(Object element, String property) {
			// TODO Auto-generated method stub
			return false;
		}

		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.IBaseLabelProvider#removeListener(org.eclipse.jface.viewers.ILabelProviderListener)
		 */
		public void removeListener(ILabelProviderListener listener) {

		}
	}

	private class ContentProvider
			implements
				ITreeContentProvider,
				IAdaAnalyzerListener {

		private HashSet<Object> fElementsToRefresh = new HashSet<Object>();
		private boolean fFullRefresh = false; 
		private AdaConstruct[] fRoots = null;
		private AdaAnalyzer fAnalyzer;
		private WithDeclarations fWith = new WithDeclarations();

		public Object[] getChildren(Object parentElement) {
			if (parentElement instanceof AdaConstruct) {
				AdaConstruct[] constructs;
				
				if (fDoGroup) {
					constructs = ((AdaConstruct) parentElement).getAllMainChildren(this);
				} else {
					constructs = ((AdaConstruct) parentElement).getChildren(this);
				}
				
				if (fDoSort) { 
					Arrays.sort(constructs, new ConstructComparator());					
				}
				
				return constructs;
			} else if (parentElement instanceof WithDeclarations) {
				if (!fDoSort) {
					return ((WithDeclarations) parentElement).getChildren();
				} else {
					AdaConstruct[] constructs = ((WithDeclarations) parentElement).getChildren(); 
					Arrays.sort(constructs, new ConstructComparator());				

					return constructs;
				}
			}

			return null;
		}

		public Object getParent(Object element) {
			if (element instanceof AdaConstruct) {
				AdaConstruct construct = (AdaConstruct) element;
				
				if (construct.getCategory() == Language_Category.Cat_With) {
					return fWith;
				} else {
					if (!fDoGroup) {
						return ((AdaConstruct) element).getParent(this);
					} else {
						return ((AdaConstruct) element).getMainParent(this);						
					}
				}
			}

			return null;
		}

		public boolean hasChildren(Object element) {
			if (element instanceof AdaConstruct) {
				if (fDoGroup) {
					return ((AdaConstruct) element).
							getAllMainChildren(this).length > 0;
				} else {
					return ((AdaConstruct) element).hasChildren(this);
				}
			} else if (element instanceof WithDeclarations) {
				return ((WithDeclarations) element).hasChildren();
			}

			return false;
		}

		public Object[] getElements(Object inputElement) {
			if (fRoots != null) {
				if (fWith.hasChildren()) {
					Object[] elements = new Object[fRoots.length + 1];
					
					elements[0] = fWith;

					for (int i = 0; i < fRoots.length; i++) {
						elements[i + 1] = fRoots[i];
					}

					return elements;
				} else {
					return fRoots;
				}
			} else {
				return new Object[]{};
			}
		}


		public void dispose() {
			// TODO Auto-generated method stub

		}

		public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {

			if (oldInput != null && oldInput instanceof AdaAnalyzer) {
				((AdaAnalyzer) oldInput).removeAdaAnalyzerListener(this);
				((AdaAnalyzer) oldInput).removeAdaAnalyzerListener(fWith);
				fRoots = null;
				fAnalyzer = null;
			}

			if (newInput != null && newInput instanceof AdaAnalyzer) {
				((AdaAnalyzer) newInput).addAdaAnalyzerListener(this);
				((AdaAnalyzer) newInput).addAdaAnalyzerListener(fWith);
				fRoots = ((AdaAnalyzer) newInput).getRoots(this);
				fWith.setRoots (((AdaAnalyzer) newInput).getRoots(fWith));
				fAnalyzer = ((AdaAnalyzer) newInput);
			}
		}

		/* (non-Javadoc)
		 * @see com.adacore.gnatbench.core.analyzer.IAdaAnalyzerListener#elementAdded(com.adacore.gnatbench.core.analyzer.AdaSemanticElement)
		 */
		public void elementAdded(AdaConstruct element) {
			if (element.getCategory() == Language_Category.Cat_With) {
				fWith.add(element);
				
				if (!fElementsToRefresh.contains(fWith)) {
					fElementsToRefresh.add(fWith);
				}
			} else if (element.getParent(this) == null) {
				fRoots = fAnalyzer.getRoots(this);
				fFullRefresh = true;
			} else {
				AdaConstruct parent = element.getParent (this);
				
				if (!fElementsToRefresh.contains(parent)) {
					fElementsToRefresh.add(parent);
				}
			}
		}

		/* (non-Javadoc)
		 * @see com.adacore.gnatbench.core.analyzer.IAdaAnalyzerListener#elementRemoved(com.adacore.gnatbench.core.analyzer.AdaSemanticElement)
		 */
		public void elementRemoved(AdaConstruct element) {
			if (element.getParent(this) == null) {
				fRoots = fAnalyzer.getRoots(this);
				fFullRefresh = true;
			} else {
				AdaConstruct parent = element.getParent (this);
				
				if (!fElementsToRefresh.contains(parent)) {
					fElementsToRefresh.add(parent);
				}
			}
		}

		/* (non-Javadoc)
		 * @see com.adacore.gnatbench.core.analyzer.IAdaAnalyzerListener#elementModified(com.adacore.gnatbench.core.analyzer.AdaSemanticElement)
		 */
		public void elementModified(AdaConstruct element) {
			if (element.getParent(this) != null) {
				AdaConstruct parent = element.getParent (this);
				
				if (!fElementsToRefresh.contains(parent)) {
					fElementsToRefresh.add(parent);
				}
			} else {
				fFullRefresh = true;
			}
		}
		
		public void reload() {
			inputChanged(fViewer, fAnalyzer, fAnalyzer);
			fViewer.refresh();
		}

		/* (non-Javadoc)
		 * @see com.adacore.gnatbench.core.analyzer.IAdaConstructFilterProvider#getFilter()
		 */
		public AdaConstructFilter getFilter() {
			return AdaContentOutlinePage.this.getFilter(false);
		}

		public void finishedChangesBatch() {
			fViewer.getControl().setRedraw(false);
			
			if (fFullRefresh) {
				fViewer.refresh();
			} else {
				for (Object element : fElementsToRefresh) {
					fViewer.refresh (element);
				}
			}
			
			fViewer.getControl().setRedraw(true);
			
			fElementsToRefresh.clear();
			fFullRefresh = false;
		}

		public void startChangesBatch() {
			
		}
	}

	/**
	 * This action manages the button "Hide Non-Public Entities"
	 * 
	 * @author Quentin Ochem
	 */
	private class HideFieldsAction extends Action {
		public HideFieldsAction() {
			super ("Hide Fields and Variables", AS_CHECK_BOX);
		}
		
		public ImageDescriptor getImageDescriptor() {
			return GNATbenchUIPlugin.getDefault().getImageDescriptor("/img/",
					"fields_co.gif");
		}
		
		public void run() {
			super.run();
			
			fHideFields = isChecked();
			fContentProvider.reload();
		}
		
		public String getToolTipText() {
			return "Hide Fields and Variables";
		}
	}

	/**
	 * This action manages the button "Hide Non-Public Entities"
	 * 
	 * @author Quentin Ochem
	 */
	private class HidePrivateAction extends Action {
		public HidePrivateAction() {
			super ("Hide Non-Public Entities", AS_CHECK_BOX);
		}
		
		public ImageDescriptor getImageDescriptor() {
			return GNATbenchUIPlugin.getDefault().getImageDescriptor("/img/",
					"methpub_obj.gif");
		}
		
		public void run() {
			super.run();
			
			fHidePrivate = isChecked();
			fContentProvider.reload();
		}
		
		public String getToolTipText() {
			return "Hide Non-Public Entities";
		}
	}

	
	/**
	 * This action manages the button "Sort"
	 * 
	 * @author Quentin Ochem
	 */
	private class SortAction extends Action {
		public SortAction() {
			super ("Sort", AS_CHECK_BOX);
		}
		
		public ImageDescriptor getImageDescriptor() {
			return GNATbenchUIPlugin.getDefault().getImageDescriptor("/img/",
					"alphab_sort_co.gif");
		}
		
		public void run() {
			super.run();
			
			fDoSort = isChecked();
			fContentProvider.reload();
		}
		
		public String getToolTipText() {
			return "Sort";
		}
	}

	/**
	 * This action manages the button "Link With Editor"
	 * 
	 * @author Quentin Ochem
	 */
	private class LinkEditorAction extends Action {
		public LinkEditorAction() {
			super("Link With Editor", AS_CHECK_BOX);
		}
		
		public ImageDescriptor getImageDescriptor() {
			return GNATbenchUIPlugin.getDefault().getImageDescriptor("/img/",
					"synced.gif");
		}
		
		public void run() {
			fLinkedWithEditor = isChecked();
			focusEditor();
		}
		
		public String getToolTipText() {
			return "Link With Editor";
		}
		
	}

	private class GroupAction extends Action {
		public GroupAction() {
			super("Group Partial Entities", AS_CHECK_BOX);
		}
		
		public ImageDescriptor getImageDescriptor() {
			return GNATbenchUIPlugin.getDefault().getImageDescriptor("/img/",
					"group_action.gif");
		}
		
		public void run() {
			fDoGroup = isChecked();
			fContentProvider.reload();
		}
		
		public String getToolTipText() {
			return "Group Partial Entities";
		}
		
	}
		
	/**
	 * This class is used for the synchronization between the outline and the
	 * text editor. Each time the selected item is changed, the corresponding
	 * position in the text editor will be highlighted.
	 *  
	 * @author Quentin Ochem
	 */
	private class FollowSelection implements ISelectionChangedListener {

		/**
		 * Focuses the text editor on the current entity.
		 */
		public void focus() {
			StructuredSelection selection = 
				(StructuredSelection) fViewer.getSelection();

			Object selected = selection.getFirstElement();

			if (selected instanceof AdaConstruct) {
				AdaConstruct element = (AdaConstruct) selected;
				AdaEditor editor = AdaContentOutlinePage.this.fEditor;
				
				int offsetFocus = element.getOffsetEntity();
				
				if (fDoGroup) {
					int offset = editor.getSourceViewerPublic().getSelectedRange().x;
				
					AdaConstruct currentElement = null;
					
					if (offset == offsetFocus) {
						currentElement = element;
					} else if (element.getPart1() != null
							&& element.getPart1().getOffsetEntity() ==
								offset) {
						currentElement = element.getPart1();
					} else if (element.getPart2() != null
							&& element.getPart2().getOffsetEntity() ==
								offset) {
						currentElement = element.getPart2();
					}
					
					if (currentElement != null) {
						AdaConstructFilter filter = getFilter(false);
						
						element = currentElement.getNextPart();
						
						while (!filter.simpleFilter(element)) {
							element = element.getNextPart();
						}
					}
				}
				
				editor.selectAndReveal
					(element.getOffsetEntity(), element.getName().length());						
			}
						
			fViewer.getControl().forceFocus();
		}

		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.ISelectionChangedListener#selectionChanged(org.eclipse.jface.viewers.SelectionChangedEvent)
		 */
		public void selectionChanged(SelectionChangedEvent event) {
			if (fFocusOutline) {
				focus();
			}
		}
		
	}
	
	public AdaContentOutlinePage(AdaEditor editor) {
		super();
		fEditor = editor;

		if (GNATbenchUIPlugin.getDefault().getImage(ICON_WITH_LIST) == null) {

			GNATbenchUIPlugin.getDefault().addImage(ICON_WITH_LIST, "/img/",
					"impc_obj.gif");
		}
	}

	public void createControl(Composite parent) {
		super.createControl(parent);

		fContentProvider = new ContentProvider();
		
		TreeViewer viewer = getTreeViewer();
		viewer.setContentProvider(fContentProvider);
		
		viewer.setLabelProvider(new DecoratingLabelProvider(
				new LabelProvider(), GNATbenchUIPlugin.getDefault()
						.getWorkbench().getDecoratorManager()
						.getLabelDecorator()));
		
		FollowSelection follower = new FollowSelection();
		
		viewer.addSelectionChangedListener(follower);
		
		viewer.setInput(fEditor.getAnalyzer());
		
		fViewer = viewer;
		
		// TODO: This line is here assuming that the editor widget will 
		// *always* be initialized before the outline. If any error occurs 
		// here, then it should be moved somewhere else, maybe in 
		// editor.createViewer.
		fEditor.getSourceViewerPublic().getTextWidget().addKeyListener(this);
		fEditor.getSourceViewerPublic().getTextWidget().addMouseListener(this);
		
		MenuManager baseManager = new MenuManager();
		
		final MenuManager referencesManager = new MenuManager("References");
		
		Menu menu = baseManager.createContextMenu(viewer.getControl());
		
		viewer.getControl().setMenu(menu);
		
		baseManager.add(referencesManager);
		
		new UIJob ("Defining bindings") {

			public IStatus runInUIThread(IProgressMonitor monitor) {
				IViewPart view = getSite().getPage().findView(
						"org.eclipse.ui.views.ContentOutline");
								
				if (view != null && view.getSite() != null) {
					Action action;
				
					action = new AdaActionReferences(AdaEditorMessages
							.getResourceBundle(), "AdaReferencesInProject.",
							fEditor, AdaContentOutlinePage.this,
							SearchScopes.SCOPE_PROJECT);
					referencesManager.add(action);		
					
					action = new AdaActionReferences(AdaEditorMessages
							.getResourceBundle(), "AdaReferencesInHierarchy.",
							fEditor, AdaContentOutlinePage.this,
							SearchScopes.SCOPE_PROJECT_HIERARCHY);
					referencesManager.add(action);		
					
					action = new AdaActionReferences(AdaEditorMessages
							.getResourceBundle(), "AdaReferencesInWorkspace.",
							fEditor, AdaContentOutlinePage.this,
							SearchScopes.SCOPE_WORKSPACE);
					referencesManager.add(action);	
				
					view.getSite().getKeyBindingService()
							.registerAction(action);
					view.getSite().getKeyBindingService().setScopes(
							new String[] { GNATbenchCorePlugin.getId()
									+ ".adaEditorContext" });
				}

				return new Status(IStatus.OK, GNATbenchUIPlugin.getId(),
						IStatus.OK, "", null);
					
			}}.schedule();
	}
	
	public void setActionBars(IActionBars actionBars) {
		super.setActionBars(actionBars);

		actionBars.getToolBarManager().add(new SortAction());
		actionBars.getToolBarManager().add(new GroupAction());
		actionBars.getToolBarManager().add(new HideFieldsAction());
		actionBars.getToolBarManager().add(new HidePrivateAction());
		actionBars.getToolBarManager().add(new LinkEditorAction());
	}

	/* (non-Javadoc)
	 * @see org.eclipse.swt.events.KeyListener#keyPressed(org.eclipse.swt.events.KeyEvent)
	 */
	public void keyPressed(KeyEvent e) {
		// TODO Auto-generated method stub
	}

	/* (non-Javadoc)
	 * @see org.eclipse.swt.events.KeyListener#keyReleased(org.eclipse.swt.events.KeyEvent)
	 */
	public void keyReleased(KeyEvent e) {
		focusEditor();
	}
	
	/**
	 * Moves the selected element to the entity corresponding to the actual 
	 * position in the editor. Only works if the proper switch is set.
	 */
	public void focusEditor() {
		if (fLinkedWithEditor) {
		
			boolean doGroup = fDoGroup;
			fDoGroup = false;
			
			AdaConstruct element = fEditor.getAnalyzer().getConstructAt(
					fEditor.getCurrentLocation(),
					new IAdaConstructFilterProvider() {
						public AdaConstructFilter getFilter() {
							return AdaContentOutlinePage.this.getFilter(true);
						}
					});
			
			fDoGroup = doGroup;
			
			if (element == null) return;
			
			if (fDoGroup) element = element.getMainConstruct();
			
			if (element != null) {
				fFocusOutline = false;
				fViewer.setSelection(new StructuredSelection(element));
				fFocusOutline = true;
				fViewer.expandToLevel(element, 0);
			}
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.swt.events.MouseListener#mouseDoubleClick(org.eclipse.swt.events.MouseEvent)
	 */
	public void mouseDoubleClick(MouseEvent e) {
		
	}

	/* (non-Javadoc)
	 * @see org.eclipse.swt.events.MouseListener#mouseDown(org.eclipse.swt.events.MouseEvent)
	 */
	public void mouseDown(MouseEvent e) {
		
	}

	/* (non-Javadoc)
	 * @see org.eclipse.swt.events.MouseListener#mouseUp(org.eclipse.swt.events.MouseEvent)
	 */
	public void mouseUp(MouseEvent e) {
		focusEditor();		
	}
	
	public AdaConstructFilter getFilter (boolean includeWith) {
		
		class BaseFilter extends AdaConstructPassTroughFilter {

			/**
			 * 
			 */
			public BaseFilter() {
				super();
			}

			/* (non-Javadoc)
			 * @see com.adacore.gnatbench.core.analyzer.AdaConstructFilter#simpleFilter(com.adacore.gnatbench.core.analyzer.AdaConstruct)
			 */
			public boolean simpleFilter(AdaConstruct construct) {
				final int category = construct.getCategory(); 

				return ((category >= Language_Category.Cat_Package && 
						category <= Language_Category.Cat_Local_Variable
						&& category != Language_Category.Cat_Case_Inside_Record) 
						|| category ==Language_Category.Cat_Field)
						&& !construct.getName().equals("");
			}
			
		}
		
		class BaseAndWithFilter extends AdaConstructPassTroughFilter {
			
			/* (non-Javadoc)
			 * @see com.adacore.gnatbench.core.analyzer.AdaConstructFilter#simpleFilter(com.adacore.gnatbench.core.analyzer.AdaConstruct)
			 */
			public boolean simpleFilter(AdaConstruct construct) {
				int category = construct.getCategory(); 
				
				return (category >= Language_Category.Cat_Package && 
						category <= Language_Category.Cat_Local_Variable) 
						|| category == Language_Category.Cat_Field
						|| category == Language_Category.Cat_With;
			}
			
		}
		
		class PrivateFilter extends AdaConstructCutFilter {

			/**
			 * @param filter
			 */
			public PrivateFilter(AdaConstructFilter filter) {
				super(filter);
			}

			/* (non-Javadoc)
			 * @see com.adacore.gnatbench.core.analyzer.AdaConstructFilter#simpleFilter(com.adacore.gnatbench.core.analyzer.AdaConstruct)
			 */
			public boolean simpleFilter(AdaConstruct construct) {
				return construct.getVisibility() == Construct_Visibility.Visibility_Public;
			}
			
		}
		
		class FieldsFilter extends AdaConstructCutFilter {

			/**
			 * @param filter
			 */
			public FieldsFilter(AdaConstructFilter filter) {
				super(filter);
			}

			/* (non-Javadoc)
			 * @see com.adacore.gnatbench.core.analyzer.AdaConstructFilter#simpleFilter(com.adacore.gnatbench.core.analyzer.AdaConstruct)
			 */
			public boolean simpleFilter(AdaConstruct construct) {
				int category = construct.getCategory();
				
				return category != Language_Category.Cat_Field 
						&& category != Language_Category.Cat_Variable
						&& category != Language_Category.Cat_Local_Variable;
			}
			
		}
				
		AdaConstructFilter filter;
		
		if (includeWith) {
			filter = new BaseAndWithFilter();
		} else {
			filter = new BaseFilter();
		}
		
		if (fHidePrivate) {
			filter = new PrivateFilter (filter);
		}
		
		if (fHideFields) {
			filter = new FieldsFilter (filter);
		}
		
		return filter;
	}

	/* (non-Javadoc)
	 * @see com.adacore.gnatbench.core.browsing.IAdaEntitySelectionProvider#getEntitySelected()
	 */
	public AdaEntityReference getEntitySelected() {
		StructuredSelection selection = (StructuredSelection) getSelection();
		Object obj = selection.getFirstElement();
		
		if (obj instanceof AdaConstruct) {
			AdaConstruct construct = (AdaConstruct) obj;

			return new AdaEntityReference(fEditor.getDocumentBuffer(),
					construct);
		}
		
		else return null;
	}
	
	/**
	 * Return the viewer used in this page.
	 * 
	 * @return
	 */
	public TreeViewer getViewer () {
		return fViewer;
	}
}
