/*******************************************************************************
 * Copyright (C) 2005-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *******************************************************************************/

package com.adacore.gnatbench.ui.internal.adaeditor;

import java.util.HashMap;

import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.Position;
import org.eclipse.jface.text.source.Annotation;
import org.eclipse.jface.text.source.projection.ProjectionAnnotation;
import org.eclipse.jface.text.source.projection.ProjectionAnnotationModel;

import com.adacore.gnatbench.core.internal.GNATbenchCorePlugin;
import com.adacore.gnatbench.core.internal.analyzer.AdaConstruct;
import com.adacore.gnatbench.core.internal.analyzer.AdaConstructCutFilter;
import com.adacore.gnatbench.core.internal.analyzer.AdaConstructFilter;
import com.adacore.gnatbench.core.internal.analyzer.IAdaAnalyzerListener;
import com.adacore.gnatbench.library.Language.Language_Category;

/**
 * This class listens to the changes of the structure of the file and
 * subsequently update the folding annotation when needed.
 *
 * @author ochem
 */
public class AdaBlocFolding implements IAdaAnalyzerListener {

	private HashMap<AdaConstruct, Annotation> fOldAnnotations = new HashMap<AdaConstruct, Annotation>();
	private ProjectionAnnotationModel fAnnotationModel = null;
	private IDocument fDocument;

	/**
	 * Sets the annotation model that will hold the folding annotations.
	 *
	 * @param model
	 */
	public void setAnnotationModel(ProjectionAnnotationModel model) {
		fAnnotationModel = model;
	}

	/**
	 * Sets the document that will be folded by this class.
	 *
	 * @param document
	 */
	public void setDocument(IDocument document) {
		fDocument = document;
	}

	/**
	 * Returns the annotation model that holds the folding annotations.
	 *
	 * @return
	 */
	public ProjectionAnnotationModel getAnnotationModel() {
		return fAnnotationModel;
	}

	/**
	 * Sets the roots elements for the annotation model. This has to be called
	 * every time the class has to be initialized on a new file.
	 *
	 * @param elements
	 */
	public void setRoots(AdaConstruct[] elements) {
		if (fAnnotationModel == null)
			return;

		fAnnotationModel.removeAllAnnotations();

		for (int i = 0; i < elements.length; ++i) {
			addElement(elements[i]);
		}
	}

	/**
	 *
	 * @param element
	 * @return True if the element is susceptible to be used by the folding
	 * structure, false otherwise.
	 */
	public boolean filter(AdaConstruct element) {
		int category = element.getCategory();

		return ((category >= Language_Category.Cat_Package
				&& category <= Language_Category.Cat_Subtype)
				|| category >= Language_Category.Cat_Loop_Statement);
	}

	/* (non-Javadoc)
	 * @see com.adacore.gnatbench.core.analyzer.IAdaAnalyzerListener#elementAdded(com.adacore.gnatbench.core.analyzer.AdaSemanticElement)
	 */
	public void elementAdded(AdaConstruct element) {
		if (fAnnotationModel == null)
			return;

		addElement(element);
	}

	/* (non-Javadoc)
	 * @see com.adacore.gnatbench.core.analyzer.IAdaAnalyzerListener#elementRemoved(com.adacore.gnatbench.core.analyzer.AdaSemanticElement)
	 */
	public void elementRemoved(AdaConstruct element) {
		if (fAnnotationModel == null)
			return;

		removeElement(element);
	}

	/* (non-Javadoc)
	 * @see com.adacore.gnatbench.core.analyzer.IAdaAnalyzerListener#elementModified(com.adacore.gnatbench.core.analyzer.AdaSemanticElement)
	 */
	public void elementModified(AdaConstruct element) {
		if (fAnnotationModel == null)
			return;

		modifyElement(element);
	}

	/**
	 * Removes the folding annotation corresponding to the given element.
	 *
	 * @param element
	 */
	private void removeAnnotation(AdaConstruct element) {
		try {
			Annotation oldAnnotation = (Annotation) fOldAnnotations.get(element);

			if (oldAnnotation == null) return;

			/* fAnnotationModel.modifyAnnotationPosition(
					oldAnnotation, new Position(0, 1));*/
			fAnnotationModel.removeAnnotation(oldAnnotation);
			fOldAnnotations.remove(element);

		} catch (Exception e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		}
	}

	/**
	 * Adds a folding annotation corresponding to the given element.
	 * @param element
	 */
	private void addAnnotation(AdaConstruct element) {
		Position position;

		try {
			position = getPosition(element);
		} catch (BadLocationException e) {
			return;
		}

		Annotation annotation = new ProjectionAnnotation();
		fAnnotationModel.addAnnotation(annotation, position);
		fOldAnnotations.put(element, annotation);
	}

	/**
	 * Removed all annotations for the elements and its children.
	 *
	 * @param element
	 */
	private void removeElement(AdaConstruct element) {

		if (!filter(element))
			return;

		AdaConstruct[] children = element.getChildren(this);
		for (int i = 0; i < children.length; ++i) {
			removeElement(children[i]);
		}

		removeAnnotation(element);
	}

	/**
	 * Adds all annotations for the elements and its children.
	 *
 	 * @param element
	 */
	private void addElement(AdaConstruct element) {
		if (!filter(element))
			return;

		AdaConstruct[] children = element.getChildren(this);
		for (int i = 0; i < children.length; ++i) {
			addElement(children[i]);
		}

		// Do not add 1 line objects
		if (element.getLineBegin() == element.getLineEnd())
			return;

		addAnnotation(element);
	}

	/**
	 * Updates all annotations for the element and its children, according to
	 * their new position.
	 *
	 * @param element
	 */
	private void modifyElement(AdaConstruct element) {
		if (!filter(element))
			return;

		AdaConstruct[] children = element.getChildren(this);
		for (int i = 0; i < children.length; ++i) {
			modifyElement(children[i]);
		}

		Position position;

		try {
			position = getPosition(element);
		} catch (BadLocationException e1) {
			return;
		}

		Annotation oldAnnotation = (Annotation) fOldAnnotations.get(element);
		Position oldPosition = fAnnotationModel.getPosition(oldAnnotation);

		if (oldPosition != null) {
			// If the element is now only 1 line, remove it.
			if (element.getLineBegin() == element.getLineEnd()) {
				removeAnnotation(element);
			}
			else if (oldPosition.length != position.length || oldPosition.offset != position.offset)
			{
				try {
					fAnnotationModel.modifyAnnotationPosition(
							(Annotation) fOldAnnotations.get(element), position);
				} catch (Exception e) {
					GNATbenchCorePlugin.getDefault().logError(null, e);
				}
		    }
		} else if (oldAnnotation == null) {
			// If the current object is now more 1 line, add it.
			if (element.getLineBegin() != element.getLineEnd()) {
				addAnnotation(element);
			}
		}
	}

	/**
	 * Returns the position of an annotation (calculates the offset).
	 *
	 * @param element
	 * @return
	 * @throws BadLocationException
	 */
	private Position getPosition(AdaConstruct element)
			throws BadLocationException {

		int offsetBegin = element.getOffsetBegin();
		int lineEnd = element.getLineEnd();
		int offsetEnd = 0;

		offsetEnd = fDocument.getLineOffset(lineEnd)
				+ fDocument.getLineLength(lineEnd) - 1;


		// TODO: These lines seemed to be important in linux to have proper
		// folding position. To be checked.
		/*String delimiter = fDocument.getLineDelimiter(lineEnd);

		 if (delimiter != null)
			offsetEnd = offsetEnd - delimiter.length();*/
		/*if (delimiter == null)
			offsetEnd = offsetEnd;*/

		return new Position(offsetBegin, offsetEnd - offsetBegin + 1);
	}

	/* (non-Javadoc)
	 * @see com.adacore.gnatbench.core.analyzer.IAdaConstructFilterProvider#getFilter()
	 */
	public AdaConstructFilter getFilter() {
		return new AdaConstructCutFilter () {
			public boolean simpleFilter(AdaConstruct construct) {
				return filter (construct);
			}
		};
	}

	public void finishedChangesBatch() {
		// TODO Auto-generated method stub

	}

	public void startChangesBatch() {
		// TODO Auto-generated method stub

	}
}