/*******************************************************************************
 * Copyright (C) 2007-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *******************************************************************************/

package com.adacore.gnatbench.ui.internal.adaeditor;

import java.util.HashMap;
import java.util.Map;
import java.util.ResourceBundle;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.eclipse.jface.text.BadLocationException;
import org.eclipse.ui.texteditor.TextEditorAction;

import com.adacore.gnatbench.core.internal.GNATbenchCorePlugin;
import com.adacore.gnatbench.core.internal.analyzer.AdaConstruct;
import com.adacore.gnatbench.core.internal.analyzer.AdaConstructCutFilter;
import com.adacore.gnatbench.core.internal.analyzer.AdaConstructFilter;
import com.adacore.gnatbench.core.internal.analyzer.GeneralizedLocation;
import com.adacore.gnatbench.core.internal.analyzer.IAdaConstructFilterProvider;
import com.adacore.gnatbench.core.internal.codingstyle.IAutoCasing;
import com.adacore.gnatbench.core.internal.codingstyle.LowerCasing;
import com.adacore.gnatbench.core.internal.codingstyle.MixedCasing;
import com.adacore.gnatbench.core.internal.codingstyle.SmartMixedCasing;
import com.adacore.gnatbench.core.internal.codingstyle.UpperCasing;
import com.adacore.gnatbench.library.Language.Language_Category;
import com.adacore.gnatbench.ui.internal.codingstyle.AdaCodingStylePreferencesProvider;

public class AdaActionSmartSpace extends TextEditorAction {

	private AdaEditor editor;

    private AdaEditorTextUtils textManager;

    // We use the minimum abbreviation length to determine if the word
    // should be expanded into a reserved word.  Hence, any word of length less
    // than this value will not be expanded.
    protected int minAbbreviation;
    // This doesn't make reserved word expansion less a "problem" if the user
    // perceives it as such, but it does mitigate it somewhat.

	final protected Map<Integer, Description> blockDescriptors = setBlockDescriptions();

	final protected static String regexLabel = "^([ \t]*)(.*):(.*)";



 	public AdaActionSmartSpace(ResourceBundle bundle, String prefix, AdaEditor editor) {
		super(bundle, prefix, editor);
		this.editor = editor;
	} // ctor


	public void run() {
		textManager = new AdaEditorTextUtils(editor);

		final boolean prefEnabled = GNATbenchCorePlugin
			.getDefault()
			.getPreferenceStore()
			.getBoolean(
				AdaEditorPreferences.PREF_USE_SMART_SPACE_KEY);

		try {

			if (!prefEnabled) {
				textManager.insertText(" ");
				return;
			} // if

			minAbbreviation = GNATbenchCorePlugin
				.getDefault()
				.getPreferenceStore()
				.getInt(
					AdaEditorPreferences.PREF_USE_MIN_ABBREVIATION_LENGTH);

			// Do the expansion if the context is appropriate and
			// return whether a space key is (also) required.
			// An expansion can occur but also require a space
			// when the expansion is only that of a reserved word.
		    final boolean requiresSpaceKey = doExpansion();

		    if (requiresSpaceKey) {
		    	textManager.insertText(" ");
		    } // if

		} catch (BadLocationException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		} // try
	} // run




	protected boolean doExpansion() throws BadLocationException {
		String originalWord = "";
		String word = "";
		boolean foundColon = false;
		String newLine = "";
		String label;
		int currentIndentation;

		final int syntaxIndentation = GNATbenchCorePlugin
			.getDefault()
			.getPreferenceStore()
			.getInt(AdaCodingStylePreferencesProvider.PREF_INDENT_LEVEL);

		String line = textManager.getLine (textManager.currentLine);

		line = trimRight(line);
		// we only expand if the cursor is at the correct position, ie, the
		// end of the line, where the line contains only the one word to be
		// expanded, optionally with a label.
		if (textManager.currentColumn != line.length()) {
			return true; // so that a space key is emitted
		} // if

		if (line.trim().equals("")) {
			return true; // so that a space key is emitted
		} // if

		label = potentialLabel();

		// check for situations like this: "foo : declare"
		if (line.indexOf(':') == -1) { // didn't find a colon
			originalWord = line.trim().toLowerCase();    // string.lower (string.strip(line));
			foundColon = false;
		} else { // found a colon, maybe an assignment
			Pattern pattern = Pattern.compile ("^([ \\t]*)(.*):(.*)($|--)", Pattern.CASE_INSENSITIVE | Pattern.DOTALL);
			Matcher match = pattern.matcher(line);

    		//  execute matcher.
    		match.matches();

			String remainder = match.group(3);
			if (remainder.indexOf("=") == 0) { // found assignment, not just a colon
				return true; // so that a space key is emitted
			} // if
			originalWord = remainder.trim().toLowerCase();
			foundColon = true;
		} // if

		if (originalWord.length() >= minAbbreviation) {
			word = expandedAbbreviation (originalWord);
			if (word.equals("")) {  // no expansion found
				// we leave it as they typed it since it cannot be a word of interest
				return true; // so that a space key is emitted
			} // if
			// we did an expansion of one of the 'expandable' reserved words
			if (word.equals(originalWord) && !word.equals("for")) {
				// we don't do anything because they typed the whole word, except
				// for the reserved word "for" which is short but still needs expansion
				// because it can take a label.
				return true; // so that a space key is emitted
			} // if
		} else {
			// the word entered was not long enough to consider for expansion, so as not
			// to be too intrusive when entering such identifiers as 'X2' (for example).
			word = originalWord;
		} // if

		if (!label.equals("")) {
			//replace occurrence of label with identifierCase(label) within line
			//note we cannot assign label first since we are searching for it in the call to replace
			// line = string.replace (line,label,identifierCase(label));
			line = line.replaceFirst(label, identifierCase(label));
			label = identifierCase (label);
		} // if

		newLine = new String(line.substring(0, (line.length() - originalWord.length())) + keywordCase(word));

		// note we cannot prepend the blank to the label before we do the following search
		if (foundColon) {
			currentIndentation = newLine.indexOf(label);
		} else {
			currentIndentation = newLine.length() - word.length();
		} // if

		// and now we can prepend the blank to the label for subsequent use, but
		// only if we have a label to work with, otherwise we leave it empty.
		// this way the code below can just append the label unconditionally.
		if (!label.equals("")) {
			label = ' ' + label;
		} // if

		if (word.equals("begin")) {
			textManager.replaceLine (newLine);
			if (!label.equals("")) {
				textManager.insertLine (blanks(currentIndentation) + keywordCase("end") + label + ";");
			} else { // no label, try the decl unit name
				String unitName = associatedDecl ();
				if (!unitName.equals("")) {
					textManager.insertLine (blanks(currentIndentation) + keywordCase("end") + " " + unitName + ";");
				} else { // no label and no decl unit name
					textManager.insertLine (blanks(currentIndentation) + keywordCase("end") + ";");
				} // if empty unit name
			} // if empty label
			textManager.gotoEOL (textManager.currentLine - 1);
			textManager.insertLine (blanks(currentIndentation + syntaxIndentation));
			return false;

		} else if (word.equals("declare")) {
			textManager.replaceLine (newLine);
			textManager.insertLine (blanks(currentIndentation) + keywordCase("begin"));
			textManager.insertLine (blanks(currentIndentation) + keywordCase("end") + label + ";");
			textManager.gotoEOL (textManager.currentLine - 2);
			textManager.insertLine (blanks(currentIndentation+syntaxIndentation));
			return false;

		} else if (word.equals("while")) {
			newLine = newLine + keywordCase("  loop");
			textManager.replaceLine (newLine);
			textManager.insertLine (blanks(currentIndentation) + keywordCase("end loop") + label + ";");
			textManager.setCursorPosition (textManager.currentLine - 1, newLine.length() - 5);
			return false;

		} else if (word.equals("loop")) {
			textManager.replaceLine (newLine);
			textManager.insertLine (blanks(currentIndentation) + keywordCase("end loop") + label + ";");
			textManager.gotoEOL (textManager.currentLine - 1);
			textManager.insertLine (blanks(currentIndentation+syntaxIndentation));
			return false;

		} else if (word.equals("for")) {
			if (withinAdaStatements ()) {
				// expand since it cannot be an attribute definition clause
				newLine = newLine + " " + keywordCase(" loop");
				textManager.replaceLine (newLine);
				textManager.insertLine (blanks(currentIndentation) + keywordCase("end loop") + label + ";");
				// place the cursor at the loop variable declaration
				textManager.setCursorPosition (textManager.currentLine - 1, newLine.length() - 5);
				return false;
			} // if

//		// these don't take a name or a label, but they are convenient for expansion.
//		} else if (word.equals("record") || word.equals("select")) {
//			textManager.replaceLine (newLine);
//			textManager.insertLine (blanks(currentIndentation+syntaxIndentation));
//			textManager.insertLine (blanks(currentIndentation) + keywordCase("end ") + keywordCase(word) + ";");
//			textManager.gotoEOL (currentLine - 1);
//			return false;

		} else {
			if (!word.equals(originalWord)) {
				// we've expanded the word but it isn't one of the interesting ones above so we just
				// make the expansion take effect
				textManager.replaceLine (newLine);
				return true; // so that a space key is emitted too
			} // if
		} // if interesting word

		return true; // so that a space key is emitted
	} // doExpansion


	protected String keywordCase (final String word) {
		IAutoCasing recaser = null;
		final String prefValue = GNATbenchCorePlugin
			.getDefault()
			.getPreferenceStore()
			.getString(
				AdaCodingStylePreferencesProvider.PREF_RESERVED_CASING);

		if (prefValue.equals(AdaCodingStylePreferencesProvider.VAL_UNCHANGED)) {
			return word;
		} // if

		if (prefValue.equals(AdaCodingStylePreferencesProvider.VAL_LOWER)) {
			recaser = new LowerCasing();
		} else if (prefValue.equals(AdaCodingStylePreferencesProvider.VAL_MIXED)) {
			recaser = new MixedCasing();
		} else if (prefValue.equals(AdaCodingStylePreferencesProvider.VAL_UPPER)) {
			recaser = new UpperCasing();
		} else if (prefValue.equals(AdaCodingStylePreferencesProvider.VAL_SMART_MIXED)) {
			recaser = new SmartMixedCasing();
		} // if

		return recaser.recase(word);
	} // keywordCase


	protected String identifierCase (final String word) {
		IAutoCasing recaser = null;
		final String prefValue = GNATbenchCorePlugin
			.getDefault()
			.getPreferenceStore()
			.getString(
					AdaCodingStylePreferencesProvider.PREF_IDENT_CASING);

		if (prefValue.equals(AdaCodingStylePreferencesProvider.VAL_UNCHANGED)) {
			return word;
		} // if

		if (prefValue.equals(AdaCodingStylePreferencesProvider.VAL_LOWER)) {
			recaser = new LowerCasing();
		} else if (prefValue.equals(AdaCodingStylePreferencesProvider.VAL_MIXED)) {
			recaser = new MixedCasing();
		} else if (prefValue.equals(AdaCodingStylePreferencesProvider.VAL_UPPER)) {
			recaser = new UpperCasing();
		} else if (prefValue.equals(AdaCodingStylePreferencesProvider.VAL_SMART_MIXED)) {
			recaser = new SmartMixedCasing();
		} // if

		return recaser.recase(word);
	} // identifierCase


	protected String blanks (final int width) {
		// there MUST be a better way to do this...
		StringBuffer result = new StringBuffer();
		for (int k = 0; k < width; k++) {
			result.append(" ");
		} // for
		return result.toString();
	} // blanks


	protected String trimRight (final String input) {
		int last = input.length() - 1;  // input'last
		while (last > 0) {
			if (!Character.isWhitespace(input.charAt(last))) {
				break;
			} // if
			--last;
		} // while
		if (last == 0) {
			return input;
		} // if
		return input.substring(0, last+1);
	} // trimRight


    // Those words to be expanded whenever the trigger key is hit immediately after the start of a word.

    private static String[] expansionWords = { "abort", "abstract",
		"accept", "access", "aliased", "array",
		"begin", "body", "case", "constant", "declare", "delay", "delta",
		"digits", "else", "elsif", "entry", "exception",
		"exit", "for", "function", "generic", "goto",
		"interface", "limited", "loop",
		"others", "overriding", "package", "pragma",
		"private", "procedure", "protected", "raise", "range", "record",
		"renames", "requeue", "return", "reverse", "select",
		"separate", "subtype", "synchronized", "tagged", "task", "terminate",
		"then", "type", "until", "when", "while", "with" };


    protected String expandedAbbreviation (final String word) {
    	if (word.equals("")) {
    		return "";
    	} // if

    	for (int k = 0; k < expansionWords.length; k++) {
    		if (expansionWords[k].startsWith(word, 0)) {
				return expansionWords[k];
			} // if
    	} // for

    	// didn't find a match
    	return "";
    } // expandedAbbreviation


    protected String potentialLabel () throws BadLocationException {
    	if (!withinAdaStatements()) {
    		return "";
    	} // if

    	String label = "";

    	String labelLine = textManager.getLine(textManager.currentLine);

    	labelLine = trimRight(labelLine);  //strip trailing whitespace

    	if (labelLine.indexOf(':') == -1) { // no colon on this line

    		// look on the previous line for a stand-alone label, ie "foo :" or "foo:"
    		// Rather than go hunting, the label, if any, must be only 1 line up.
    		// This will be ok since a label is never the first line of a program unit.

    		if (textManager.currentLine > 0) {
    			labelLine = textManager.getLine(textManager.currentLine - 1);
    			if (labelLine.indexOf(':') != -1) { // found a colon, which might be for a label
    				Pattern pattern = Pattern.compile (regexLabel, Pattern.CASE_INSENSITIVE | Pattern.DOTALL);
    				Matcher match = pattern.matcher(labelLine);
    				if (match.matches()) {
        				String remainder = match.group(3).trim();
        				if (remainder.equals("")) { // right syntax so far
        					String tempLabel = match.group(2);
        					if (!tempLabel.equals("")) { //found a label
        						label = tempLabel.trim();
        					} // if is a label
        				} // if could be a label
					} // if
    			} // if found colon
    		} // if line above

    	} else { // found ':' on this line

    		Pattern pattern = Pattern.compile (regexLabel, Pattern.CASE_INSENSITIVE | Pattern.DOTALL);
    		Matcher match = pattern.matcher(labelLine);

    		// execute matcher
    		match.matches();

     		//remainder = string.lstrip (match.group(3));
    		String remainder = match.group(3);
    		label = match.group(2);
    		if (remainder != null && !remainder.trim().equals("") && remainder.charAt(0) == '=') {
    			// found assignment operation ":="
    			return "";
    		} // if
    		// Treat as a label, even if it won't be, such as in variable declarations.
    		// Since we only use it where allowed, this isn't a problem.
    		label = label.trim();

    	} // if

    	return label;
    } // potentialLabel


    protected String associatedDecl () throws BadLocationException {

		final GeneralizedLocation currentLocation = editor.getCurrentLocation();

		// force analysis so that latest changes are guaranteed to be recognized
		editor.getAnalyzer().analyze();

		AdaConstruct construct = editor.getAnalyzer().getConstructAt(
				currentLocation,
				new ClosableConstructs());

		if (construct == null) {
			return "";
		} // if

	    final Integer blockType = new Integer(construct.getCategory());

	    if (!blockDescriptors.containsKey(blockType)) {
			return "";
		} // if

	    Description descriptor = (Description) blockDescriptors.get(blockType);

		String result = "";

        String line = textManager.getLine(construct.getLineBegin());

        Pattern pattern = Pattern.compile(descriptor.pattern,
        		Pattern.CASE_INSENSITIVE | Pattern.DOTALL);

        Matcher match = pattern.matcher(line);

        if (match.matches()) {
        	result = match.replaceFirst(descriptor.terminal);
        } else {
        	// The pattern does not match the content.
        	// This is possible with loops and block stmnts without a label
        	// on the same line, so we look on the line above for a label.
        	// There's no guarantee it is one line above either, but that's
        	// as far as we're willing to search.
        	line = textManager.getLine(construct.getLineBegin() - 1);

        	pattern = Pattern.compile(regexLabel, Pattern.DOTALL);

        	match = pattern.matcher(line);

        	if (match.matches()) {
        		result = match.replaceFirst(descriptor.terminal);
         	} // if
        } // if

        return result;
    } // associatedDecl


    protected boolean withinAdaStatements () throws BadLocationException {
    	int blockCount = 0;
    	String prevLine;
    	int searchLineNum = textManager.currentLine - 1;

    	while (searchLineNum >= 0) {
    		prevLine = textManager.getLine(searchLineNum).toLowerCase();
    		if (prevLine.indexOf("begin") != -1 || prevLine.indexOf("accept") != -1) {
    			if (blockCount == 0) {
    				return true;
    			} else {
    				++blockCount;
    			} // if
    		} else if (significantEnd (prevLine)) {
    			--blockCount;
    		} // if
    		--searchLineNum;
    	} // while
    	return false;
    } // withinAdaStatements


    // does this line contain either "end;" or "end <identifier>;"?
    protected boolean significantEnd (final String thisLine) {
    	String targetLine = thisLine.toLowerCase();

    	// an "end" followed by a semicolon is not for a construct and
    	// must be for either a block or a unit, so it is a significant end.
    	if (targetLine.indexOf("end;") != -1) {
    		return true;
    	} // if

    	// is there an "end" followed by another word?
		Pattern pattern = Pattern.compile ("^([ \t]*)end([ \t]*)(.*);(.*)",
				Pattern.CASE_INSENSITIVE | Pattern.DOTALL);
		Matcher match = pattern.matcher(targetLine);
    	if (!match.matches()) {
    		// no "end" at all on this line
    		return false;
    	} // if

    	// there is an "end" followed by a word, but is it an "end" for a construct?

    	// these are the reserved words that can follow "end" and thus
    	// are construct terminators.  If we found one of these, we don't have
    	// an "end" for a unit or block.
    	final String[] reserved = new String[] {"loop", "record", "if", "case", "select"};

    	// the original line might have been "end loop foo;" such that group(3)
    	// is "loop foo".  We want the first word in that group, no matter how
    	// many words there are.
    	String remainder [] = match.group(3).split(" ");

      	if (isMember(remainder[0], reserved)) {
    		return false;
    	} // if

    	// we found "end <identifier>;"
    	return true;
    } // significantEnd


    protected boolean isMember (final String target, final String[] members) {
    	for (int k = 0; k < members.length; k++) {
			if (target.equals(members[k])) {
				return true;
			} // if
		} // for
    	return false;
    } // isMember


	protected class ClosableConstructs implements IAdaConstructFilterProvider {

		public boolean isClosable(AdaConstruct element) {
			int category = element.getCategory();
			return category == Language_Category.Cat_Package
			    || category == Language_Category.Cat_Task
			    || category == Language_Category.Cat_Protected
			    || category == Language_Category.Cat_Procedure
			    || category == Language_Category.Cat_Function
		  	    || category == Language_Category.Cat_Loop_Statement
			    || category == Language_Category.Cat_Accept_Statement
			    || category == Language_Category.Cat_Declare_Block
			    || category == Language_Category.Cat_Simple_Block;
		} // isClosable

		public AdaConstructFilter getFilter() {
			return new AdaConstructCutFilter () {
				public boolean simpleFilter(AdaConstruct construct) {
					return isClosable (construct);
				}
			};
		} // getFilter

	} // ClosableConstructs


	protected class Description {

		public String terminal;
		public String pattern;

		public Description(final String term, final String pattern) {
			this.pattern = pattern;
			this.terminal = term;
		} // ctor

	} // Description


	protected Map <Integer, Description> setBlockDescriptions() {
		HashMap<Integer, Description> result = new HashMap<Integer, Description>();
		result.put(
				new Integer(Language_Category.Cat_Package),
				new Description("$2", "\\s*package\\s+(body\\s+)?([^ \\n]+).*"));
		result.put(
				new Integer(Language_Category.Cat_Loop_Statement),
				new Description("$1", "\\s*([^ ]+)\\s*:(.*)?\\s*loop.*"));
		result.put(
				new Integer(Language_Category.Cat_Procedure),
				new Description("$1", "\\s*procedure\\s+([^ \\n]+).*"));
		result.put(
				new Integer(Language_Category.Cat_Function),
				new Description("$1", "\\s*function\\s+([^ \\n]+).*"));
		result.put(
				new Integer(Language_Category.Cat_Task),
				new Description("$2", "\\s*task\\s+(body\\s+)?([^ \\n]+).*"));
		result.put(
				new Integer(Language_Category.Cat_Protected),
				new Description("$2", "\\s*protected\\s+(body\\s+)?([^ \\n]+).*"));
		result.put(
				new Integer(Language_Category.Cat_Declare_Block),
				new Description("$1", "\\s*([^ ]+)\\s*:\\s*declare.*"));
		result.put(
				new Integer(Language_Category.Cat_Simple_Block),
				new Description("$1", "\\s*([^ ]+)\\s*:\\s*begin.*"));
		result.put(
				new Integer(Language_Category.Cat_Accept_Statement),
				new Description("$1", "\\s*accept\\s+([^ \\n\\(]+).*"));

		return result;
	} // setBlockDescriptions


} // AdaActionSmartSpace