/*******************************************************************************
 * Copyright (C) 2007-2009, AdaCore
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     AdaCore - Initial API and implementation
 *******************************************************************************/

package com.adacore.gnatbench.ui.internal.adaeditor;

import java.util.ResourceBundle;

import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.TextSelection;
import org.eclipse.jface.text.TextViewer;
import org.eclipse.ui.texteditor.TextEditorAction;

import com.adacore.gnatbench.core.internal.GNATbenchCorePlugin;
import com.adacore.gnatbench.ui.internal.codingstyle.AdaCodingStylePreferencesProvider;

public class AdaActionRefill extends TextEditorAction {

	private AdaEditor editor;

    private int tabWidth;

    private int rightMargin;

    private String spaceBeforeComment;
    // Spaces/TAB before comment line

    private String commentSymbol;
    //  Note that if commentSymbol is null we are not working with a comment

    private int spaceAfterComment;
    // Spaces after comment symbol, before the first non-blank character

    private StringBuffer newText;
    //  The new content that will replace the selected lines of text

    private boolean isLineStart;
    // True if at the start of a new line

    private int lineLength;
    // Current line size


	public AdaActionRefill(ResourceBundle bundle, String prefix, AdaEditor editor) {
		super(bundle, prefix, editor);
		this.editor = editor;
	} // ctor


	public void run() {
		final String text = editor.getDocumentBuffer().getContent();

		final IDocument document = editor.getDocumentBuffer().getDocument();

		final TextSelection selection = (TextSelection) ((TextViewer) editor
				.getSourceViewerPublic()).getSelection();

		try {
			final int offsetLineBegin = document.getLineOffset(selection.getStartLine());

			int offsetLineEnd = text.indexOf("\n",
					selection.getOffset() + selection.getLength());

			if (offsetLineEnd < 0) {
				offsetLineEnd = text.length();
			} // if

		    // we get these prefs here, in the run() function, so that any user
			// changes are effective immediately in this action.
			rightMargin = AdaEditorPrefUtils.prefPrintMargin(0);
			if (rightMargin == 0) {
				// no margin so nothing to wrap around
				return;
			} // if

		    tabWidth = GNATbenchCorePlugin
				.getDefault()
				.getPreferenceStore()
				.getInt(AdaCodingStylePreferencesProvider.PREF_INDENT_LEVEL);

		    // init for this invocation
			newText = new StringBuffer();
			commentSymbol = null;
			spaceBeforeComment = null;
		    isLineStart = true;
		    lineLength = 0;

		    String selected = text.substring(offsetLineBegin, offsetLineEnd);

		    if (selected.indexOf('\r') != -1) {
		    	--offsetLineEnd;
		    	selected = selected.replaceAll("\\r", "");
		    } // if

			LineParser parser = new LineParser(selected);

			analyze(parser.firstLine());

		    //  Create new content

			while (parser.moreLines()) {
				final String line = parser.nextLine();

				if (isEmpty(line)) {
					//  Terminate current line and skip one line
					addEOL();
					addEOL();
				} else {
					addAllWords(line);
				} // if
			} // while

			//  Replace text with the new content

			document.replace(offsetLineBegin, offsetLineEnd - offsetLineBegin,
					newText.toString());

		} catch (BadLocationException e) {
			GNATbenchCorePlugin.getDefault().logError(null, e);
		} // try
	} // run


	private void add (final String str) {
		newText.append (str);

		for (int k = 0; k < str.length(); k++) {
			if (str.charAt(k) == '\t') {
				lineLength = lineLength + tabWidth;
			} else {
				lineLength++;
			} // if
		} // for
	} // add


    private void addEOL() {
       add ("\n");
       isLineStart = true;
       lineLength = 0;
    } // addEOL


    private void addWord (final String word) {
    	final String space = " ";

    	//  Does this word fit on the line? If not, start a new line.
    	if ((lineLength + word.length() + 1) > rightMargin) {
    		//  +1 for the space before the word
    		addEOL();
    	} // if

    	if (isLineStart) {

    		//  Add spaces before comment symbol, if we should
    		if (spaceBeforeComment != null) {
    			add (spaceBeforeComment);
    		} // if

    		if (commentSymbol != null) {
    			add (commentSymbol);
    			//  Add spaces after comment symbol
    			for (int k = 0; k < spaceAfterComment; k++) {
    				add (space);
    			} // loop
    		} // if

    		isLineStart = false;

    	} else {
    		//  Add a space before this word.  this is not done for the
    		//  first word on the line.
    		add (space);
    	} // if

    	add (word);
    } // addWord


    private boolean isCommentSymbolNext
    	(final String line, final String commentSymbol, final int index) {

       	if (commentSymbol == null) {
       		return false;
       	} // if

    	if (index + commentSymbol.length() - 1 > line.length()-1) {
    		return false;
       	} // if

    	final CharSequence next = line.subSequence
    								(index, index + commentSymbol.length());

    	return commentSymbol.equals(next);
    } // isCommentSymbolNext


    private void addAllWords (final String line) {

    	int first, last;

    	first = 0;

    	//  Skip spaces and HT

    	while ((first < line.length()-1) &&
    		   (line.charAt(first) == ' ' || (line.charAt(first) == '\t')))
    	{
    		first++;
    	} // loop

    	//  Skip comment symbol

    	if (isCommentSymbolNext(line, commentSymbol, first)) {
    		first = first + commentSymbol.length();
    	} // if

    	//  Is there something remaining

    	boolean isEmpty = true;
    	//  True if only spaces or HT remaining on the line

    	int N = 0;
    	//  Number of spaces after the comment symbol, if any.  This is
    	// just a temporary counter; we store the value elsewhere.
    	for (int k = first; k < line.length()-1; k++) {
    		if ((line.charAt(k) != ' ') && (line.charAt(k) != '\t')) {
    			isEmpty = false;
    			break;
    		} // if
    		N++;
    	} // loop

    	if (isEmpty) {
    		//  No word found, we want to keep the current empty comment
    		//  line as-is.
    		addEOL(); //  Terminate current line
    		addWord("");
    		addEOL();

    	} else {
    		//  There is something after, record the new indent level
    		//  inside the comment.
    		spaceAfterComment = N;
    	} // if

    	//  Read and insert all words on the line

    	while (first <= line.length()-1) {
    		//  Skip spaces

    		while ((first < line.length()-1) && (line.charAt(first) == ' ')) {
    			first++;
    		} // loop

    		//  Get word

    		last = first;
    		while ((last < line.length()) &&
    				(line.charAt(last) != ' ') &&
    				(line.charAt(last) != '\n'))
    		{
    			last++;
    		} // loop

    		if (last == line.length()) {
    			last = line.length() - 1;
    		} // if

    		if (line.charAt(last) == ' ' || line.charAt(last) == '\n') {
    			addWord (line.substring(first, last));
    		} else {
    			addWord (line.substring(first, last+1));
    		} // if

    		first = last + 1;
    	} // loop
    } // addAllWords


    // determine if the line starts with the Ada comment symbol, and if so,
    // returns the index past it.
    private int detectStartPattern(final String line, final int startP) {
    	int P = startP;
    	if ((P < line.length() - 2) && line.substring(P, P+2).equals("--")) {
    		P = P + 2;
    	} // if
    	return P;
    } // detectStartPattern


    // determine the indentation level, the comment symbol (if any), and the
    // spaces after the comment symbol (if any).
    // the actual parameter is intended to be the first line of the selected
    // lines to be refilled.
    private void analyze (final String line) {

        int first, last;

        if (line.trim().equals("")) {
           //  Empty line, nothing to do
           return;
        } // if

        //  Get current indent level based on this, the first line. Note that
        //  this routine will change the indentation level based on this
        //  line.

        first = 0;
        for (int k = 0; k < line.length(); k++) {
			if (line.charAt(k) == ' ' || line.charAt(k) == '\t') {
				++first;
			} else {
				break;
			} // if
		} // for

        if (first > 0) {
           spaceBeforeComment = new String(line.substring(0, first));
        } // if

        //  Get comment symbol, if any

        last = detectStartPattern (line, first);

        if (last > first) {
           commentSymbol = new String(line.substring(first, last));
        } // if

        //  Get spaces after comment symbol

        if (commentSymbol != null) {
        	spaceAfterComment = 0;
            for (int k = last; k < line.length(); k++) {
    			if (line.charAt(k) == ' ') {
    				++spaceAfterComment;
    			} else {
    				break;
    			} // if
    		} // for
        } // if

    } // analyze


    private boolean isEmpty (final String line) {
    	if (line.trim().equals("")) {
    		return true;
    	} // if

    	for (int k = 0; k < line.length(); k++) {
    		if (line.charAt(k) != ' ') {
    			return false;
    		} // if
    		if (line.charAt(k) != '\t') {
    			return false;
    		} // if
    	} // for

    	return true;
    } // isEmpty


    protected class LineParser {

    	private String lines;

    	private int first = 0;


    	LineParser(final String input) {
    		lines = input;
    	} // ctor


    	public String nextLine() {
    		if (first >= lines.length()) {
				return "";
			} // if
    		int last = first;
    		while ((last < lines.length()) && (lines.charAt(last) != '\n')) {
				++last;
			} // while
    		String result = lines.substring(first, last);
    		first = last + 1;
    		return result;
    	} // nextLine


    	public boolean moreLines() {
    		return first < lines.length();
    	} // moreLines


    	public String firstLine() {
    		if (lines.equals("")) {
				return "";
			} // if
    		int last = 0;
    		while ((last < lines.length()) && (lines.charAt(last) != '\n')) {
				++last;
			} // while
    		return lines.substring(0, last);
    	} // firstLine

    } // LineParser


} // AdaActionRefill